define(function (require) {
  'use strict';

  var baja = require('baja!');
  var Promise = require('Promise');
  var typeUtils = require('nmodule/webEditors/rc/fe/baja/util/typeUtils');
  var MgrColumn = require('nmodule/webEditors/rc/wb/mgr/model/MgrColumn');
  var Column = require('nmodule/webEditors/rc/wb/table/model/Column');
  var types = require('baja!nE2Link:NanoComponent');
  var lexs = require('lex!nE2Link');
  var getSlotDisplayName = typeUtils.getSlotDisplayName,
    getSlotType = typeUtils.getSlotType,
    isComplex = typeUtils.isComplex;
  /**
   * API Status: **Development**
   *
   * Column for showing the value of a Complex's Property slot.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn
   * @extends module:nmodule/webEditors/rc/wb/table/model/Column
   * @param {String|baja.Slot} prop the property slot to read from a Complex
   * @param {Object} [params]
   * @param {Type} [params.type] if reading a frozen slot, pass in the Type
   * instance to be able to resolve the display name/facets for that frozen slot
   * @param {Function} [params.getDefaultValue] a function to provide a custom
   * default value for getValueFor to return if the row's Complex does not
   * have the column prop. The function will be passed the row's Complex.
   */

  var NanoPropertyColumn = function NanoPropertyColumn(prop, displayName, params) {
    var that = this;
    Column.apply(that, arguments);
    that.$prop = prop; //that.$slotName
    that.$displayName = displayName; //that.$slotCaption
    that.$type = params && params.type;
    that.$getDefaultValue = params && params.getDefaultValue;
    that.$slotFormat = params && params.slotFormat;
    that.$altSlotFormat = params && params.altSlotFormat;
    that.$slotFlags = params && params.slotFlags;
    that.$slotInfo = params && params.slotInfo;
  };
  NanoPropertyColumn.prototype = Object.create(Column.prototype);
  NanoPropertyColumn.prototype.constructor = NanoPropertyColumn;

  /**
   * Returns the slotCaption
   */
  NanoPropertyColumn.prototype.toDisplayName = function () {
    return this.$displayName;
  };

  /**
   * Shows a display string of the desired property value of the row's
   * loaded Complex.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom
   * @returns {Promise} resolves when the cell has been built
   */
  NanoPropertyColumn.prototype.buildCell = function (row, dom) {
    var that = this;
    return Promise.resolve(this.getValueFor(row).then(function (value) {
      var content = value != undefined ? value.toString(that.data('context')) : '--';
      dom.text(content);
      //if (value != undefined)
      //console.log("NanoPropertyColumn.prototype.buildCell: value = " + value);
    }));
  };

  /**
   * If a `type` param was given, then return an icon for the given slot on
   * that type. Otherwise, return null.
   *
   * @returns {String}
   */

  NanoPropertyColumn.prototype.getColumnIcon = function () {
    var prop = this.$prop,
      type = this.$type;
    if (type) {
      try {
        var slotType = getSlotType(type, prop);
        if (slotType) {
          return slotType.getIcon().getImageUris()[0];
        }
      } catch (ignore) {}
    }
    return null;
  };

  /**
   * Get the value of the Property specified in the constructor from the row's
   * loaded Complex.
   *
   * If the Complex does not have that Property:
   *
   * If a `getDefaultValue` param was passed into this column's constructor, the
   * row's Complex will be passed into the function and the result will be
   * returned.
   *
   * If a `type` param was passed into this column's constructor, the value
   * of the property from the default instance of the given type will be
   * returned.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {baja.Value} the Property value read from the Complex
   * @throws {Error} if the row does not actually have a Complex loaded, or
   * does not have the specified Property (and type is unknown)
   */

  NanoPropertyColumn.prototype.getValueFor = function (row) {
    var comp = row.getSubject(),
      that = this,
      getDefaultValue = that.$getDefaultValue,
      prop = that.$prop,
      type = that.$type;

    //console.log("NanoPropertyColumn.prototype.getValueFor: comp = " + comp + "; prop = " + prop + "; slotFormat = " + that.$slotFormat + "; altSlotFormat = " + that.$altSlotFormat);

    var fmt = baja.Format.make(that.$slotFormat);
    return fmt.format({
      object: comp
    }).then(function (value) {
      return value;
    }).then(function (value) {
      var str = value.toString(that.data('context'));
      if (str.includes('err:') || str.includes('error:')) {
        //console.log("NanoPropertyColumn.prototype.getValueFor: " + prop + " = " + value + "; $slotFormat = " + fmt + " trying with $altSlotFormat: " + that.$altSlotFormat);
        fmt = baja.Format.make(that.$altSlotFormat);
        return fmt.format({
          object: comp
        }).then(function (value) {
          var str = value.toString(that.data('context'));
          if (str.includes('err:') || str.includes('error:')) {
            console.log("NanoPropertyColumn.prototype.getValueFor: returning undefined for " + prop + " = " + value + "; $altSlotFormat = " + fmt);
            return undefined;
          }
          return value;
        });
      }
      return value;
    });
  };
  return NanoPropertyColumn;
});
