/**
 * @file A manager for navigating back and forth between PageViews.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!mobile', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/mobile/mobile', 'mobile/util/ord', 'mobile/util/nav', 'mobile/util/mobile/commands', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/pages', 'mobile/util/mobile/views/PageView', 'mobile/util/views/ViewManager'], function (baja, lexs, $, jqm, Promise, _, mobileUtil, ordUtil, nav, commands, dialogs, pages, PageView, ViewManager) {

  "use strict";

  var encodePageId = mobileUtil.encodePageId,
      decodePageId = mobileUtil.decodePageId,
      mobileLex = lexs[0],
      PageViewManager;

  /**
   * A manager for view widgets. This class, in conjunction with
   * `niagara.util.mobile.nav.NavModel`, manages the creation of and
   * navigation between views within a component hierarchy. jQuery Mobile pages
   * will, where possible, be dynamically created and inserted into the DOM
   * to hold `PageView`s, then removed when no longer needed/accessible.
   * 
   * Note that this class does not perform dynamic linking between different
   * apps, e.g. a property sheet and a schedule app. It is used for
   * navigation between components using the same app. If linking to a
   * completely different app (e.g. property sheet to History), a full page
   * refresh will occur and this PageViewManager will be left behind.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * 
   * @param {niagara.util.nav.NavModel} navModel a NavModel this
   * page manager should use - if omitted just defaults to a new NavModel()
   */
  PageViewManager = baja.subclass(function PageViewManager(navModel) {
    baja.callSuper(PageViewManager, this, ['mobile', 'PageViewManager ']);

    var that = this;

    that.navModel = navModel || new nav.NavModel();

    that.navModel.attach('cut', function (node) {
      var pageView = node.getValue();
      if (!(pageView instanceof PageView)) {
        return;
      }

      pageView.destroy();
      $('#' + pageView.pageId).remove();
    });

    window.onbeforeunload = function () {
      var selectedView = that.$selectedView;

      if (selectedView.isModified()) {
        var dn;

        //hopefully this will be synchronous
        selectedView.toDisplayName().then(function (displayName) {
          dn = displayName;
        });

        return mobileLex.get({
          key: 'message.viewModified',
          args: [dn]
        });
      }
    };

    baja.iterate(commands.getDefaultCommands(), function (cmd) {
      that.getCommandGroup().add(cmd);
    });
  }, ViewManager);

  /**
   * Performs resolution of an ORD (the same ORD from the URL currently
   * viewed in the browser). By default, this simply resolves the ord - but
   * if your view manager needs to incorporate specialized ORD resolution
   * logic (e.g. ORDs matching a certain pattern should resolve to the output
   * of a server side call instead), it may do so by overriding this method.
   * 
   * @name niagara.util.mobile.PageViewManager#doResolveOrd
   * @function
   * @param {String} ord the ORD to resolve
   * @returns {Promise}
   */
  PageViewManager.prototype.doResolveOrd = function doResolveOrd(ord) {
    return baja.Ord.make(ord).get({ lease: true });
  };

  /**
   * Registers navigation behavior for the Pages framework. The default
   * behavior is as follows:
   * 
   * - ORDs will be encoded/decoded using the usual Niagara station url
   *   scheme of `/ord?station:|slot:/etcetc`
   * - on `pagecreate`, the ORD will be retrieved from the URL, the component
   *   will be retrieved from the server via the usual Bajascript calls, and a
   *   new `PageView` will be created and displayed
   * - upon every `pageshow` the page's corresponding `PageView` will be
   *   retrieved and set as the currently selected view (for saving etc)
   * - the Pages `createPage` function will delegate to `this.createPage`
   * 
   * @name niagara.util.mobile.PageViewManager#registerPages
   * @function
   * @returns {Object} an object literal containing Pages handlers
   */
  PageViewManager.prototype.createPagesHandlers = function createPagesHandlers() {
    var that = this;

    function createPage(obj) {

      var spinner = mobileUtil.spinnerTicket(1000, 5000),
          pageData = obj.pageData,
          ord = pageData.ord,
          ordValue;

      that.doResolveOrd(ord).then(function (result) {
        ordValue = result;
        return that.instantiateView(ordValue, pageData);
      }).then(function (view) {
        var page = view.createPage();
        page.jqmData('view', view);
        page.jqmData('value', ordValue);
        page.jqmData('ord', ord);
        obj.ok(page);
      }).catch(baja.error).finally(function () {
        spinner.hide();
      });
    }

    function encodeUrl(pageData) {
      var ord = pageData.ord || 'station:|slot',
          url = '/ord/' + ord;

      return url;
    }

    function decodeUrl(url) {
      //clean up the href as best we can
      var href = decodeURI(url.pathname + url.search).replace('&ui-state=dialog', ''),
          ord,
          viewQuery;

      if (href.indexOf('/ord?') === 0 || href.indexOf('/ord/') === 0) {
        ord = href.substring(5);
      } else {
        return {
          ord: 'station:|slot:'
        };
      }

      viewQuery = baja.Ord.make(ord).parse().get('view');

      return {
        ord: ord,
        viewQuery: viewQuery
      };
    }

    function pagecreate(obj) {
      var page = obj.page,
          view = page.jqmData('view'),
          value = page.jqmData('value'),
          ord = page.jqmData('ord');

      //so on page create, we want to instantiate and draw the component view
      //being shown on that page. BUT! we want that to happen immediately
      //after $.mobile.pageChange - e.g. when the page can be properly laid out
      //but not waiting until scrolling animation is fully complete, so it's
      //drawn properly before it starts to scroll onto the screen. the 
      //solution is to bind to our custom pagelayout event (see 
      //pagebeforechange in util.mobile.pages) - just be sure that the addView
      //function only executes ONCE or stuff goes haywire.
      function doAddView(e) {
        that.addView(view, page, value, ord).catch(baja.error);

        //we don't want to add the view on every pageshow
        page.off('pagelayout', doAddView);
        //        return false; //don't call the regular pagelayout method
      }

      page.on('pagelayout', doAddView);
    }

    function pagelayout(obj) {
      var page = obj.page,
          view = page.jqmData('view'),
          value = view.value(),
          type = value && value.getType();

      /*
       * For virtual components, we want to refresh the view each time the
       * page is shown, as the virtual component could have gone unsubscribed
       * server-side. Perform a check to make sure we don't do a refresh on
       * the very first pageload, since it will be fresh from the server at
       * that point.
       */
      if (type && view.$layoutDone) {
        if (type.isComponent() && type.is('baja:VirtualComponent')) {
          that.refresh(view);
        }
      }

      view.$layoutDone = true;
    }

    function pageshow(obj) {
      var page = obj.page,
          view = page.jqmData('view'); //set in pagecreate
      if (view && view.pageId) {
        that.setSelectedView(view).catch(baja.error);
        view.layout();
      }
    }

    function pagebeforechange(obj) {
      var selectedView = that.selectedView,
          targetUrl;

      if (selectedView && selectedView.isModified()) {
        targetUrl = decodePageId(obj.nextPage.attr('id'));

        that.confirmAbandonChanges(targetUrl);
        obj.event.preventDefault();
      }
    }

    function getCommands(obj) {
      return obj.commands;
    }

    return {
      createPage: createPage,
      encodeUrl: encodeUrl,
      decodeUrl: decodeUrl,
      pagelayout: pagelayout,
      pageshow: pageshow,
      pagecreate: pagecreate,
      pagebeforechange: pagebeforechange,
      getCommands: getCommands
    };
  };

  /**
   * Registers the handlers created in `createPagesHandlers` with the Pages
   * framework.
   * 
   * @name niagara.util.mobile.PageViewManager#registerPages
   * @function
   * @param {String|Regex|Function} matcher a matcher to be passed to the
   * `niagara.util.mobile.pages.register` function. If omitted, performs the
   * default matching on `^ord`. 
   */
  PageViewManager.prototype.registerPages = function registerPages(matcher) {
    pages.register(matcher || function ordMatch(str) {
      return str.match(/^ord/);
    }, this.createPagesHandlers());
  };

  /**
   * Creates a new PageView object and adds it to the screen. A JQM page
   * to hold this view must have already been created and appended to the DOM.
   * The view will be returned fully initialized, subscribed, and ready to use.
   * 
   * @private
   * @name niagara.util.mobile.PageViewManager#initializeView
   * @function
   * @param {baja.Component} component the component to create a new view for
   * @param {jQuery} page the JQM page in which the view will live
   * @param {baja.Value} initialValue The initial value that was resolved for the view.
   * @param {String} ord The ORD for the view to resolve.
   * @returns {Promise} promise to be resolved with the view added (will be an
   * instance of whatever view constructor was passed into the
   * `PageViewManager` constructor).
   */
  PageViewManager.prototype.initializeView = function initializeView(view, page, initialValue, ord) {
    var that = this;

    // Resolve the value again so the bajaux Widget can load properly.
    return view.initialize(page).then(function () {
      return view.load(initialValue);
    }).then(function () {
      that.armHandlers(view);
      that.attachSubscriberEvents(view);
      return view;
    });
  };

  /**
   * Creates a new instance of `PageView`. This is this function's only job -
   * it does not build HTML, subscribe or perform any actions upon the
   * instantiated view. This function may be overridden to perform additional
   * logic, possibly returning an instance of a different `PageView` subclass
   * depending on the type of the given component.
   * 
   * This function _must_ be implemented in your subclass or your 
   * `PageViewManager` will not work.
   * 
   * @name niagara.util.mobile.PageViewManager#instantiateView
   * @function
   * @abstract
   * 
   * @param {baja.Complex} component the component we are creating a new view
   * for
   * @param {Object} [pageData] any page data the `PageViewManager` may have
   * parsed from the currently viewed URL
   * @return {Promise} promise to be resolved with a new instance of a
   * `PageView` or subclass. By default, this will reject if not implemented in
   * a subclass.
   */
  PageViewManager.prototype.instantiateView = function instantiateView(component, pageData) {
    return Promise.reject("instantiateView not implemented");
  };

  /**
   * Adds a component to our list of views. If the `component`  parameter is an
   * ORD, this method will retrieve that ORD and add the resulting component;
   * otherwise the input component will be added directly. After the view is
   * added it will be set as the selected view and scrolled to.
   * 
   * @name niagara.util.mobile.PageViewManager#addView
   * @function
   * @private
   * 
   * @param {String|baja.Component} component the component to be added
   * @param {jQuery} page The JQM page this component is being added to. This is
   * an actual `:jqmData[role=page]` page, not just a target div.
   * @param {baja.Value} value The resolved value.
   * @param {String} ord The ORD for the view.
   * @return {Promise} Promise to be resolved with the newly added `PageView`
   */
  PageViewManager.prototype.addView = function addView(view, page, value, ord) {
    var that = this;
    return that.initializeView(view, page, value, ord).then(function (view) {
      that.navModel.add(view);
      return that.setSelectedView(view).then(function () {
        return view;
      });
    });
  };

  /**
   * Arms basic necessary event handlers on a component view. The only default
   * behavior is that when a list item with class `expandable` is clicked, the
   * corresponding component view will have its `doExpand()` function called.
   * 
   * Also subscribes to basic Bajascript events.
   *
   * @name niagara.util.mobile.PageViewManager#armHandlers
   * @function
   * 
   * @param {niagara.util.mobile.PageView} view the view to arm click handlers
   * on
   */
  PageViewManager.prototype.armHandlers = function armHandlers(view) {
    var that = this,
        el = view.jq();

    el.on('click', 'a.link', function () {
      that.linkToOrd($(this).jqmData('ord'));
    });

    el.on('click', 'a.commandsButton', function () {
      that.showCommands(view);
    });
  };

  PageViewManager.prototype.showCommands = function (view) {
    var viewCommands = view.getCommandGroup(),
        myCommands = this.getCommandGroup(),
        commandsToShow = viewCommands.merge(myCommands, { mergeCommands: false }).flatten({
      recurse: true
    });

    commands.showCommandsDialog(commandsToShow);
  };

  PageViewManager.prototype.attachSubscriberEvents = function (view) {
    var that = this,
        navModel = that.navModel,
        sub = view.contentView.getSubscriber && view.contentView.getSubscriber();

    if (!sub) {
      return;
    }

    sub.attach('renamed', function (prop, oldName, cx) {
      var propId = encodePageId(prop.getName()),
          propOrdId = view.pageId + '_2F' + propId,
          oldPropId = encodePageId(oldName),
          oldPropOrdId = view.pageId + '_2F' + oldPropId;

      $('#' + oldPropOrdId).attr('id', propOrdId);
    });

    sub.attach('unmount', function () {
      var prev = navModel.prev(view),
          prevIndex = navModel.indexOf(prev),
          selectedIndex = navModel.indexOf(that.selectedView),
          msg;

      //only need to hassle the user if he is currently looking at the unmounted
      //component, or one of its children.
      if (selectedIndex > prevIndex) {
        msg = mobileLex.get({
          key: 'propsheet.message.unmounted',
          def: 'Component "{0}" has been unmounted.',
          args: [view.getDisplayName()]
        });

        dialogs.ok({
          content: msg,
          ok: function ok(callbacks) {
            //iterate backwards until I find a view whose component has NOT
            //been unmounted
            while (prev && !prev.getNavOrd()) {
              prev = navModel.prev(prev);
              view = navModel.prev(view);
            }

            var ord = ordUtil.deriveOrd(prev.getValue().value());

            if (ord && ord !== 'null') {
              that.linkToOrdInternal(ord);
            }

            navModel.cut(view);

            callbacks.ok();
          }
        });
      } else {
        navModel.cut(view);
      }
    });
  };

  /**
   * If the currently selected view is modified, this function delegates to 
   * `niagara.util.mobile.dialogs.confirmAbandonChanges`. It displays the name
   * of the selected view in the message to confirm abandoning changes; it will
   * save this view if the user clicks 'yes', abandon changes and refresh the
   * view if the user clicks 'no', and just stays put if the user clicks
   * 'cancel'.
   *  
   * @name niagara.util.mobile.PageViewManager#confirmAbandonChanges
   * @function
   * @param {String|jQuery|Function} [redirect] the redirect to be passed to
   * the dialog invocation - this will be the page/url we should navigate to if
   * 'yes' or 'no' is clicked (or a function that does the page navigation
   * for us).
   */
  PageViewManager.prototype.confirmAbandonChanges = function confirmAbandonChanges(redirect) {
    var that = this,
        selectedView = this.$selectedView;

    return new Promise(function (resolve, reject) {
      if (selectedView.isModified()) {
        selectedView.toDisplayName().then(function (displayName) {
          return dialogs.confirmAbandonChanges({
            yes: function yes(cb) {
              var dialog = this;

              selectedView.save().then(function () {
                that.refresh(selectedView).then(function () {
                  resolve();
                  cb.ok();
                }, function (err) {
                  reject(err);
                  cb.fail(err);
                });
              }, function (err) {
                dialog.redirect(decodePageId(selectedView.pageId));
                reject(err);
                cb.fail(err);
              });
            },
            no: function no(cb) {
              selectedView.setModified(false);
              that.refresh(selectedView).then(function () {
                resolve();
                cb.ok();
              }, function (err) {
                reject(err);
                cb.fail(err);
              });
            },
            viewName: displayName,
            redirect: redirect
          });
        });
      } else {
        resolve();
      }
    });
  };

  /**
   * Saves the given view. This is different from just calling `view.save()`
   * only in that view will also be be refreshed and resubscribed after the
   * save is complete.
   * 
   * @name niagara.util.mobile.PageViewManager#save
   * @function
   * @param {niagara.util.mobile.PageView} view the PageView to save
   */
  PageViewManager.prototype.save = function save(view) {
    return view.save();
  };

  /**
   * @returns {Promise}
   */
  PageViewManager.prototype.saveSelected = function saveSelected() {
    var that = this;

    return that.getSelectedView().then(function (selectedView) {
      return that.save(selectedView);
    });
  };

  /**
   * Sets a given view as our currently selected view.
   * 
   * @name niagara.util.mobile.PageViewManager#setSelectedView
   * @function
   * @param {niagara.util.mobile.PageView} view the view to select 
   */
  PageViewManager.prototype.setSelectedView = function setSelectedView(view) {
    var that = this;

    return baja.callSuper('setSelectedView', PageViewManager, this, [view]).then(function () {
      return view.toDisplayName();
    }).then(function (displayName) {
      var page;

      document.title = displayName;

      page = $('#' + view.pageId.replace(/(:|\.)/g, '\\$1'));
      page.children(':jqmData(role=header)').children('h1.viewName').text(displayName);

      that.currentPage = page;
      that.navModel.setSelectedNode(view);
    });
  };

  /**
   * Refreshes a given view by calling its `refresh` method. Re-arms event
   * handlers once the view's HTML has been completely rebuilt. Note that
   * calling `refresh` on a view will always perform a refresh, potentially
   * throwing out user-entered changes. `refreshSelected` will confirm with
   * the user if there are any outstanding changes to the current view.
   * 
   * @name niagara.util.mobile.PageViewManager#refresh
   * @function
   * @param {niagara.util.mobile.PageView} view the view to refresh
   * @returns {Promise} to be resolved after the view is done refreshing
   */
  PageViewManager.prototype.refresh = function refresh(view) {
    var that = this;
    return view.load(view.value()).then(function () {
      that.attachSubscriberEvents(view);
    });
  };

  /**
   * Refreshes the currently selected view. If this view has any outstanding
   * user-entered changes, will pop up a dialog confirming that the user wishes
   * to abandon these changes before refreshing.
   * 
   * @name niagara.util.mobile.PageViewManager#refreshSelected
   * @function
   * @returns {Promise} promise to be resolved after the selected view is
   * refreshed
   */
  PageViewManager.prototype.refreshSelected = function refreshSelected() {
    var that = this;

    return that.getSelectedView().then(function (selectedView) {
      if (selectedView.isModified()) {
        return that.confirmAbandonChanges(undefined);
      } else {
        return that.refresh(selectedView);
      }
    });
  };

  /**
   * @memberOf niagara.util.mobile
   * @private
   */
  function shouldScrollReverse(pageViewManager, ord) {
    var that = pageViewManager,
        selectedView = that.$selectedView,
        ords,
        selectedIndex,
        targetIndex;

    if (selectedView) {
      ords = that.navModel.getOrds();
      selectedIndex = _.indexOf(ords, ordUtil.deriveOrd(selectedView.value()));
      targetIndex = _.indexOf(ords, ord);

      if (targetIndex >= 0 && selectedIndex >= 0 && targetIndex < selectedIndex) {
        return true;
      }
    }

    return false;
  }

  /**
   * Delegates to `niagara.util.mobile.linkToOrd`.
   * 
   * If the current view has outstanding changes, will confirm with the user
   * whether to abandon those changes before linking to the new view.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrd
   * @function
   * @see niagara.util.mobile.linkToOrd
   * 
   * @param {baja.Ord|String} ord the Ord to navigate to
   * @param {baja.ViewQuery} viewQuery any view parameters to append to the ORD in the
   * case of an internal link
   */
  PageViewManager.prototype.linkToOrd = function pageViewManagerLinkToOrd(ord, viewQuery) {
    var that = this;

    function doLink() {
      return mobileUtil.linkToOrd(ord, {
        viewQuery: viewQuery,
        reverse: shouldScrollReverse(that, ord)
      });
    }

    return that.getSelectedView().then(function (selectedView) {
      if (selectedView.isModified()) {
        return that.confirmAbandonChanges(doLink);
      } else {
        return doLink();
      }
    });
  };

  /**
   * Delegates to `niagara.util.mobile.linkToOrdInternal`, but performs some
   * extra logic to cause JQM to scroll left or right depending on whether
   * we're moving up or down in the component tree.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrdInternal
   * @function
   * @private
   * @see niagara.util.mobile.linkToOrdInternal
   * 
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition -
   * will be passed directly into `$.mobile.changePage()`
   * @param {Object} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  PageViewManager.prototype.linkToOrdInternal = function pageViewManagerLinkToOrdInternal(ord, options) {
    options = baja.objectify(options);
    options.reverse = shouldScrollReverse(this, ord);
    mobileUtil.linkToOrdInternal(ord, options);
  };

  /**
   * Delegates to `niagara.util.mobile.linkToOrdExternal`.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrdExternal
   * @function
   * @private
   * @see niagara.util.mobile.linkToOrdExternal
   * 
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition
   * @param {Object} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  PageViewManager.prototype.linkToOrdExternal = function pageViewManagerLinkToOrdExternal(ord, options) {
    mobileUtil.linkToOrdExternal(ord, options);
  };

  /**
   * Attempt to layout the selected view.
   * 
   * @returns {Promise} A promise that's resolved once the 
   * selected view has been laid out.
   */
  PageViewManager.prototype.layout = function () {
    return this.getSelectedView().then(function (view) {
      if (view) {
        return view.layout();
      }
    });
  };

  return PageViewManager;
});
