var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Functions related to the management and display of JQM
 * dialog boxes.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!mobile', 'jquery', 'jquerymobile', 'Promise', 'mobile/util/mobile/pages', 'mobile/util/mobile/mobile', 'css!mobile/util/mobile/dialogs'], function (baja, lexs, $, jqm, Promise, pages, mobileUtil) {

  "use strict";

  var callbackify = baja.callbackify,
      decodePageId = mobileUtil.decodePageId,
      encodePageId = mobileUtil.encodePageId,
      getActivePage = mobileUtil.getActivePage,
      getPageContainer = mobileUtil.getPageContainer,
      hidePageLoadingMsg = mobileUtil.hidePageLoadingMsg,
      _lexs = _slicedToArray(lexs, 1),
      mobileLex = _lexs[0];

  /**
   * @private
   * @exports mobile/util/mobile/dialogs
   */
  var exports = {};

  var currentInvocation = void 0,
      dialogHtml = void 0,
      buttonHtml = void 0,
      dialogActive = false;

  dialogHtml = '<div data-role="dialog" class="dynamicDialog" data-overlay-theme="b">\n' + '<div class="dynamicDialogHeader" data-role="header" data-theme="a">\n' + '<h1 class="title" />\n' + '</div>\n' + '<div class="dynamicDialogContent" data-role="content" data-theme="c">\n' + '<div class="dialogDisplay"></div>\n' + '<div class="dialogButtons" data-role="controlgroup" data-type="horizontal"></div>\n' + '</div>\n' + '</div>';

  buttonHtml = '<a data-value="{value}" data-role="button" data-theme="a"></a>';

  ////////////////////////////////////////////////////////////////
  // DialogInvocation utility functions
  ////////////////////////////////////////////////////////////////


  /**
   * Loads the dialog's content div. That content may be one of several 
   * different types.
   * 
   * If it is a jQuery object, it will be inserted directly into the dialog's
   * content div.
   * 
   * If it is a string or other type of object, it will be converted into
   * a string and inserted into the dialog's content div.
   * 
   * If it is a function, it will be executed with the (empty) content div 
   * as the first parameter, and an object with ok/fail callbacks as the
   * second parameter.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   * @param {String|JQuery|Function} content the main dialog display content
   * @param {JQuery} contentDiv the dialog's content div
   * @returns {Promise} promise to be resolved after the content has fully
   * loaded
   */
  function loadContent(content, contentDiv) {
    if (content instanceof $) {
      contentDiv.html(content);
      return Promise.resolve();
    } else if ((typeof content === 'undefined' ? 'undefined' : _typeof(content)) === 'object' || typeof content === 'string') {
      if (content instanceof Error) {
        baja.error(content);
      }
      contentDiv.text(String(content));
      return Promise.resolve();
    } else if (typeof content === 'function') {
      return Promise.try(function () {
        return content(contentDiv.empty());
      });
    }
  }

  ////////////////////////////////////////////////////////////////
  // DialogInvocation
  ////////////////////////////////////////////////////////////////


  /**
   * @name niagara.util.mobile.dialogs.DialogInvocation
   * @private
   * @class
   * 
   * @param {Object} [params] an object literal containing parameters to be
   * used by this dialog. Can also be a function, in which case it will be
   * treated as `params.content`.
   * @param {String} [params.title] the dialog title to be shown
   * @param {String|jQuery|Function} [params.content] the main dialog display 
   * content
   * @param {jQuery|String} [params.returnPage] The page to return to after 
   * the dialog is closed. This can be changed by calling redirect(). If
   * omitted, when this dialog is closed it will simply return to whatever
   * page is currently referenced by `location.href`.
   * @param {Object} [params.callbacks] an object containing ok/fail callbacks.
   * These callbacks will be called when the dialog is closed by calling 
   * `invoke()` (e.g., when the user clicks Yes/No/Cancel/etc). Note
   * that they are called even if this dialog itself does not return us to the
   * previously viewed JQM page - they will still fire if this dialog opens
   * another dialog.
   * 
   * @see niagara.util.mobile.dialogs.loadContent
   */
  var DialogInvocation = function DialogInvocation(params, page) {
    /**
     * the parameters of the currently viewed dialog - title, content, handlers
     * etc.
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$params
     * @field
     * @type {Object}
     * @private
     */
    this.$params = baja.objectify(params, 'content');

    /**
     * The page to return to after the dialog is closed. This can be changed
     * by calling redirect().
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$returnPage
     * @field
     * @type {jQuery|String}
     * @private
     */
    this.$returnPage = params.returnPage;

    /**
     * The JQM page holding the actual dialog. Should already be constructed
     * and appended to the DOM when the DialogInvocation is created.
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$page
     * @field
     * @type {JQuery}
     * @private
     * @see niagara.util.mobile.dialogs.createDialogPage
     */
    this.$page = page;

    /**
     * Set to false if this dialog is opened when we are NOT already looking at
     * another dialog; set to true if this dialog is being opened while another
     * dialog is already active. When this dialog is closed, we will return
     * to the JQM page we were previously looking at if and only if this is
     * false. 
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$dialogActive
     * @field
     * @type {Boolean}
     * @private
     * @see niagara.util.mobile.dialogs.DialogInvocation#invoke
     */
    dialogActive = true;
  };
  exports.DialogInvocation = DialogInvocation;

  function isRedirect(returnPage) {
    var currentPage = pages.getCurrent();

    if (typeof returnPage === 'string') {
      returnPage = encodePageId(returnPage);

      if (returnPage.charAt(0) === '#') {
        returnPage = $(returnPage);
      } else {
        returnPage = $('#' + returnPage);
      }
    }

    // when changing to returnPage, should the URL change as well?
    // if the current URL references an existing JQM page, and that
    // JQM page is the same as the one referenced by returnPage, NO.
    // otherwise yes.
    if (currentPage && returnPage && currentPage[0] === returnPage[0]) {
      return false;
    } else {
      return true;
    }
  }

  /**
   * Closes a dialog and returns to another JQM pages, invoking a 
   * function after the dialog has fully closed.
   *
   * @param {String} [clickedValue] the value of the button that was clicked
   * to close the dialog - defaults to 'cancel' if omitted
   */
  DialogInvocation.prototype.close = function close(clickedValue) {
    clickedValue = clickedValue || 'cancel';

    var that = this,
        options = that.$options,
        returnPage = that.$returnPage;

    if (typeof returnPage === 'string') {
      //if we're going to localhost/ord?.... then we don't want a # if we can
      //help it - we want to update the entire URL. otherwise, we're going
      //directly to a JQM page referenced by a hash
      if (returnPage.indexOf('ord?') !== 0 && returnPage.charAt(0) !== '#') {
        returnPage = '#' + returnPage;
      }
    }

    if (returnPage instanceof $ || typeof returnPage === 'string') {
      jqm.changePage(returnPage, $.extend({
        transition: 'none',
        changeHash: isRedirect(returnPage)
      }, options));
    } else if (typeof returnPage === 'function') {
      returnPage.call(this, clickedValue);
    } else {
      // Since we're never changing the hash for these dialogs, simply change
      // back to the current page to get rid of the dialogs
      jqm.changePage(location.href, {
        transition: "none",
        changeHash: false
      });
    }

    currentInvocation = undefined;
  };

  /**
   * Performs the action corresponding to one of the dialog buttons (e.g.
   * 'ok', 'yes', 'no') and then closes the dialog.
   * 
   * @param {String|Function} value the value corresponding to the button 
   * whose action you wish to invoke. This can also be a function to be
   * invoked directly. The function should take a single argument of an ok/fail
   * callback object literal - it should call `ok()` or `fail()` on this object
   * to signal that the handler has completed and the dialog should be closed.
   * 
   * @param {Object} [callbacks] an object containing ok/fail callbacks. 
   * `callbacks.ok` will be called when the dialog closes (or another dialog is
   * shown). `callbacks.fail` will be called if the click handler calls its own
   * `fail()` or throws an error.
   */
  DialogInvocation.prototype.invoke = function invoke(value, callbacks) {
    var ok = baja.ok,
        error = exports.error;
    callbacks = callbackify(callbacks, ok, error);

    var that = this,
        params = that.$params,
        closeCallbacks = callbackify(params.callbacks, ok, error),
        handler = typeof value === 'function' ? value : params[value];

    if (typeof handler !== 'function') {
      handler = function handler(callbacks) {
        callbacks.ok();
      };
    }

    function doFail(err) {
      dialogActive = false;

      //run the fail callbacks first in case one of them shows another dialog.
      closeCallbacks.fail(err);
      callbacks.fail(err);

      //if no more dialogs were shown, then we can close.
      if (!dialogActive) {
        that.close(value);
      }
    }

    try {
      dialogActive = false;
      handler.call(that, {
        ok: function ok() {
          //if the ok/yes/no/whatever click handler causes another dialog page,
          //we don't want to return to the previous page. 
          if (!dialogActive) {
            that.close(value);
          }
          closeCallbacks.ok();
          callbacks.ok();
        },
        fail: doFail
      });
    } catch (err) {
      doFail(err);
    }
  };

  /**
   * Allows a dialog to redirect to another page instead of the one that was
   * in view when the dialog was initially invoked. This method may be called
   * inside of a dialog click handler e.g. `this.redirect('#otherPage')`. The
   * `#otherPage` JQM page will be shown after the dialog is closed. 
   *
   * @param {jQuery|String|Function} returnPage the new return page. If 
   * `returnPage` is a function, then this function will be executed _instead_
   * of doing `$.mobile.changePage()` or `$('.ui-dialog').dialog('close')`.
   * This allows you to perform your own page management when the dialog is
   * closed - performing an asynchronous action before redirecting to another
   * page, for instance.
   * 
   * @param {Object} [options] options to be passed directly to 
   * `$.mobile.changePage` - only to be used if `returnPage` is a jQuery object
   * or string.
   */
  DialogInvocation.prototype.redirect = function redirect(returnPage, options) {
    this.$returnPage = returnPage;
    this.$options = options || {};
  };

  /**
   * Populates a JQM dialog page with the dialog content. At the point this
   * function is called, the JQM page has already been created, with all
   * necessary buttons, and inserted into the DOM.
   * 
   * This function runs on the output of `createDialogPage`, just before it is
   * shown by JQM.
   *
   * @returns {Promise} promise to be resolved when the dialog has loaded
   */
  DialogInvocation.prototype.load = function load() {
    var that = this,
        page = that.$page,
        params = that.$params,
        header = page.find(':jqmData(role=header)'),
        content = page.find(':jqmData(role=content)'),
        display = content.children('.dialogDisplay'),
        xButton = header.find('a:jqmData(icon=delete)'),
        buttons = content.find('.dialogButtons a');

    header.find('h1.title').text(params.title || '');

    buttons.off('click').addClass('ui-disabled');

    //arm our own handler on the X button
    xButton.off() //remove any previous click handlers we already armed
    .on('click', function () {
      that.invoke('cancel'); //we will do it ourselves thank you
      return false;
    });

    return loadContent(params.content || '', display).then(function () {
      buttons.on('click', function () {
        var value = $(this).jqmData('value');

        that.invoke(value);
      });
      buttons.removeClass('ui-disabled');

      exports.repositionDialog(page);
    }, function (err) {
      exports.error("Failed to load dynamic dialog content: " + err);
    });
  };

  ////////////////////////////////////////////////////////////////
  // Dialog utility functions
  ////////////////////////////////////////////////////////////////

  /**
   * Dynamically builds up the HTML for a dialog page, appending all necessary
   * buttons, and inserts it into the DOM.
   * 
   * This function will run immediately before showing a particular type
   * of dialog for the first time.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   * @param {String} id the desired HTML id of the page
   * @param {Array} buttons an array of button values (e.g.
   * `[ 'yes', 'no', 'cancel' ]`
   * @returns {jQuery} a JQM dialog page
   */
  function createDialogPage(id, buttons) {
    var page = $(dialogHtml.patternReplace({
      theme: 'a'
    })),
        buttonsDiv = page.children(':jqmData(role=content)').children('.dialogButtons');

    baja.iterate(buttons, function (buttonValue, i) {
      var button = $(buttonHtml.patternReplace({
        value: buttonValue
      }));

      button.text(mobileLex.get({
        key: buttonValue,
        def: buttonValue
      }));
      button.appendTo(buttonsDiv);
    });

    page.attr('id', id);
    if (id.indexOf('error') === 0) {
      page.addClass('error');
    }
    page.appendTo(getPageContainer());
    return page;
  }

  /**
   * Attempts to center the dialog vertically on the screen. If the dialog is
   * too tall it will simply butt the top against the upper edge of the screen.
   * 
   * @param {JQuery} page the dialog page to position onscreen
   */
  exports.repositionDialog = function (page) {
    var containerDiv = page.children('.ui-dialog-contain'),
        headerDiv = page.find(':jqmData(role=header)'),
        dialogHeight = containerDiv.outerHeight(),
        windowHeight = $(window).height(),
        headerAdjust = (windowHeight - dialogHeight) / 2,
        marginTop = Math.max(headerAdjust, 15);

    containerDiv.css('margin-top', marginTop);
    headerDiv.css('margin-top', marginTop);
  };

  ////////////////////////////////////////////////////////////////
  // Public dialog functions
  ////////////////////////////////////////////////////////////////

  /**
   * Shortcut to closing whatever dialog is currently being displayed.
   */
  exports.closeCurrent = function () {
    if (currentInvocation) {
      currentInvocation.close();
    }
  };

  /**
   * Registers a new dialog type in Pages. Once registered, can be displayed
   * by calling the `load` method on the Pages handler. The `load` method, and
   * all built-in dialog methods like `yesNoCancel` and `okCancel`, return an
   * object reference that can be used to programmatically close the dialog or 
   * click buttons.
   *
   * @param {String} pageId the HTML ID to use for the dialog page
   * @param {Array} buttons an array of button values (e.g. 'ok', 'yes', 'no').
   * These should unique within this dialog (cannot have two 'ok', for example)
   * and should correspond to entries in the `mobile` lexicon whose values will
   * be used to display button titles.
   * 
   * @example
   * registerDialog('leftRightDialog', [ 'left', 'right' ]);
   * var dialog = pages.getHandler('leftRightDialog').load({
   *   content: 'please click "left" or "right" below',
   *   title: 'left or right',
   *   //function names correspond to button titles
   *   left: function (cb) { console.log('you clicked "left"'); cb.ok(); }
   *   right: function (cb) { console.log('you clicked "right"'); cb.ok(); }
   *   callbacks: function () { console.log('all done. thanks for clicking'); }
   * });
   * //now you can call methods on the returned object directly, like
   * //dialog.close(); or dialog.left(); or dialog.right();
   */
  exports.registerDialog = function (pageId, buttons) {
    var handler;

    function pagebeforeshow(obj) {
      if (currentInvocation) {
        currentInvocation.load();
      } else {
        obj.page.find('.dialogDisplay').empty();
      }
    }

    function load(params) {
      var page = $('#' + pageId);

      if (!page.length) {
        page = createDialogPage(pageId, buttons);
      }

      params.returnPage = currentInvocation && currentInvocation.$returnPage || params.returnPage;

      currentInvocation = new DialogInvocation(params, page);

      jqm.changePage(page, {
        changeHash: false,
        allowSamePageTransition: true,
        transition: "none"
      });

      return currentInvocation;
    }

    handler = {
      load: load,
      pagebeforeshow: pagebeforeshow
    };

    pages.register(pageId, handler);
  };

  ////////////////////////////////////////////////////////////////
  // Functions for displaying predefined dialog types
  ////////////////////////////////////////////////////////////////


  (function registerPages() {
    var registerDialog = exports.registerDialog;

    registerDialog('yesNoCancelDialog', ['yes', 'no', 'cancel']);
    registerDialog('yesNoDialog', ['yes', 'no']);
    registerDialog('okCancelDialog', ['ok', 'cancel']);
    registerDialog('okDialog', ['ok']);
    registerDialog('errorDialog', ['ok']);
    registerDialog('cancelDialog', ['cancel']);
  })();

  /**
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  var loadHandler = function loadHandler(name, params) {
    return pages.getHandler(name).load(params);
  };

  /**
   * Shows a standard dialog with 'Yes', 'No', and 'Cancel' buttons.
   * 
   * With this and all dialogs, you can simply pass in the content directly
   * rather than with an object literal with properties. This will cause all
   * the buttons to do nothing except close the dialog. Obviously, this is
   * most appropriate when used with a one-button dialog like 'ok' or 'cancel'.
   *
   * @param {Object} params an object containing the behavior to execute on
   * each button click
   * @param {String} [params.title] the dialog title
   * @param {String|Function|jQuery} [params.content] the content of the dialog
   * @see niagara.util.mobile.dialogs.loadContent
   * @param {Function} [params.yes] a function to execute when 'yes' is clicked
   * @param {Function} [params.no] a function to execute when 'no' is clicked
   * @param {Function} [params.cancel] a function to execute when 'cancel' is 
   * clicked
   * @param {Function} [params.onClose] a function to execute after the
   * dialog closes (this will not execute if your dialog triggers the opening
   * of another dialog). The value of the clicked button will be passed as an
   * argument to this function (if the X button is clicked, 'cancel' will
   * always be the value passed).
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.yesNoCancel = function (params) {
    return loadHandler('yesNoCancelDialog', params);
  };

  /**
   * Shows a standard dialog with an 'OK' button.
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.ok = function (params) {
    return loadHandler('okDialog', params);
  };

  /**
   * Shows a standard dialog with 'OK' and 'Cancel' buttons.
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.okCancel = function (params) {
    return loadHandler('okCancelDialog', params);
  };

  /**
   * Shows a standard dialog with a 'Cancel' button.
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.cancel = function (params) {
    return loadHandler('cancelDialog', params);
  };

  /**
   * Shows a standard dialog with 'Yes' and 'No' buttons.
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.yesNo = function (params) {
    return loadHandler('yesNoDialog', params);
  };

  /**
   * Displays the error in a standard OK dialog with 'Application Error' as the
   * dialog title.
   * @param {String|Error|Function} err the error to be displayed
   * @param {Function} [okHandler] a function to be executed when 'OK' is clicked
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.error = function error(err, okHandler) {
    okHandler = okHandler || function (callbacks) {
      callbacks.ok();
    };

    if (typeof err !== 'function') {
      baja.error(err);
    }

    hidePageLoadingMsg();
    return loadHandler('errorDialog', {
      title: mobileLex.get('appError'),
      content: err,
      ok: okHandler
    });
  };

  /**
   * Displays the error message - but the OK button will
   * refresh the entire page instead of continuing on. This represents an error
   * that should render the entire app unusable, forcing a page reload.
   *
   * @param {String|Error|Function} err the error to be displayed
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.unrecoverableError = function (err) {
    var content,
        okHandler = function okHandler() {
      location.reload();
    };

    if (typeof err === 'function') {
      content = err;
    } else {
      content = function content(contentDiv) {
        var errDiv = $('<div/>').css({ "margin-bottom": "1em" }).text(String(err)),
            msgDiv = $('<div/>').text(mobileLex.get('message.clickOKToReload'));

        contentDiv.append(errDiv);
        contentDiv.append(msgDiv);
      };
      baja.error(err);
    }

    hidePageLoadingMsg();
    return loadHandler('errorDialog', {
      title: mobileLex.get('unrecoverableError'),
      content: content,
      ok: okHandler
    });
  };

  /**
   * To be called when navigating away from a page that still has
   * unsaved changes. Will display a yes/no/cancel dialog asking the user to
   * confirm whether or not to save the outstanding changes. The dialog title
   * and content will automatically use the appropriate mobile lexicon entries.
   * 
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @param {String} params.viewName the name of the view that has unsaved
   * changes
   * @param {String|jQuery|Function} [params.redirect] the page to navigate to
   * if the user choose to save or abandon changes (i.e. clicks 'yes' or 'no'
   * and not 'cancel')
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.confirmAbandonChanges = function (params) {
    params = baja.objectify(params);

    var noop = function noop(callbacks) {
      callbacks.ok();
    },
        yes = params.yes || noop,
        no = params.no || noop,
        redirect = params.redirect;

    params.title = mobileLex.get('abandonChanges');
    params.content = mobileLex.get({
      key: 'message.confirmUnsavedChanges',
      args: [params.viewName]
    });

    //this is the page with unsaved changes - if we don't redirect in yes()
    //or no(), we go back here
    if (!params.returnPage) {
      params.returnPage = decodePageId(pages.getCurrent());
    }
    params.yes = function (callbacks) {
      if (redirect) {
        this.redirect(redirect);
      }
      yes.call(this, callbacks);
    };
    params.no = function (callbacks) {
      if (redirect) {
        this.redirect(redirect);
      }
      no.call(this, callbacks);
    };

    return exports.yesNoCancel(params);
  };

  /**
   * Displays a registered dialog page.
   * 
   * @param {String} pageId the page ID of the dialog to show (must have
   * already be registered via `registerDialog`)
   * @param {Object} params an object containing dialog parameters - content,
   * button handlers, etc.
   * @returns {niagara.util.mobile.dialogs.DialogInvocation}
   */
  exports.showDialog = function (pageId, params) {
    var handler = pages.getHandler(pageId);
    if (handler) {
      return handler.load(params);
    } else {
      return exports.error(mobileLex.get({
        key: 'dialogs.noPageRegistered',
        args: [pageId]
      }));
    }
  };

  /**
   * Binds to the window resize event - always attempts to re-center the
   * dialog vertically whenever the window is resized (or, on mobile devices,
   * orientation changed). Throttled to 500ms. 
   * 
   * This function is self-executing - will be run just
   * by including the js file.
   */
  function repositionOnResize() {
    $(window).on('resize', baja.throttle(function () {
      var page = getActivePage();
      if (page.jqmData('role') === 'dialog') {
        exports.repositionDialog(page);
      }
    }, 500));
  }

  /**
   * Prevent Pages handler from navigating us to the wrong page in case
   * we're trying to initially navigate to a dialog on first page load. If
   * the window hash contains `&ui-state=dialog` when we first load the page,
   * have JQM redirect us to `#`. (This doesn't work on Android, so you'll be
   * left with a `&ui-state=dialog` hanging on the URL, but won't do any harm.)
   * 
   * This function is self-executing - will be run just
   * by including the js file.
   */
  function noDialogOnLoad() {
    var hash = window.location.hash,
        dialogStr = '&ui-state=dialog';
    if (hash.indexOf(dialogStr) > 0) {
      baja.started(function () {
        jqm.changePage(hash.replace(dialogStr, ''), {
          changeHash: true,
          transition: 'none'
        });
      });
    }
  }

  repositionOnResize();
  noDialogOnLoad();

  return exports;
});
