var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Functions relating to the Schedule object that represents
 * a BWeeklySchedule component.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

/**
 * @private
 * @module mobile/schedule/schedule
 */
define(['baja!baja:StatusString,baja:StatusNumeric,baja:StatusEnum,' + 'baja:StatusBoolean,baja:Weekday', 'css!mobile/schedule/schedule', 'baja!', 'lex!baja', 'jquery', 'Promise', 'underscore', 'mobile/util/slot', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/fieldeditors.mobile', 'mobile/fieldeditors/mobile/feUtils', 'mobile/fieldeditors/MobileFieldEditor', 'mobile/schedule/util.schedule', 'mobile/schedule/DayEditor'], function (unusedTypes, unusedCss, baja, lexs, $, Promise, _, slotUtil, fe, mobileFE, feUtils, MobileFieldEditor, scheduleUtil) {

  "use strict";

  var getEnumRangeDisplay = feUtils.getEnumRangeDisplay,
      appendTimeLabels = scheduleUtil.appendTimeLabels,
      getLastEnteredValue = scheduleUtil.getLastEnteredValue,
      relayoutLabelsDiv = scheduleUtil.relayoutLabelsDiv,
      getFacets = slotUtil.getFacets,
      _lexs = _slicedToArray(lexs, 1),
      bajaLex = _lexs[0];

  /*
   * Returns an array of Weekday names, ordered according to the first day
   * of the week for the user's locale.
   */
  var getOrderedDayNames = _.once(function () {
    var firstDay,
        firstDayTag,
        i,
        j = 0,
        dayNames = [];

    firstDayTag = bajaLex.get('weekday.firstDayOfWeek');
    if (!firstDayTag) {
      firstDayTag = 'sunday';
    }
    firstDay = baja.$('baja:Weekday').get(firstDayTag);
    dayNames[j++] = firstDayTag;
    for (i = firstDay.getOrdinal() + 1; i <= 6; i++) {
      dayNames[j++] = baja.$('baja:Weekday').get(i).getTag();
    }
    for (i = 0; i < firstDay.getOrdinal(); i++) {
      dayNames[j++] = baja.$('baja:Weekday').get(i).getTag();
    }

    return dayNames;
  });

  /**
   * Gets a display string for a boolean schedule, taking into account
   * the trueText/falseText facets set on the schedule itself.
   *
   * @param {baja.Component} weeklySchedule a `schedule:BooleanSchedule`
   * @param {baja.Complex} statusBoolean a `baja:StatusBoolean` value used in a
   * boolean schedule
   * @returns {Promise}
   */
  function toStatusBooleanString(weeklySchedule, statusBoolean) {
    var facets = getFacets(weeklySchedule),
        bool = statusBoolean.get('value'),
        result;

    if (facets) {
      result = facets.get(bool ? 'trueText' : 'falseText');
    }

    if (result) {
      return baja.Format.format({ pattern: result });
    } else {
      return Promise.resolve(bajaLex.get(bool ? 'true' : 'false'));
    }
  }

  function toStatusNumericString(weeklySchedule, statusNumeric) {
    var facets = getFacets(weeklySchedule),
        units = facets && facets.get('units'),
        precision = facets && facets.get('precision') || 0,
        num = statusNumeric.getValue(),
        str = num.toFixed(precision);

    if (units && units !== baja.Unit.NULL) {
      var unitSymbol = units.getSymbol();
      if (units.getIsPrefix()) {
        str = unitSymbol + ' ' + str;
      } else {
        str += ' ' + unitSymbol;
      }
    }

    return Promise.resolve(str);
  }

  /**
   * A field editor displaying the seven days of a `schedule:WeeklySchedule`
   * in a calendar-like grid, like the existing schedule editor in Workbench.
   * A Schedule is not directly user-editable - it is intended to link off to
   * field editors for the individual day schedules, special events, etc.
   *
   * The `WeeklySchedule` it contains is really intended to be an unmounted
   * snapshot (retrieve using `niagara.schedule.ui.getSnapshot`), since there
   * are many different properties that need to be edited individually and
   * saved all at once. To facilitate this, `doSave` sends the entire
   * `WeeklySchedule` up to a server side handler for an all-or-nothing save operation.
   *
   * @private
   * @class
   * @alias module:mobile/schedule/schedule
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */
  var Schedule = function Schedule() {
    MobileFieldEditor.apply(this, arguments);
    this.dayEditors = {};
  };
  Schedule.prototype = Object.create(MobileFieldEditor.prototype);
  Schedule.prototype.constructor = Schedule;

  /**
   * Creates a `<div>` element to display this schedule - seven divs side by
   * side that will hold the week's seven days. (The actual editors for the
   * days themselves are instantiated and built in `doLoad`).
   *
   * @param {JQuery} dom
   */
  Schedule.prototype.doInitialize = function (dom) {
    var scheduleDiv = $('<div class="schedule" />');

    appendTimeLabels(scheduleDiv);
    $('<div class="weekdays" />').appendTo(scheduleDiv);

    scheduleDiv.appendTo(dom);
  };

  /**
   * Rebuilds all the field editors for the days in the schedule (Sunday to
   * Saturday). Day field editors are all in read-only mode (must navigate
   * to editDay/editSpecialEvent to use a writable day editor).
   *
   * @param {baja.Complex} weeklySchedule the `schedule:WeeklySchedule` to
   * load
   * @returns {Promise|undefined}
   */
  Schedule.prototype.doLoad = function (weeklySchedule) {
    if (!weeklySchedule.getType().is('schedule:WeeklySchedule')) {
      return;
    }

    var that = this,
        readonly = !that.isEnabled(),
        dom = that.jq(),
        parent = dom.parent(),
        week = weeklySchedule.get('schedule').get('week'),
        weekdaysDiv;

    dom.detach();
    weekdaysDiv = dom.find('div.weekdays').empty();

    var buildDays = _.map(getOrderedDayNames(), function (dayName) {
      var dailySchedule = week.get(dayName),
          dayContainer = $('<div class="dayContainer"/>').appendTo(weekdaysDiv);

      return fe.makeFor({
        container: dailySchedule,
        slot: 'day',
        element: dayContainer,
        readonly: readonly,
        label: bajaLex.get(dayName + '.short')
      }).then(function (editor) {
        that.dayEditors[dayName] = editor;
      });
    });

    that.defaultValue = weeklySchedule.get('defaultOutput');

    return Promise.all(buildDays).then(function () {
      dom.appendTo(parent);
    });
  };

  /**
   * Just returns `this.value()`.
   *
   * @returns {baja.Component}
   */
  Schedule.prototype.doRead = function () {
    return this.value();
  };

  /**
   * Passes our edited local instance of the `WeeklySchedule` up to the
   * server to be saved. It does this via server side call handler
   * `mobile:ScheduleServerSideCallHandler#save`.
   *
   * Note: the save() server side call returns a new snapshot of the
   * schedule just as if you had called `getSnapshot()`. However,
   * there's no way to simply return this snapshot to the promise, as field
   * editors on Components require edit-by-ref semantics (therefore the
   * return value is always the input value). After saving the schedule
   * editor, the new snapshot will be available simply as `this.snapshot`.
   * 
   * @returns {Promise}
   */
  Schedule.prototype.doSave = function () {
    var that = this,
        sched = that.value();

    return baja.Ord.make(niagara.view.ord).get().then(function (component) {
      return component.serverSideCall({
        typeSpec: 'mobile:ScheduleServerSideCallHandler',
        methodName: 'save',
        value: sched
      });
    }).then(function (snapshot) {
      if (baja.hasType(snapshot, "baja:String")) {
        throw new Error(snapshot.toString());
      }
      that.snapshot = snapshot;
      return sched;
    });
  };

  Schedule.prototype.layout = function () {
    var dom = this.jq(),
        height = dom.find('.blocksDisplay').height();

    _.each(this.dayEditors, function (dayEditor) {
      dayEditor.layout();
    });

    relayoutLabelsDiv(dom.find('.scheduleLabels'), height);
  };

  /**
   * Wipes out this schedule's collection of `DayEditor`s from Monday to
   * Friday and replaces them all with the `BTimeSchedule`s from the given day.
   **
   * @param {module:mobile/schedule/DayEditor} dayEditor the day to apply
   * Monday-Friday
   */
  Schedule.prototype.applyMF = function applyMF(dayEditor) {
    var oldDayEditor,
        that = this;

    // Iterate by weekday tag rather than array index - don't assume that a
    // particular day of week is always at a known position in the dayEditors
    // array.

    baja.iterate(getOrderedDayNames(), function (dayName) {
      if (dayName === 'saturday' || dayName === 'sunday') {
        return;
      }

      oldDayEditor = that.dayEditors[dayName];

      if (oldDayEditor.value() !== dayEditor.value()) {
        dayEditor.copyTo(oldDayEditor);
        oldDayEditor.save();
      }
    });

    this.setModified(true);
  };

  /**
   * Overwrites all time schedule data in one day with that from another day.
   *
   * @param {module:mobile/schedule/DayEditor} srcDayEditor the day whose
   * schedule blocks we want to use to overwrite the recipient day
   * @param {string} dayName the day whose schedule data will be overwritten
   */
  Schedule.prototype.overwrite = function overwrite(srcDayEditor, dayName) {
    srcDayEditor.copyTo(this.dayEditors[dayName]);

    this.setModified(true);
  };

  /**
   * Removes all data from all days.
   */
  Schedule.prototype.empty = function empty() {
    baja.iterate(this.dayEditors, function (day) {
      day.empty();
      day.save();
    });
    this.setModified(true);
  };

  /**
   * Returns a proper string representation of a StatusValue used in a weekly
   * schedule. If the status is null, returns the lexicon value for "null",
   * otherwise prints a proper string, taking into account any relevant
   * display facets set on the schedule itself.
   *
   * @param {baja.Complex} statusValue a baja:StatusValue used in a weekly
   * schedule
   * @returns {Promise} promise to be resolved with the display string
   */
  Schedule.prototype.stringify = function (statusValue) {
    if (statusValue.get('status').isNull()) {
      return Promise.resolve(bajaLex.get('Status.null'));
    }

    var type = statusValue.getType(),
        weeklySchedule = this.value(),
        value = statusValue.get('value');
    if (type.is('baja:StatusBoolean')) {
      return toStatusBooleanString(weeklySchedule, statusValue);
    } else if (type.is('baja:StatusString')) {
      return Promise.resolve(value);
    } else if (type.is('baja:StatusNumeric')) {
      return toStatusNumericString(weeklySchedule, statusValue);
    } else if (type.is('baja:StatusEnum')) {
      return getEnumRangeDisplay(value.getOrdinal(), weeklySchedule.get('facets').get('range'));
    } else {
      return Promise.resolve("N/A");
    }
  };

  /**
   * Decides what the default value should be when creating a new schedule
   * block.
   *
   * If the user has not yet created a new schedule block, then use the
   * default output value from the currently edited Schedule.
   *
   * Otherwise, remember the last value the user entered
   * (`util.schedule.getLastEnteredValue` - set in
   * `niagara.schedule.ui.editSelectedBlock()`) and reuse that.
   *
   * @returns {baja.Struct} the `baja:StatusValue` to use when creating a new
   * block
   */
  Schedule.prototype.getNewBlockValue = function getNewBlockValue() {
    if (getLastEnteredValue()) {
      return getLastEnteredValue().newCopy();
    }

    var weeklySchedule = this.value(),
        defaultValue = weeklySchedule.get('defaultOutput').newCopy(),
        value = defaultValue.getValue(),
        facets = getFacets(weeklySchedule),
        min,
        max,
        range;

    defaultValue.setStatus(baja.Status.ok);

    if (defaultValue.getType().is('baja:StatusBoolean')) {
      defaultValue.setValue(!value);
    } else if (defaultValue.getType().is('baja:StatusNumeric')) {
      value = 0;
      if (facets) {
        min = facets.get('min');
        max = facets.get('max');
        if (value < min) {
          value = min;
        } else if (value > max) {
          value = max;
        }
        defaultValue.setValue(value);
      }
    } else if (defaultValue.getType().is('baja:StatusEnum')) {
      if (value.getType().is('baja:DynamicEnum')) {

        range = facets && facets.get('range');
        if (range) {
          defaultValue.setValue(baja.DynamicEnum.make({
            range: range
          }));
        }
      }
    }
    return defaultValue;
  };

  fe.register('schedule:WeeklySchedule', Schedule);

  return Schedule;
});
