/**
 * @file Functions relating to the DayEditor field editor for a
 * `schedule:DaySchedule` object.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/schedule/DayEditor
 */
define(['baja!schedule:TimeSchedule', 'css!mobile/schedule/DayEditor', 'baja!', 'jquery', 'underscore', 'mobile/schedule/util.schedule', 'mobile/schedule/ScheduleBlock', 'mobile/schedule/schedule.ui.day', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/BaseFieldEditor'], function (typesUnused, cssUnused, baja, $, _, scheduleUtil, ScheduleBlock, dayUI, fe, BaseFieldEditor) {

  "use strict";

  /**
   * Represents one day in a schedule object.
   *
   * @private 
   * @class
   * @alias module:mobile/schedule/DayEditor
   * @extends module:mobile/fieldeditors/BaseFieldEditor
   */

  var DayEditor = function DayEditor() {
    BaseFieldEditor.apply(this, arguments);
    this.scheduleBlocks = [];
    this.$dayDisplay = this.params.label || this.container.getDisplayName();
    this.dragStart = null;
  };
  DayEditor.prototype = Object.create(BaseFieldEditor.prototype);
  DayEditor.prototype.constructor = DayEditor;

  /**
   * Creates a `<div>` element to display this day - a vertical stack of eight
   * blocks (representing three hours apiece) on top of which additional divs
   * for the schedule blocks will be overlaid.
   *
   * If this editor is not configured as readonly (readonly parameter passed
   * into field editor constructor), then will arm event handlers to enable
   * creating, dragging, and resizing schedule blocks.
   * 
   * @param {JQuery} dom
   */
  DayEditor.prototype.doInitialize = function (dom) {
    var dayDiv,
        blocksDiv,
        container = this.container,
        name = container.getName(),
        displayName = this.$dayDisplay;

    dayDiv = $('<div class="day" />').addClass(name) //"monday"
    .append($('<div class="dayTitle"/>').text(displayName)); //"Monday"

    blocksDiv = $('<div class="blocksDisplay"/>').appendTo(dayDiv);
    blocksDiv.append(_.range(0, 8).map(function (i) {
      return '<div class="block block' + i + ' {name}"/>';
    }).join(''));

    dayDiv.data('daySchedule', this.value());

    dayDiv.appendTo(dom);

    if (this.isEnabled()) {
      dayUI.armHandlers(this, dayDiv);
    }
  };

  /**
   * Given a `DaySchedule`, converts all of its current
   * `schedule:TimeSchedule` slots into `ScheduleBlock`s to be displayed and
   * edited.
   *
   * @param {baja.Complex} daySchedule the `schedule:DaySchedule` to be
   * edited
   */
  DayEditor.prototype.doLoad = function (daySchedule) {
    var that = this,
        blocksDiv = $('.blocksDisplay', that.jq());

    that.empty();

    daySchedule.getSlots().is('schedule:TimeSchedule').eachValue(function (timeSchedule) {
      that.addBlock(timeSchedule.get('start'), timeSchedule.get('finish'), timeSchedule.get('effectiveValue'));
    });

    that.setModified(false);

    $('.scheduleBlockContainer', blocksDiv).remove();

    this.getBlocks().forEach(function (block) {
      return block.generateDiv(blocksDiv);
    });
  };

  /**
   * Save data for a DayEditor is an array of all the `ScheduleBlocks` it
   * currently contains.
   *
   * @returns {Array.<module:mobile/schedule/ScheduleBlock>} an array of this
   * day editor's schedule blocks
   */
  DayEditor.prototype.doRead = function () {
    return this.scheduleBlocks;
  };

  /**
   * Sets the `schedule:TimeSchedule` slots on the currently edited
   * `schedule:DaySchedule` to reflect the day editor's current set of
   * schedule blocks (returned from `doRead()`).
   * @param {Array.<module:mobile/schedule/ScheduleBlock>} scheduleBlocks
   */
  DayEditor.prototype.doSave = function (scheduleBlocks) {
    var value = this.value();

    value.getSlots().is('schedule:TimeSchedule').each(function (slot) {
      value.remove({ slot: slot });
    });

    scheduleBlocks.forEach(function (block) {
      value.add({ slot: 'time?', value: block.createTimeSchedule() });
    });
  };

  /**
   * Wipes out and redraws all schedule block divs. Will be called once
   * automatically when the editor is instantiated; should be called again after
   * removing or adding new blocks.
   */
  DayEditor.prototype.refreshWidgets = function refreshWidgets() {
    var blocksDiv = $('.blocksDisplay', this.jq());
    $('.scheduleBlockContainer', blocksDiv).remove();

    this.getBlocks().forEach(function (block) {
      return block.generateDiv(blocksDiv);
    });
  };

  /**
   * Adds a new schedule block to this day - does not make any changes to the
   * underlying component. Should only be called when constructing the object
   * or adding new schedule blocks.
   * 
   * @param {baja.Time} start the start time
   * @param {baja.Time} finish the finish time
   * @param {baja.Struct} value the effective value (`baja:StatusValue`)
   * @return {module:mobile/schedule/ScheduleBlock} the block that was added
   */
  DayEditor.prototype.addBlock = function (start, finish, value) {
    var block = new ScheduleBlock(start, finish, value);
    this.scheduleBlocks.push(block);
    this.setModified(true);
    return block;
  };

  /**
   * Empties out any other events from this day and converts the given block
   * to a midnight-to-midnight 24-hour event.
   * 
   * @param {baja.Struct} value the value to set for the entirety of this day
   * (`baja:StatusValue`)
   */
  DayEditor.prototype.allDayEvent = function (value) {
    this.empty();

    this.addBlock(baja.Time.make(0), baja.Time.make(0), value);
  };

  /**
   * Checks to see if the given block is able to set its own start/finish to
   * the given start/finish times. Ensures that setting the times will not
   * result in an overlap/conflict with another schedule block. 
   * 
   * This method should be called upon a drag
   * or resize on a schedule block, and the drag or resize should be cancelled
   * if this method returns false.
   * 
   * @param {module:mobile/schedule/ScheduleBlock} block the block attempting a
   * move/resize
   * @param {Number} startms the block's desired start time (in ms past midnight)
   * @param {Number} finishms the block's desired finish time (in ms past midnight)
   * @return {Boolean} true if the block can move to this date range
   */
  DayEditor.prototype.canMoveBlockTo = function (block, startms, finishms) {
    return !baja.iterate(this.scheduleBlocks, function (myBlock) {
      if (myBlock && myBlock !== block && myBlock.overlaps(startms, finishms)) {
        return true;
      }
    });
  };

  /**
   * Overwrites all blocks of the given day editor with the blocks from this day
   * editor.
   * 
   * @param {module:mobile/schedule/DayEditor} destDayEditor the day editor we
   * wish to overwrite
   */
  DayEditor.prototype.copyTo = function (destDayEditor) {
    var that = this;

    destDayEditor.empty();

    baja.iterate(that.getBlocks(), function (block) {
      destDayEditor.addBlock(block.start, block.finish, block.value);
    });
  };

  /**
   * Removes all schedule blocks from this day (and from the underlying
   * `DaySchedule` component).
   */
  DayEditor.prototype.empty = function () {
    this.scheduleBlocks = [];
    this.setModified(true);
  };

  /**
   * Returns the schedule blocks contained in this day.
   * 
   * @return {Array.<module:mobile/schedule/ScheduleBlock>}
   */
  DayEditor.prototype.getBlocks = function () {
    return this.scheduleBlocks;
  };

  /**
   * Removes the specified block from this day (and from the underlying
   * `DaySchedule` component).
   * 
   * @param {module:mobile/schedule/ScheduleBlock} block the block to be removed
   * @return {module:mobile/schedule/ScheduleBlock} the block that was removed
   */
  DayEditor.prototype.removeBlock = function (block) {
    var scheduleBlocks = this.scheduleBlocks,
        index = _.indexOf(scheduleBlocks, block);

    if (index >= 0) {
      scheduleBlocks.splice(index, 1);
      this.setModified(true);
      return block;
    }
  };

  /**
   * Returns the currently selected/highlighted schedule block.
   * 
   * @return {module:mobile/schedule/ScheduleBlock} the currently selected block, or
   * undefined if none selected
   */
  DayEditor.prototype.getSelectedBlock = function () {
    return this.jq().find('.scheduleBlockContainer.selected').data('scheduleBlock');
  };

  DayEditor.prototype.layout = function () {
    scheduleUtil.relayoutDayDiv(this.jq().children('.day'));
  };

  fe.register('schedule:DaySchedule', DayEditor);

  return DayEditor;
});
