/**
 * @file History app utilities
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */
/*jshint strict: false */
/*global niagara*/

define(['baja!baja:AbsTimeRange,mobile:HistoryServerSideHandler,' + 'bql:DynamicTimeRangeType', 'baja!', 'lex!mobile', 'jquery', 'Promise', 'mobile/util/ord', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/fieldeditors.mobile', 'mobile/fieldeditors/MobileFieldEditor'], function (types, baja, lexs, $, Promise, ordUtil, fe, mobileFE, MobileFieldEditor) {

  //imports
  var MILLIS_IN_DAY = baja.RelTime.MILLIS_IN_DAY,


  //local vars
  mobileLex = lexs[0],


  //constants
  ABS_TIME_RANGE_TYPE = 'baja:AbsTimeRange',
      BQL_DYNAMIC_TIME_RANGE_TYPE = 'bql:DynamicTimeRangeType',
      MTD_GET_HISTORY_RANGE = 'getHistoryRange',
      SSH_TYPE = 'mobile:HistoryServerSideHandler',
      START_AFTER_END_LEX_KEY = 'history.message.startAfterEnd',
      STRING_TYPE = 'baja:String',
      TIME_RANGE_TAG_NAME = 'timeRange',
      TIME_RANGE_REGEX = /start=([^;]*);end=([^;\|]*)/,
      PERIOD_REGEX = /period=([^;\|%]*)/,


  //exports
  HistoryQueryParamsEditor;

  /**
   * Attempts to parse a period ID from a history query params string using the
   * `period` param.
   * 
   * @memberOf niagara.util.mobile.history
   * @param {String} a history query params string
   * @returns {String} the period ID, or undefined if could not be found
   */
  function parsePeriodId(paramString) {
    var periodMatch = PERIOD_REGEX.exec(paramString);
    if (periodMatch) {
      return periodMatch[1];
    }
  }

  /**
   * Attempts to parse a `baja:AbsTimeRange` from a history query params string
   * using the `start` and `end` params.
   * 
   * @memberOf niagara.util.mobile.history
    * @param {String} a history query params string
   * @returns {baja.Struct} a `baja:AbsTimeRange`, or undefined if could not be
   * parsed
   */
  function parseStartEnd(paramString) {
    var rangeMatch = TIME_RANGE_REGEX.exec(decodeURI(paramString)),
        decodeFromString = baja.AbsTime.DEFAULT.decodeFromString;

    if (rangeMatch) {
      return baja.$(ABS_TIME_RANGE_TYPE, {
        startTime: decodeFromString(decodeURIComponent(rangeMatch[1])),
        endTime: decodeFromString(decodeURIComponent(rangeMatch[2]))
      });
    }
  }

  /**
   * Constructs a `BAbsTimeRange` using a tag from a  `BDynamicTimeRangeType`,
   * relative to the current system time.
   * 
   * @private
   * @memberOf niagara.util.mobile.history
   * @param {String} tag the selected tag from a `BDynamicTimeRangeType`
   * @returns {baja.Struct} a `baja:AbsTimeRange`
   */
  function getDynamicAbsTimeRange(tag) {
    var startDate,
        endDate,
        now = new Date(),
        year = now.getFullYear(),
        month = now.getMonth(),
        day = now.getDate(),
        dayOfWeek = now.getDay(),
        time = now.getTime(),

    // convert offset from minutes to millis, and JS returns a negative 
    // value for some reason
    offset = now.getTimezoneOffset() * -60000;

    switch (tag) {

      case 'today':
        startDate = new Date(year, month, day);
        endDate = new Date(year, month, day + 1);
        break;

      case 'last24Hours':
        startDate = new Date(time - MILLIS_IN_DAY);
        endDate = now;
        break;

      case 'yesterday':
        startDate = new Date(year, month, day - 1);
        endDate = new Date(year, month, day);
        break;

      case 'weekToDate':
        startDate = new Date(year, month, day - dayOfWeek);
        endDate = now;
        break;

      case 'lastWeek':
        startDate = new Date(year, month, day - dayOfWeek - 7);
        endDate = new Date(year, month, day - dayOfWeek);
        break;

      case 'last7Days':
        startDate = new Date(year, month, day - 7);
        endDate = new Date(year, month, day);
        break;

      case 'monthToDate':
        startDate = new Date(year, month, 1);
        endDate = now;
        break;

      case 'lastMonth':
        startDate = new Date(year, month - 1, 1);
        endDate = new Date(year, month, 1);
        break;

      case 'yearToDate':
        startDate = new Date(year, 0, 1);
        endDate = now;
        break;

      case 'lastYear':
        startDate = new Date(year - 1, 0, 1);
        endDate = new Date(year, 0, 1);
        break;

      default:
        startDate = now;endDate = now;
    }

    return baja.$('baja:AbsTimeRange', {
      startTime: baja.AbsTime.make({ jsDate: startDate, offset: offset }),
      endTime: baja.AbsTime.make({ jsDate: endDate, offset: offset })
    });
  }

  /**
   * @memberOf niagara.util.mobile.history
   */
  function toTimeRange(paramString) {
    var timeRange = parseStartEnd(paramString),
        periodId;

    if (timeRange) {
      return timeRange;
    } else {
      periodId = parsePeriodId(paramString);
      if (periodId) {
        return getDynamicAbsTimeRange(periodId);
      }
    }
  }

  /**
   * Takes a value (either a ` BAbsTimeRange` or a range type tag) and,
   * optionally, delta mode, and converts them into a usable history query
   * param string.
   * 
   * @memberOf niagara.util.mobile.history
   * @param {String|baja.Struct} value a `baja:AbsTimeRange` or
   * * `BDynamicTimeRangeType` tag to convert to a history query param string
   * @param {Boolean} isDelta true if delta mode is turned on
   */
  function toParamString(value, isDelta) {
    var paramString = '',
        deltaString = isDelta ? 'delta=true' : '',
        start,
        end,
        startTime,
        endTime;

    if (value.getType().is(ABS_TIME_RANGE_TYPE)) {
      start = value.getStartTime();
      end = value.getEndTime();
      startTime = start.encodeToString();
      endTime = end.encodeToString();

      paramString = '?start=' + startTime + ';end=' + endTime;
    } else if (value !== 'none' && value !== 'timeRange') {
      paramString = '?period=' + value;
    }

    if (isDelta) {
      if (paramString) {
        paramString += ';' + deltaString;
      } else {
        paramString = '?' + deltaString;
      }
    }

    return paramString;
  }

  /**
   * Retrieves the default date range for the currently viewed history from the
   * server, using the `mobile:HistoryServerSideHandler`'s `getHistoryRange`
   * method.
   * 
   * @memberOf niagara.util.mobile.history
   * @private
   * @returns {Promise} promise to be resolved with the default `BAbsTimeRange`
   */
  function getDefaultTimeRange() {
    //set our initial time query range
    return baja.Ord.make(niagara.view.app.appOrd).get({ lease: true }).then(function (component) {
      return component.serverSideCall({
        typeSpec: SSH_TYPE,
        methodName: MTD_GET_HISTORY_RANGE,
        value: baja.Ord.make(niagara.view.history.query)
      });
    });
  }

  /**
   * A field editor for a parameter string used by a history query. Includes
   * a dropdown for a `BDynamicTimeRangeType` and an editor for a
   * `BAbsTimeRange` (for when "Time Range" is selected).
   * 
   * Registered on `baja:String` and instantiated using key
   * `"historyQueryParams"`.
   *  
   * @class
   * @name niagara.util.mobile.history.HistoryQueryParamsEditor
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  HistoryQueryParamsEditor = function HistoryQueryParamsEditor() {

    /**
     * Parses out information from a history query params string. Once parsed,
     * an array will be resolved with the following two members:
     * 
     * - `timeRange`: the `BAbsTimeRange` represented by these history params
     *   (either encoded directly into the string using `start` and `end`
     *   parameters, or if the string represents a period, a time range
     *   constructed via `niagara.util.mobile.history.getDynamicAbsTimeRange`)
     * - `periodId`: if the value is a params string representing a period
     *   (using a `BDynamicTimeRangeType`), this will be the period ID string,
     *   e.g. `last24Hours` - otherwise undefined
     * - `isDelta`: true if delta mode is turned on
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor.parseHistoryParams
     * @function
     * @private
     * @inner
     * @param {String|baja.Struct} value either a history params string
     * (or an ORD containing a history params string), or a `baja:AbsTimeRange`
     * to convert to a history params string.
     * @returns {Promise} promise to be resolved with an array containing 
     * time range, period id, and delta
     */
    function parseHistoryParams(value) {
      var type = value.getType(),
          periodId,
          isDelta = false,
          timeRange;

      if (type.is(STRING_TYPE)) {
        //we have a params string!
        periodId = parsePeriodId(value);
        if (!periodId) {
          //it's not a period, so try and parse it from start/end params
          timeRange = parseStartEnd(value);
          //set period dropdown accordingly
          if (timeRange) {
            periodId = 'timeRange';
          }
        }
        // else { if it IS a period, we're going to pull down the default range 
        // from the station }

        isDelta = value.indexOf('delta=true') >= 0;
      } else if (type.is(ABS_TIME_RANGE_TYPE)) {
        timeRange = value;
      }

      if (timeRange) {
        return Promise.resolve([timeRange, periodId, isDelta]);
      } else {
        // no time range, no start/end params - so pull down a
        // default time range from the station
        return getDefaultTimeRange().then(function (timeRange) {
          return [timeRange, periodId, isDelta];
        });
      }
    }

    /**
     * Creates a JQM select dropdown containing the different values of a
     * `bql:DynamicTimeRangeType`. These tags will be appended directly into
     * the history query as `periodId`.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor.makeSelect
     * @function
     * @private
     * @inner
     */
    function makeSelect() {
      var option = $('<option value="none"/>').text(mobileLex.get('loading')),
          select = $('<select data-role="selectmenu" data-theme="b"/>');

      select.append(option);

      return select;
    }

    function populateSelect(select) {
      var rangeType = baja.$(BQL_DYNAMIC_TIME_RANGE_TYPE),
          range = rangeType.getRange();

      select.empty();
      $('<option/>').attr('value', 'none').text(mobileLex.get('history.selectTimeRange')).appendTo(select);

      baja.iterate(range.getOrdinals(), function (ordinal) {
        $('<option/>').attr('value', range.getTag(ordinal)).text(rangeType.make(ordinal).getDisplayTag()).appendTo(select);
      });

      select.trigger('change');
    }

    var deltaButtonHTML = '<div class="deltaButtonContainer">' + '<a data-theme="a" data-role="button" data-icon="delta" title="Delta" ' + 'data-iconpos="notext"/>' + '</div>';
    /**
     * Appends the `BDynamicTimeRangeType` select, and (optionally) a toggle
     * button for delta mode, then instantiates the default editor for a
     * `BAbsTimeRange` below it.
     * 
     * Whenever the user selects a new value from the `BDynamicTimeRangeType`
     * dropdown, if the user selects anything but "Time Range" (manually
     * entering times), set the `BAbsTimeRange` field editor to readonly.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#doInitialize
     * @function
     * @see niagara.fieldEditors.BaseFieldEditor#doInitialize
     */
    function doInitialize(dom) {
      var that = this,
          select = that.$select = makeSelect(),
          deltaButton = $(deltaButtonHTML);

      that.$deltaButton = deltaButton.children('a');

      dom.append(select);

      if (that.params.showDelta) {
        dom.append(deltaButton);
      }

      dom.addClass('historyQueryParams');

      select.change(function () {
        that.updateReadonlyStatus();
      });

      deltaButton.click(function () {
        that.setDelta(!that.$isDelta);
      });
    }

    /**
     * Loads up a history query params string. The `BAbsTimeRange` parsed from
     * the params string will be loaded into the `BAbsTimeRange` field editor.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#doLoad
     * @function
     * @param {String} value a history query params string
     * @see niagara.fieldEditors.BaseFieldEditor#doLoad
     * @see niagara.util.mobile.history.HistoryQueryParamsEditor.parseHistoryParams
     */
    function doLoad(value) {
      var that = this;

      return parseHistoryParams(value).spread(function (timeRange, periodId, isDelta) {
        populateSelect(that.$select);

        if (periodId) {
          that.$select.val(periodId).selectmenu('refresh');
          that.updateReadonlyStatus();
        }

        that.setDelta(isDelta);

        return that.makeChildFor({
          value: timeRange
        }).then(function (editor) {
          that.$editor = editor;
        });
      });
    }

    /**
     * If "Time Range" is selected, then the `BAbsTimeRange`'s current value
     * will be converted to a `start`/`end` history query param string.
     * Otherwise, the currently selected tag in the dropdown will be converted
     * into a `period` param string. 
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#doRead
     * @function
     * @see niagara.fieldEditors.BaseFieldEditor#doRead
     */
    function doRead() {
      var that = this,
          selectedTag = that.$select.val(),
          delta = that.$isDelta;

      if (selectedTag === TIME_RANGE_TAG_NAME) {
        return this.$editor.read().then(function (timeRange) {
          return toParamString(timeRange, delta);
        });
      } else {
        return toParamString(selectedTag, delta);
      }
    }

    /**
     * Check to see if the given time range is valid for a history query (that
     * the start time is before the end time).
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#validate
     * @function
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#validate
     */
    function validate(paramString) {
      return new Promise(function (resolve, reject) {
        var timeRange = parseStartEnd(paramString),
            startDate,
            startMillis,
            endDate,
            endMillis,
            show;

        if (!timeRange) {
          // we're a period instead of a start/end range - so can't be invalid
          resolve();
          return;
        }

        startDate = timeRange.getStartTime();
        startMillis = startDate.getJsDate().getTime();
        endDate = timeRange.getEndTime();
        endMillis = endDate.getJsDate().getTime();

        if (startMillis >= endMillis) {
          show = baja.TimeFormat.SHOW_DATE | baja.TimeFormat.SHOW_TIME;
          startDate.toDateTimeString({
            show: show,
            ok: function ok(startString) {
              endDate.toDateTimeString({
                show: show,
                ok: function ok(endString) {
                  reject(mobileLex.get({
                    key: START_AFTER_END_LEX_KEY,
                    args: [startString, endString]
                  }));
                }
              });
            }
          });
        } else {
          resolve();
        }
      });
    }

    function postCreate() {

      this.$isDelta = false;

      /**
       * Updates the `BAbsTimeRange` editor's readonly status - `false` if
       * "Time Range" is selected from the dropdown, and true otherwise.
       * 
       * @name niagara.util.mobile.history.HistoryQueryParamsEditor#updateReadonlyStatus
       * @function
       * @private
       */
      this.updateReadonlyStatus = function () {
        var that = this,
            editor = that.$editor,
            selectedTag,
            timeRangeSelected;

        if (editor) {
          selectedTag = that.$select.val();
          timeRangeSelected = selectedTag === TIME_RANGE_TAG_NAME;

          editor.setReadonly(!timeRangeSelected);
        }
      };

      /**
       * Sets/unsets delta mode. This automatically updates the highlighting for
       * the delta button as well.
       * 
       * @name niagara.util.mobile.history.HistoryQueryParamsEditor#setDelta
       * @function
       * @param {Boolean} delta
       * @private
       */
      this.setDelta = function (delta) {
        this.$isDelta = delta;

        this.$deltaButton.toggleClass('ui-btn-active', !!delta);

        this.setModified(true);
      };
    }

    return fe.defineEditor(MobileFieldEditor, {
      doInitialize: doInitialize,
      doLoad: doLoad,
      doRead: doRead,
      postCreate: postCreate,
      validate: validate
    });
  }();

  fe.register(STRING_TYPE, HistoryQueryParamsEditor, 'historyQueryParams');

  /**
   * @namespace
   * @name niagara.util.mobile.history
   */
  return {
    HistoryQueryParamsEditor: HistoryQueryParamsEditor,
    parsePeriodId: parsePeriodId,
    parseStartEnd: parseStartEnd,
    toParamString: toParamString,
    toTimeRange: toTimeRange
  };
});
