/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/* eslint-env browser */
define(['jquery'], function ($) {
  'use strict';
  /**
   * API Status: **Private**
   * @exports nmodule/bajaui/rc/ux/shape/shapeUtil
   */

  var exports = {};
  /**
   * Gets the needed information for rotating a set of points and the information to rotate a shape's gradient information to match the rotation.
   * This rotation helps display gradients better. If no rotation is required then an empty object is returned.
   * @param {Object} params
   * @param {Array.<{ x:Number, y:Number}>} params.points
   * @param {Number} [params.pointAngle] if provided, sets the desired point angle (in degrees) instead of calculating it from the point data.
   * @param {boolean} [params.avoidGradientDistortion=false] if provided, does not rotate the data unless the width or height is less than 1.
   *
   * @return {{} | {rotatedPoints: [], transformParams: {linearGradientTransform: string, radialXFactor: number, radialYFactor: number, radialGradientTransform: string, patternTransform: string}, group: Element}}
   */

  exports.getRotateInfo = function (_ref) {
    var points = _ref.points,
        pointAngle = _ref.pointAngle,
        _ref$avoidGradientDis = _ref.avoidGradientDistortion,
        avoidGradientDistortion = _ref$avoidGradientDis === void 0 ? false : _ref$avoidGradientDis;

    //need at least 2 points to get an angle
    if (points.length < 2) {
      return {};
    }

    var rotatedPoints = [];

    var _getInfo = getInfo(points),
        centerX = _getInfo.centerX,
        centerY = _getInfo.centerY,
        width = _getInfo.width,
        height = _getInfo.height;

    if (avoidGradientDistortion) {
      if (height > 1 && width > 1) {
        return {};
      }
    } //utilize the first two points to decide the angle of rotation if its not provided as an argument


    var actualAngle = pointAngle;

    if (pointAngle === undefined) {
      actualAngle = exports.getAngle(points);
    } //flat or vertical polygons may report the wrong angle, here but they will greatly benefit from rotation


    if (avoidGradientDistortion && width < 1 && height > 1) {
      actualAngle = 0;
    }

    if (avoidGradientDistortion && width > 1 && height < 1) {
      actualAngle = 90;
    }

    var rotateAngle = actualAngle % 90;
    var angle = 45 - rotateAngle;

    if (angle % 90 === 0) {
      angle = 0;
    }

    var transformAngle = -angle;

    for (var i = 0; i < points.length; i++) {
      var _points$i = points[i],
          x = _points$i.x,
          y = _points$i.y;
      rotatedPoints.push(exports.rotatePoint({
        x: x,
        y: y,
        cx: centerX,
        cy: centerY,
        angle: angle
      }));
    } //Anything already at a 45 degree angle is already OK


    if (angle === 0) {
      return {};
    }

    var radians = Math.PI / 180 * angle;
    var a = Math.abs(width * Math.sin(radians)) + Math.abs(height * Math.cos(radians));
    var b = Math.abs(width * Math.cos(radians)) + Math.abs(height * Math.sin(radians));
    var radialScale = Math.max(width, height) / a;
    var scaleHeight = height / b;
    var scaleWidth = width / a;
    var radialGradientTransform = 'translate(0.5, 0.5) rotate(' + angle + ') scale(' + radialScale + ', ' + radialScale + ') translate(-0.5, -0.5)';
    var radialXFactor = 1;
    var radialYFactor = 1;

    if (width > height) {
      radialYFactor = height / width;
    } else {
      radialXFactor = width / height;
    }

    var linearGradientTransform, patternTransform;
    var rotateString = '';

    if (angle) {
      rotateString = 'rotate(' + angle + ')';
      patternTransform = "rotate(" + angle + ", " + centerX + ", " + centerY + ")";
    }

    if (scaleWidth && scaleHeight && (scaleWidth !== 1 || scaleHeight !== 1) && isFinite(scaleHeight) && isFinite(scaleWidth)) {
      linearGradientTransform = 'translate(0.5, 0.5) ' + rotateString + ' scale(' + scaleWidth + ', ' + scaleHeight + ') translate(-0.5, -0.5)';
    } else if (angle) {
      linearGradientTransform = "rotate(" + angle + ", 0.5, 0.5)";
    }

    var transform = "rotate(" + transformAngle + ", " + centerX + "," + centerY + ")"; //this is the group that will correct the rotation to the points

    var group = document.createElementNS('http://www.w3.org/2000/svg', 'g');

    if (angle) {
      group.setAttribute("transform", transform);
    }

    var transformParams = {
      linearGradientTransform: linearGradientTransform,
      radialGradientTransform: radialGradientTransform,
      radialXFactor: radialXFactor,
      radialYFactor: radialYFactor,
      patternTransform: patternTransform
    };
    return {
      rotatedPoints: rotatedPoints,
      group: group,
      transformParams: transformParams
    };
  };
  /**
   *
   * @param {Object} params
   * @param {Number} params.x
   * @param {Number} params.y
   * @param {Number} params.cx
   * @param {Number} params.cy
   * @param {Number} params.angle
   * @return {{x: Number, y: Number}}
   */


  exports.rotatePoint = function (_ref2) {
    var x = _ref2.x,
        y = _ref2.y,
        cx = _ref2.cx,
        cy = _ref2.cy,
        angle = _ref2.angle;
    var radians = Math.PI / 180 * angle,
        cos = Math.cos(radians),
        sin = Math.sin(radians),
        nx = cos * (x - cx) - sin * (y - cy) + cx,
        ny = cos * (y - cy) + sin * (x - cx) + cy;
    return {
      x: nx,
      y: ny
    };
  };
  /**
   * For an Array of Points, get the angle of the first two points.
   * @param {Array.<{ x:Number, y:Number}>} points
   * @return {Number} degrees
   */


  exports.getAngle = function (points) {
    if (points.length < 2) {
      return 0;
    }

    var dy = points[1].y - points[0].y;
    var dx = points[1].x - points[0].x;
    var theta = Math.atan2(dy, dx);
    theta *= 180 / Math.PI;
    return theta;
  };
  /**
   * @param {Array.<{ x:Number, y:Number}>} points
   * @return {string}
   */


  exports.pointsToString = function (points) {
    var result = '';

    for (var i = 0; i < points.length; i++) {
      if (i > 0) {
        result += ' ';
      }

      result += points[i].x + "," + points[i].y;
    }

    return result;
  };
  /**
   * Return true if the Shape has a gradient.
   * @private
   * @return {boolean}
   */


  exports.isGradient = function (shape) {
    var properties = shape.properties(),
        strokeBrush = properties.getValue('stroke'),
        fillBrush = properties.getValue('fill');
    return !!(strokeBrush && strokeBrush.isGradientBrush()) || fillBrush && fillBrush.isGradientBrush();
  };
  /**
   * @param {Array.<{ x:Number, y:Number}>} points
   * @return {{centerY: number, centerX: number, width: number, height: number}}
   */


  function getInfo(points) {
    var minX = Number.POSITIVE_INFINITY,
        maxX = Number.NEGATIVE_INFINITY,
        minY = Number.POSITIVE_INFINITY,
        maxY = Number.NEGATIVE_INFINITY;

    for (var i = 0; i < points.length; i++) {
      var _points$i2 = points[i],
          x = _points$i2.x,
          y = _points$i2.y;
      minX = Math.min(minX, x);
      maxX = Math.max(maxX, x);
      minY = Math.min(minY, y);
      maxY = Math.max(maxY, y);
    }

    return {
      centerX: (minX + maxX) / 2,
      centerY: (minY + maxY) / 2,
      width: maxX - minX,
      height: maxY - minY
    };
  }

  return exports;
});
