/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Cody Short
 */

/*eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/bajaui/rc/ux/ToolTip
 */
define(['jquery', 'css!nmodule/bajaui/rc/bajaui'], function ($) {
  'use strict';
  /**
   * @typedef {Object} module:nmodule/bajaui/rc/ux/ToolTip~hooks
   *
   * @property {module:nmodule/bajaui/rc/ux/ToolTip~setDisplay} setDisplay sets
   * the value currently displayed on the ToolTip.
   * @property {module:nmodule/bajaui/rc/ux/ToolTip~cancel} cancel this function
   * will cancel the tooltip. This will give rid of the created ToolTip and
   * remove it from the DOM.
   */

  /**
   * A callback that will update the value displayed on the ToolTip
   *
   * @callback module:nmodule/bajaui/rc/ux/ToolTip~setDisplay
   * @param {*} value the value to display. This value will be converted to a
   * string.
   */

  /**
   * A callback to be invoked when the given ToolTip is cancelled whether done
   * by the ToolTip itself or the cancel hook has been invoked.
   *
   * @callback module:nmodule/bajaui/rc/ux/ToolTip~onCancel
   * @param {String} [reason] the reason the ToolTip was cancelled. This can
   * either be specified via the call to cancel or will be decided by ToolTip
   * if available. Values generated via ToolTip itself include `multitouch` when
   * the user touches the screen with more than one finger. `end` when the user
   * has performed a standard end gesture such as touchup/mouseup.
   */

  /**
   * A callback that will cancel the ToolTip.
   *
   * @callback module:nmodule/bajaui/rc/ux/ToolTip~cancel
   * @param {String} [reason] provides a way for the caller to specify the reason
   * for the ToolTip to be cancelled. This will be passed along to the onCancel
   * callback.
   */

  var MOVE_EVENTS = 'mousemove touchmove',
      CANCEL_EVENTS = 'mouseup touchend touchcancel',
      TOUCH_Y_OFFSET = -50,
      MOUSE_Y_OFFSET = 15;
  /**
   * Creates a ToolTip that will follow the pointer around the screen (mouse or
   * touch). This ToolTip with remove itself when the user releases the pointer.
   *
   * This ToolTip will also exit early / not be created if more than one touch
   * is detected.
   *
   * @param {MouseEvent|TouchEvent} e
   * @param {*} value the value to display in the ToolTip. This value will be
   * converted to a string.
   * @params {Object} [params]
   * @params {module:nmodule/bajaui/rc/ux/ToolTip~onCancel} [params.onCancel]
   * called when the given ToolTip is cancelled.
   * @returns {module:nmodule/bajaui/rc/ux/ToolTip~hooks|undefined} returns hooks
   * if the ToolTip was created.
   */

  function create(e, value, params) {
    if (e.touches && e.touches.length > 1) {
      return;
    }

    function updatePosition(e) {
      if (e.touches && e.touches.length) {
        if (e.touches.length > 1) {
          cancel('multitouch');
          return;
        }

        var touch = e.touches[0];
        dom.css('left', "".concat(touch.pageX, "px"));
        dom.css('top', "".concat(touch.pageY + TOUCH_Y_OFFSET, "px"));
      } else {
        var toolTipBottomPositionIfBelow = $(window).height() - (MOUSE_Y_OFFSET + dom.height()),
            isToolTipAboveMouse = e.clientY > toolTipBottomPositionIfBelow,
            appliedYOffset = isToolTipAboveMouse ? -MOUSE_Y_OFFSET - dom.height() : MOUSE_Y_OFFSET;
        dom.css('left', "".concat(e.clientX, "px"));
        dom.css('top', "".concat(e.clientY + appliedYOffset, "px"));
      }
    }

    var cancelEventHandler = function cancelEventHandler() {
      return cancel('end');
    };

    function cancel(reason) {
      params && params.onCancel && params.onCancel(reason);
      dom.remove();
      var body = $('body');
      body.off(CANCEL_EVENTS, cancelEventHandler);
      body.off(MOVE_EVENTS, updatePosition);
    }

    function createToolTip() {
      var dom = $("<div class=\"ux-ToolTip\"></div>"),
          body = $('body');
      body.append(dom);
      body.on(MOVE_EVENTS, updatePosition);
      body.on(CANCEL_EVENTS, cancelEventHandler);
      return dom;
    }

    function updateDisplay(val) {
      dom.text(val);
    }

    var dom = createToolTip();
    updatePosition(e);
    updateDisplay(value);
    return {
      setDisplay: function setDisplay(val) {
        return updateDisplay(val);
      },
      cancel: cancel
    };
  }

  return {
    create: create
  };
});
