function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bajaui/rc/util/ScaledLayout
 */
define(['jquery', 'underscore'], function ($, _) {
  'use strict';

  var isNumber = _.isNumber;
  /**
   * @typedef {object} module:nmodule/bajaui/rc/util/ScaledLayout~ScaledLayoutConfig
   * @property {number} [outerWidth] width of parent element. If omitted, will
   * be calculated according to scaling mode and outerHeight.
   * @property {number} [outerHeight] height of parent element. If omitted, will
   * be calculated according to scaling mode and outerWidth.
   * @property {number} innerWidth width of inner element
   * @property {number} innerHeight height of inner element
   * @property {string} scale scaling mode (a BScaleMode tag)
   * @property {number} [minScaleFactor=0] scaling minimum
   * @property {number} [maxScaleFactor=0] scaling maximum (0 means
   * unconstrained scaling up)
   * @property {string} halign how to horizontally align the child element (a
   * BHalign tag)
   * @property {string} valign how to vertically align the child element (a
   * BValign tag)
   */

  /**
   * This class is responsible for taking a child element and scaling it up to
   * its parent, taking into account scale mode, alignment, and min/max scale
   * factors. Primary use cases are CanvasPane and Picture.
   *
   * @alias module:nmodule/bajaui/rc/util/ScaledLayout
   */

  return /*#__PURE__*/function () {
    /**
     * @param {module:nmodule/bajaui/rc/util/ScaledLayout~ScaledLayoutConfig} config
     */
    function ScaledLayout(config) {
      _classCallCheck(this, ScaledLayout);

      this.$config = config;
    }
    /**
     * Gets the scale ratio in each direction.
     * @returns {{scaleX: number, scaleY: number}}
     */


    _createClass(ScaledLayout, [{
      key: "getScale",
      value: function getScale() {
        return this.$scale || (this.$scale = toScale(this.$config));
      }
      /**
       * @returns {number} width of the inner element after scaling
       */

    }, {
      key: "getScaledWidth",
      value: function getScaledWidth() {
        return this.getScale().scaleX * this.$config.innerWidth;
      }
      /**
       * @returns {number} height of the inner element after scaling
       */

    }, {
      key: "getScaledHeight",
      value: function getScaledHeight() {
        return this.getScale().scaleY * this.$config.innerHeight;
      }
      /**
       * @returns {{transform: string, transform-origin: string}} CSS to apply
       * the scaling using CSS transforms, to be applied to the child element.
       * The parent element is still responsible for doing the actual alignment
       * of the child element.
       */

    }, {
      key: "getTransformCss",
      value: function getTransformCss() {
        var _this$$config = this.$config,
            halign = _this$$config.halign,
            valign = _this$$config.valign;

        var _this$getScale = this.getScale(),
            scaleX = _this$getScale.scaleX,
            scaleY = _this$getScale.scaleY;

        return {
          transform: "scale(".concat(scaleX, ", ").concat(scaleY, ")"),
          transformOrigin: alignToPercent(halign) + ' ' + alignToPercent(valign)
        };
      }
      /**
       * Scale an inner element to its outer container.
       *
       * @param {HTMLElement|JQuery} inner
       * @param {HTMLElement|JQuery} outer
       * @param {module:nmodule/bajaui/rc/util/ScaledLayout~ScaledLayoutConfig} config
       * @returns {module:nmodule/bajaui/rc/util/ScaledLayout}
       */

    }], [{
      key: "scale",
      value: function scale(inner, outer, config) {
        inner = $(inner);
        outer = $(outer);
        var outerWidth = outer.width();
        var outerHeight = outer.height();
        var container = outer.closest('.ux-scaled-layout-support');

        if (container.length) {
          if (container.hasClass('ux-scaled-layout-scale-to-width')) {
            outerHeight = undefined;
          } else if (container.hasClass('ux-scaled-layout-scale-to-height')) {
            outerWidth = undefined;
          }
        }

        return new ScaledLayout(Object.assign({
          outerWidth: outerWidth,
          outerHeight: outerHeight,
          innerWidth: inner.width(),
          innerHeight: inner.height()
        }, config));
      }
    }]);

    return ScaledLayout;
  }();
  /**
   * @param {module:nmodule/bajaui/rc/util/ScaledLayout~ScaledLayoutConfig} config
   */

  function toScale(config) {
    var innerWidth = config.innerWidth,
        innerHeight = config.innerHeight,
        scale = config.scale,
        _config$minScaleFacto = config.minScaleFactor,
        min = _config$minScaleFacto === void 0 ? 0 : _config$minScaleFacto,
        _config$maxScaleFacto = config.maxScaleFactor,
        max = _config$maxScaleFacto === void 0 ? 0 : _config$maxScaleFacto;

    var _getOuterDimensions = getOuterDimensions(config),
        outerWidth = _getOuterDimensions.outerWidth,
        outerHeight = _getOuterDimensions.outerHeight;

    if (max <= 0) {
      max = Number.POSITIVE_INFINITY;
    }

    var clamp = function clamp(num) {
      return Math.max(min, Math.min(num, max));
    };

    var scaleX, scaleY;
    var xRatio = outerWidth / innerWidth;
    var yRatio = outerHeight / innerHeight;

    switch (scale) {
      case 'none':
        scaleX = 1;
        scaleY = 1;
        break;

      case 'fit':
        scaleX = clamp(xRatio);
        scaleY = clamp(yRatio);
        break;

      case 'fitRatio':
        var ratioScale = clamp(Math.min(xRatio, yRatio));
        scaleX = ratioScale;
        scaleY = ratioScale;
        break;

      case 'fitWidth':
        scaleX = clamp(xRatio);
        scaleY = 1;
        break;

      case 'fitHeight':
        scaleX = 1;
        scaleY = clamp(yRatio);
        break;
    }

    return {
      scaleX: scaleX,
      scaleY: scaleY
    };
  }

  function getOuterDimensions(config) {
    var innerWidth = config.innerWidth,
        innerHeight = config.innerHeight,
        scale = config.scale;
    var outerWidth = config.outerWidth,
        outerHeight = config.outerHeight;
    var hasWidth = isNumber(outerWidth);
    var hasHeight = isNumber(outerHeight);

    if (hasWidth && hasHeight) {
      // we already have them
      return {
        outerWidth: outerWidth,
        outerHeight: outerHeight
      };
    }

    if (!hasWidth && !hasHeight) {
      // perform no scaling as we have nothing to scale to
      return {
        outerWidth: innerWidth,
        outerHeight: innerHeight
      };
    }

    switch (scale) {
      case 'none':
        outerWidth = innerWidth;
        outerHeight = innerHeight;
        break;

      case 'fitRatio':
        if (hasWidth) {
          outerHeight = outerWidth / innerWidth * innerHeight;
        } else {
          outerWidth = outerHeight / innerHeight * innerWidth;
        }

        break;

      case 'fit':
      case 'fitWidth':
      case 'fitHeight':
        if (hasWidth) {
          outerHeight = innerHeight;
        } else {
          outerWidth = innerWidth;
        }

        break;
    }

    return {
      outerWidth: outerWidth,
      outerHeight: outerHeight
    };
  }
  /**
   * @param {string} align a BHalign or BValign tag
   * @returns {string} percentage string for use in `transform-origin`
   */


  function alignToPercent(align) {
    switch (align) {
      case 'left':
      case 'top':
        return '0';

      case 'center':
      default:
        return '50%';

      case 'right':
      case 'bottom':
      case 'fill':
        return '100%';
    }
  }
});
