/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bajaui/rc/mixin/LayoutHintsSupport
 */
define(['jquery', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function ($, mixinUtils) {
  'use strict';

  var EVENT_NAME = 'bajaui:LayoutHints';
  var MIXIN_NAME = 'LAYOUT_HINTS';
  var applyMixin = mixinUtils.applyMixin;
  /**
   * Certain widgets will not work unless their ancestor widgets make
   * accommodations for them. For instance, FlowPane will not work unless its
   * containing ScrollPane prevents horizontal scrolling - otherwise due to its
   * flexbox nature, the FlowPane will just expand horizontally instead of
   * wrapping at the correct breakpoints.
   *
   * This mixin allows a widget (in this example, FlowPane) to trigger a hint to
   * its ancestor widgets as to what kind of accommodations it needs. It also
   * allows those ancestors to respond to hints from its descendent widgets.
   *
   * @mixin
   * @alias module:nmodule/bajaui/rc/mixin/LayoutHintsSupport
   * @example
   * <caption>I, a FlowPane, need my ancestors to constrain horizontal scrolling
   * for my layout to work right.</caption>
   * const { SCROLL, RESTRICT_HORIZONTAL } = LayoutHintsSupport;
   * class FlowPane {
   *   constructor() {
   *     super(...arguments);
   *     LayoutHintsSupport(this, { [SCROLL]: RESTRICT_HORIZONTAL });
   *   }
   * }
   *
   * @example
   * <caption>I, a ScrollPane, may restrict my own horizontal scrolling if one
   * of my descendents requires it.</caption>
   * const { SCROLL, RESTRICT_HORIZONTAL } = LayoutHintsSupport;
   * class ScrollPane {
   *   constructor() {
   *     super(...arguments);
   *     LayoutHintsSupport(this);
   *   }
   *   applyLayoutHints(hints = {}) {
   *     if (hints[SCROLL] === RESTRICT_HORIZONTAL) {
   *       this.jq().addClass('horizontally-restricted');
   *     }
   *   }
   * }
   *
   * @example
   * <caption>I, a CanvasPane, define my very own layout context. Layout hints
   * should not bubble above me.</caption>
   * class CanvasPane {
   *   constructor() {
   *     super(...arguments);
   *     LayoutHintsSupport(this);
   *   }
   *   applyLayoutHints() {
   *     return false;
   *   }
   * }
   */

  var LayoutHintsSupport = function LayoutHintsSupport() {};
  /**
   * Applies layout hints triggered from a child element.
   *
   * @param {object} hints
   * @see module:nmodule/bajaui/rc/mixin/LayoutHints
   * @return {boolean} false if this widget should "eat" the event so no more
   * layout hints get propagated
   */


  LayoutHintsSupport.prototype.applyLayoutHints = function (hints) {};

  function applyLayoutHintsSupport(target, hints) {
    if (!applyMixin(target, MIXIN_NAME, LayoutHintsSupport.prototype)) {
      return;
    }

    var layout = target.layout;
    var initialize = target.initialize;

    if (hints) {
      target.layout = function () {
        target.jq().trigger($.Event(EVENT_NAME, {
          hints: hints
        }));
        return layout.apply(this, arguments);
      };
    }

    target.initialize = function (dom) {
      var _this = this;

      dom.on(EVENT_NAME, function (_ref) {
        var hints = _ref.hints;
        return _this.applyLayoutHints(hints);
      });
      return initialize.apply(this, arguments);
    };
  }

  applyLayoutHintsSupport.EVENT_NAME = EVENT_NAME;
  applyLayoutHintsSupport.SCROLL = 'LayoutHints:scroll';
  applyLayoutHintsSupport.RESTRICT_HORIZONTAL = 'LayoutHints:restrictHorizontal';
  applyLayoutHintsSupport.ALLOW_HORIZONTAL = 'LayoutHints:allowHorizontal';
  return applyLayoutHintsSupport;
});
