/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/FacetsFilterConfigEditor
 */
define([
  'baja!',
  'baja!bql:FacetsFilter,bql:IBqlFilter,bql:BooleanFilter,bql:IntegerFilter,bql:StringFilter,bql:EnumFilter,bql:AbsTimeFilter,bql:FloatFilter,bql:DoubleFilter,bql:BitStringFilter,bql:NullFilter',
  'jquery',
  'underscore',
  'Promise',
  'bajaux/events',
  'bajaux/commands/Command',
  'bajaux/util/CommandButton',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
  'nmodule/bql/rc/fe/filter/saveUtil',
  'nmodule/webEditors/rc/fe/baja/OrderedMapEditor',
  'css!nmodule/bql/rc/fe/filter/templates/editors'
], function (baja, types, $, _, Promise, events,
             Command, CommandButton, BaseEditor, fe,
             DepthSubscriber, saveUtil,
             OrderedMapEditor) {

  "use strict";

  var LEXICON_PREFIX                   = "facetsFilterConfig.filterType.",

      FACET_SELECTION_CLASS            = "facetSelector",
      FACET_KEY_CLASS                  = "facetKey",
      FACET_TYPE_CLASS                 = 'facetType',
      ADD_FILTER_ENTRY_CMD_CLASS       = "addEntry",
      FILTER_ENTRY_TABLE_CLASS         = "filterEntryTable",
      FILTER_ENTRY_CLASS               = "filterEntry",
      FILTER_ENTRY_KEY_CLASS           = 'filterEntryKey',
      FILTER_ENTRY_VALUE_CLASS         = "filterEntryValue",
      REMOVE_FILTER_ENTRY_CMD_CLASS    = "removeEntry",

      FACET_SELECTION_SELECTOR         = "." + FACET_SELECTION_CLASS,
      FACET_KEY_SELECTOR               = "." + FACET_KEY_CLASS,
      FACET_TYPE_SELECTOR              = "." + FACET_TYPE_CLASS,
      ADD_FILTER_ENTRY_CMD_SELECTOR    = "." + ADD_FILTER_ENTRY_CMD_CLASS,
      FILTER_ENTRY_TABLE_SELECTOR      = "." + FILTER_ENTRY_TABLE_CLASS,
      FILTER_ENTRY_SELECTOR            = "." + FILTER_ENTRY_CLASS,
      FILTER_ENTRY_KEY_SELECTOR        = "." + FILTER_ENTRY_KEY_CLASS,
      FILTER_ENTRY_VALUE_SELECTOR      = "." + FILTER_ENTRY_VALUE_CLASS,
      REMOVE_FILTER_ENTRY_CMD_SELECTOR = "." + REMOVE_FILTER_ENTRY_CMD_CLASS,

      FACET_KEYS_PROPERTIES_FILE       = "file:!defaults/workbench/facetKeys.properties",
      DEFAULT_FACET_KEY_PROPERTIES     = ('falseText=s,trueText=s,min=f,max=f,precision=i,radix=i,range=E,' +
      'showDate=b,showTime=b,showSeconds=b,showMilliseconds=b,showTimeZone=b,multiLine=b,fieldWidth=i,' +
      'allowNull=b,ordRelativize=b,fieldEditor=s,cellEditor=s,units=u,targetType=s,realms=s,' +
      'showUnits=b,showSeparators=b,forceSign=b,maxOverrideDuration=r').split(','),

      ENABLE_EVENT                     = events.ENABLE_EVENT,
      DISABLE_EVENT                    = events.DISABLE_EVENT,
      MODIFY_EVENT                     = events.MODIFY_EVENT,
      INITIALIZE_EVENT                 = events.INITIALIZE_EVENT,
      LOAD_EVENT                       = events.LOAD_EVENT,
      READONLY_EVENT                   = events.READONLY_EVENT,
      WRITABLE_EVENT                   = events.WRITABLE_EVENT,
      DESTROY_EVENT                    = events.DESTROY_EVENT,

      filterTypes                      = [
        Boolean.DEFAULT,
        baja.Integer.DEFAULT,
        baja.Long.DEFAULT,
        baja.Float.DEFAULT,
        baja.Double.DEFAULT,
        String.DEFAULT,
        baja.DynamicEnum.DEFAULT,
        baja.AbsTime.DEFAULT
      ];

  /**
   * FacetsFilterConfigEditor is a field editor adding/removing bql:IBqlFilter filters'
   * to a bql:FacetsFilter
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/FacetsFilterEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var FacetsFilterConfigEditor = function () {
    BaseEditor.apply(this, arguments);
  };

  FacetsFilterConfigEditor.prototype = Object.create(BaseEditor.prototype);
  FacetsFilterConfigEditor.prototype.constructor = FacetsFilterConfigEditor;

  function initAddFilterEntryCommand(editor) {
    var dom       = editor.jq().find(FACET_SELECTION_SELECTOR),
        buttonDom = $('<button type="button"></button>')
          .addClass("ux-btn-tb").addClass(ADD_FILTER_ENTRY_CMD_CLASS)
          .appendTo(dom);

    return fe.buildFor({
      dom: buttonDom,
      value: new Command({
        icon: "module://icons/x16/add.png",
        func: function () {
          return addFilterEntry(editor);
        }
      }),
      type: CommandButton,
      formFactor: 'mini'
    });
  }

  function updateKeyWidths(editor) {
    baja.runAsync(function () {
      var width = 0;
      editor.$getFilterEntryKeys().css('width', 'auto').each(function () {
        var keyLength = $(this).width();
        width = keyLength > width ? keyLength : width;
      }).css('width', (width + 10) + 'px');
    });
  }

  function addFilterEntry(editor, facetKey, facetValue) {

    var keyEd    = editor.$getFacetKeyEditor(),
        typeEd   = editor.$getFacetTypeEditor(),
        tableDom = editor.jq().find(FILTER_ENTRY_TABLE_SELECTOR),
        promise  = Promise.resolve(facetKey || keyEd.read());

    return promise.then(function (key) {
      key = key.trim();

      if (key && !isDuplicateKey(key, editor)) {

        var entryRowDom = $("<tr/>").addClass(FILTER_ENTRY_CLASS);

        //append facet key
        $("<td/>").append($("<div/>").addClass(FILTER_ENTRY_KEY_CLASS).text(key))
          .appendTo(entryRowDom);

        //append facet value filter editor
        var selectedValue = typeEd.getSelectedValue(),
            valueDom      = $("<div/>").addClass(FILTER_ENTRY_VALUE_CLASS);

        valueDom.appendTo($("<td/>").appendTo(entryRowDom));

        // There is no bql:LongFilter server side so switch back to Integer.
        if (selectedValue === "Long") {
          selectedValue = "Integer";
        }

        selectedValue = "bql:" + (selectedValue === 'DynamicEnum' ? "Enum" : selectedValue) + "Filter";

        return fe.buildFor({
          dom: valueDom,
          value: facetValue || baja.$(selectedValue),
          formFactor: 'mini'
        }).then(function () {

          //append remove command
          var buttonDom = $('<button type="button"></button>').addClass("ux-btn-tb").addClass(REMOVE_FILTER_ENTRY_CMD_CLASS);
          $("<td/>").append(buttonDom).appendTo(entryRowDom);

          return fe.buildFor({
            dom: buttonDom,
            value: new Command({
              icon: "module://icons/x16/delete.png",
              func: function () {
                entryRowDom.remove();
                updateKeyWidths(editor);
              }
            }),
            type: CommandButton,
            formFactor: 'mini'
          });

        }).then(function () {
          tableDom.append(entryRowDom);
          updateKeyWidths(editor);

        }).catch(function (err) {
          baja.outln(err);
        });
      }
    });
  }

  function isDuplicateKey(key, editor) {
    return editor.$getFilterEntryKeys().filter(function () {
      return ($(this).text() === key);
    }).length;
  }

  function initKeyEditor(editor) {
    var dom = editor.jq().find(FACET_SELECTION_SELECTOR);

    return getPredefinedFacetsProperties().then(function (props) {
      var map  = _.map(props, function (prop) {
            return prop.split('=');
          }),
          keys = _.map(map, _.first);

      return fe.buildFor({
        dom: $("<div/>").addClass(FACET_KEY_CLASS).appendTo(dom),
        value: " ",
        properties: { datalist: keys.join(';') },
        formFactor: 'mini'
      }).then(function (ed) {

        function setFacetType() {
          var key     = ed.jq().find('input').val(),
              mapping = _.find(map, function (pair) {
                return pair[0] === key;
              });

          if (mapping) {
            var symbol = mapping[1];
            editor.$getFacetTypeEditor().setSelectedKey(LEXICON_PREFIX + symbol);
          }
        }

        ed.jq().on(MODIFY_EVENT, setFacetType);
      });
    });
  }

  function initTypeEditor(editor) {
    var dom   = editor.jq().find(FACET_SELECTION_SELECTOR),
        map   = new baja.OrderedMap(),
        types = filterTypes;

    _.each(types, function (type) {
      map.put(LEXICON_PREFIX + type.getDataTypeSymbol(), type.getType().getTypeName());
    });

    return fe.buildFor({
      dom: $("<div/>").addClass(FACET_TYPE_CLASS).appendTo(dom),
      value: map,
      type: OrderedMapEditor,
      properties: { lexicon: 'bql', sortKeys: true },
      formFactor: 'mini'
    });
  }

  function getPredefinedFacetsProperties() {
    return baja.Ord.make(FACET_KEYS_PROPERTIES_FILE).get().then(function (file) {
      return Promise.resolve($.get(file.getReadUri())).then(function (data) {
        return data.match(/.*=./g);
      });
    }).catch(function () {
      return DEFAULT_FACET_KEY_PROPERTIES;
    });
  }

  /**
   * Returns the StringEditor for facet keys
   * @private
   * @returns {StringEditor}
   * @see module:nmodule/webEditors/rc/fe/baja/StringEditor
   */
  FacetsFilterConfigEditor.prototype.$getFacetKeyEditor = function () {
    return this.jq().find(FACET_KEY_SELECTOR).data('widget');
  };

  /**
   * Returns the OrderedMapEditor for facet filter types
   * @private
   * @returns {OrderedMapEditor}
   * @see module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor
   */
  FacetsFilterConfigEditor.prototype.$getFacetTypeEditor = function () {
    return this.jq().find(FACET_TYPE_SELECTOR).data('widget');
  };

  /**
   * Returns the CommandButton for adding a new facet filter entry
   * @private
   * @returns {CommandButton}
   * @see bajaux/util/CommandButton
   */
  FacetsFilterConfigEditor.prototype.$getAddFilterEntryCmdBtn = function () {
    return this.jq().find(ADD_FILTER_ENTRY_CMD_SELECTOR).data('widget');
  };

  /**
   * Returns a list of jQuery elements representing filter entries
   * @private
   * @returns {jQuery}
   */
  FacetsFilterConfigEditor.prototype.$getFilterEntries = function () {
    return this.jq().find(FILTER_ENTRY_SELECTOR);
  };

  /**
   * Returns the jQuery filter entry corresponding to the specified key
   * @param {String} key facet key
   * @private
   * @returns {jQuery}
   */
  FacetsFilterConfigEditor.prototype.$getFilterEntry = function (key) {
    return this.$getFilterEntries().filter(function () {
      return $(this).find(FILTER_ENTRY_KEY_SELECTOR).text() === key;
    });
  };

  /**
   * Returns the jQuery filter entry filter editor corresponding to the specified key
   * @param {String} key facet key
   * @private
   * @returns {Widget} Widget editor for the filter entry
   */
  FacetsFilterConfigEditor.prototype.$getFilterEntryValue = function (key) {
    return this.$getFilterEntry(key).find(FILTER_ENTRY_VALUE_SELECTOR).data('widget');
  };

  /**
   * Returns the jQuery list representing the facet filter keys
   * @private
   * @returns {jQuery}
   */
  FacetsFilterConfigEditor.prototype.$getFilterEntryKeys = function () {
    return this.jq().find(FILTER_ENTRY_KEY_SELECTOR);
  };

  /**
   * Returns an array of pre-defined facet keys
   * @private
   * @returns {Promise} Promise resolved with array of pre-defined keys
   */
  FacetsFilterConfigEditor.prototype.$getFacetKeys = function () {
    return getPredefinedFacetsProperties().then(function (properties) {
      return _.map(properties, function (property) {
        return property.split('=')[0];
      });
    });
  };

  /**
   * Returns an array of predefined facet properties of the form key=symbol
   * @private
   * @returns {Promise} Promise resolve with array of pre-defined properties
   */
  FacetsFilterConfigEditor.prototype.$getFacetProperties = function () {
    return getPredefinedFacetsProperties();
  };

  /**
   * Returns the jQuery list representing the facet filter values
   * @private
   * @returns {jQuery}
   */
  FacetsFilterConfigEditor.prototype.$getFilterEntryValues = function () {
    return this.jq().find(FILTER_ENTRY_VALUE_SELECTOR);
  };

  /**
   * Initializes the dom with editors for specifying facet key/value pairs
   * and a command to add the specified facet
   * @param {JQuery} dom
   * @returns {Promise}
   */
  FacetsFilterConfigEditor.prototype.doInitialize = function (dom) {
    var that           = this,

        selectors      = [
          FACET_KEY_SELECTOR, FACET_TYPE_SELECTOR, ADD_FILTER_ENTRY_CMD_SELECTOR,
          FILTER_ENTRY_KEY_SELECTOR, FILTER_ENTRY_VALUE_SELECTOR, REMOVE_FILTER_ENTRY_CMD_SELECTOR
        ].join(","),

        standardEvents = [
          ENABLE_EVENT, DISABLE_EVENT,
          READONLY_EVENT, WRITABLE_EVENT,
          INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT
        ].join(' ');

    dom.addClass("FacetsFilterConfigEditor")
      .append($("<div/>").addClass(FACET_SELECTION_CLASS))
      .append($("<table/>").addClass(FILTER_ENTRY_TABLE_CLASS))
      .on(standardEvents, selectors, false)
      .on(MODIFY_EVENT, selectors, function () {
        that.setModified(true);
        return false;
      });

    return initKeyEditor(this).then(function () {
      return Promise.join(initTypeEditor(that), initAddFilterEntryCommand(that));
    });
  };

  /**
   * Loads the editor with the filter entries from a bql:FacetsFilter instance
   * @param filter instance of bql:FacetsFilter
   * @returns {Promise} Promise resolved when all filter entries have been loaded
   */
  FacetsFilterConfigEditor.prototype.doLoad = function (filter) {

    if (!baja.hasType(filter, 'bql:FacetsFilter')) {
      return Promise.reject(new Error('editor must be loaded with bql:FacetsFilter'));
    }

    var that     = this,
        entries  = filter.getFilterEntries(),
        promises = _.map(entries.getSlots().is('bql:IBqlFilter').toValueMap(), function (value, key) {
          return addFilterEntry(that, key, value);
        });

    return Promise.all(promises);
  };

  /**
   * Returns a new instance of a bql:FacetsFilter with it's filter
   * entries initialized from the editors filter entry list
   *
   * @returns {Promise} Promise resolved with the new filter instance
   */
  FacetsFilterConfigEditor.prototype.doRead = function () {

    var filter         = baja.$('bql:FacetsFilter'),
        filterEntries  = filter.getFilterEntries(),
        promisesObject = _.object(
          //key array
          this.$getFilterEntryKeys().map(function () {
            return $(this).text();
          }).get(),

          //promise array
          this.$getFilterEntryValues().map(function () {
            return $(this).data('widget').read();
          }).get()
        );

    return Promise.props(promisesObject).then(function (result) {
      _.each(result, function (filter, key) {
        filterEntries.add({ slot: key, value: filter });
      });

      return filter;
    });
  };

  /**
   * Sets the readonly state of the FacetsFilterConfigEditor
   * @param {boolean} readonly
   * @returns {Promise}
   */
  FacetsFilterConfigEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Sets the enabled state of the FacetsFilterConfigEditor
   * @param {boolean} enabled
   * @returns {Promise}
   */
  FacetsFilterConfigEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * Destroys all child editors
   * @returns {Promise}
   */
  FacetsFilterConfigEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  return (FacetsFilterConfigEditor);
});
