/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/EnumFilterEditor
 */
define([
  'baja!',
  'baja!bql:EnumFilter,baja:DynamicEnum,baja:EnumRange',
  'lex!bql',
  'jquery',
  'underscore',
  'dialogs',
  'Promise',
  'bajaux/events',
  'bajaux/commands/Command',
  'bajaux/util/CommandButton',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/EnumSetEditor',
  'nmodule/webEditors/rc/fe/fe',
  'bajaux/mixin/batchSaveMixin',
  'bajaux/mixin/subscriberMixIn',
  'nmodule/bql/rc/fe/filter/saveUtil'
], function (baja, types, lexs, $, _, dialogs, Promise, events, Command, CommandButton,
             BaseEditor, EnumSetEditor, fe,
             batchSaveMixin, subscriberMixin, saveUtil) {

  "use strict";

  var ENABLE_EVENT                = events.ENABLE_EVENT,
      DISABLE_EVENT               = events.DISABLE_EVENT,
      MODIFY_EVENT                = events.MODIFY_EVENT,
      INITIALIZE_EVENT            = events.INITIALIZE_EVENT,
      LOAD_EVENT                  = events.LOAD_EVENT,
      READONLY_EVENT              = events.READONLY_EVENT,
      WRITABLE_EVENT              = events.WRITABLE_EVENT,
      DESTROY_EVENT               = events.DESTROY_EVENT,
      ENUM_VALUES_EDITOR_CLASS    = 'enumValues',
      ENUM_VALUES_EDITOR_SELECTOR = '.' + ENUM_VALUES_EDITOR_CLASS,
      EDIT_CMD_CLASS              = "editEnumValues",
      EDIT_CMD_SELECTOR           = "." + EDIT_CMD_CLASS,
      ENUM_SET_EDITOR_CLASS       = 'enumSet',
      save                        = saveUtil.save,
      EnumSet                     = baja.EnumSet,
      lex                         = lexs[0];

  /**
   * Initializes a readonly StringEditor for displaying the selected enums
   * @param dom
   * @returns {Promise}
   */
  function initEnumValuesEditor(editor) {

    var dom           = editor.jq(),
        enumValuesDom = $("<div/>").addClass(ENUM_VALUES_EDITOR_CLASS)
          .css('display', 'inline').appendTo(dom);

    return fe.buildFor({
      dom: enumValuesDom,
      value: "",
      readonly: true,
      enabled: false,
      formFactor: 'mini'
    });
  }

  /**
   * Initializes a CommandButton for editing a list of enums via an EnumSetEditor
   * @param dom
   * @returns {Promise}
   */
  function initEditEnumsCmd(editor) {

    var dom       = editor.jq(),
        buttonDom = $('<button type="button"></button>')
          .addClass("ux-btn-tb")
          .addClass(EDIT_CMD_CLASS)
          .appendTo(dom);

    return fe.buildFor({
      dom: buttonDom,
      value: new Command({
        func: function () {
          return dialogs.showOkCancel({
            title: lex.get('editEnums.title'),
            content: function (dlg, dlgDom) {
              return initEnumSetEditor(dlg, editor, dlgDom);
            }
          });
        },
        icon: "module://icons/x16/edit.png"
      }),
      type: CommandButton,
      formFactor: 'mini'
    });
  }

  /**
   * Initializes an EnumSetEditor for selecting a list of enums to filter on
   * @param dlg
   * @param dom
   * @returns {Promise}
   */
  function initEnumSetEditor(dlg, filterEditor, dom) {

    var filter           = filterEditor.value(),
        enumRange        = filter.getEnumType().getRange(),
        enumValuesEd     = filterEditor.$getEnumValuesEditor(),
        enumSetEditorDom = $("<div/>").addClass(ENUM_SET_EDITOR_CLASS).appendTo(dom);

    function decodeToOrdinals(encoding) {
      return encoding.length ? _.map(encoding.split(" | "), function (tag) {
        return enumRange.tagToOrdinal(tag);
      }) : [];
    }

    //arm the OK handler to save the EnumSet to the filter editors
    //enum values field
    dlg.ok(function () {

      var enumSetEditor = enumSetEditorDom.data('widget');
      if (enumSetEditor) {
        return enumSetEditor.read().then(function (enumSet) {
          var ordinals = enumSet.getOrdinals(),
              encoding = _.map(ordinals, function (ordinal) {
                return enumRange.get(ordinal).getTag();
              }).join(" | ");

          return enumValuesEd.load(encoding).then(function () {
            $("input[type='text']", enumValuesEd.jq()).attr('size', encoding.length);
            return enumValuesEd.setModified(true);
          });
        });
      }
    });

    return enumValuesEd.read().then(function (encoding) {
      var enumSet = EnumSet.make({
        ordinals: decodeToOrdinals(encoding),
        range: enumRange
      });

      return fe.buildFor({
        dom: enumSetEditorDom,
        value: enumSet,
        type: EnumSetEditor
      });
    });
  }

  /**
   * EnumFilterEditor is a field editor for a bql:EnumFilter instance
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/EnumFilterEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var EnumFilterEditor = function () {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
    subscriberMixin(this);
  };

  EnumFilterEditor.prototype = Object.create(BaseEditor.prototype);
  EnumFilterEditor.prototype.constructor = EnumFilterEditor;

  /**
   * Returns the underlying StringEditor field listing the selected enums
   * @private
   * @returns {module:nmodule/bql/rc/fe/filter/EnumFilterEditor}
   */
  EnumFilterEditor.prototype.$getEnumValuesEditor = function () {
    return this.jq().find(ENUM_VALUES_EDITOR_SELECTOR).data('widget');
  };

  /**
   * Returns the underlying CommandButton for editing the list of selected enums
   * @private
   */
  EnumFilterEditor.prototype.$getEditEnumsCmdBtn = function () {
    return this.jq().find(EDIT_CMD_SELECTOR).data('widget');
  };

  /**
   * Initializes a StringEditor field for displaying the filter's enum values and a CommandButton
   * for editing the enum value list
   * @param dom
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doInitialize = function (dom) {

    var that = this;

    //prevent subeditors from bubbling up events
    dom.on([ENABLE_EVENT, DISABLE_EVENT,
        READONLY_EVENT, WRITABLE_EVENT,
        INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '),
      ENUM_VALUES_EDITOR_SELECTOR + "," + EDIT_CMD_SELECTOR, false);

    dom.on(MODIFY_EVENT, ENUM_VALUES_EDITOR_SELECTOR + "," + EDIT_CMD_SELECTOR, function () {
      that.setModified(true);
      return false;
    });

    return Promise.join(initEnumValuesEditor(this), initEditEnumsCmd(this));
  };

  /**
   * Loads the filter's vector of baja:Enum values into the StringEditor field
   * @param filter
   * @param params
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doLoad = function (filter) {

    if (!baja.hasType(filter, 'bql:EnumFilter')) {
      return Promise.reject('editor must be loaded with bql:EnumFilter');
    }

    var that         = this,
        enumValues   = filter.getEnumValues(),
        enumValuesEd = that.$getEnumValuesEditor();

    return Promise.resolve(filter.isMounted() && enumValues.lease()).then(function () {
      var enums    = enumValues.getSlots().is("baja:Enum").toValueArray(),
          encoding = _.map(enums, function (_enum) {
            return _enum.getTag();
          }).join(" | ");

      return enumValuesEd.load(encoding).then(function () {
        $("input[type='text']", enumValuesEd.jq()).attr('size', encoding.length);
      });
    });
  };

  /**
   * Reads the current encoding of selected enums into a new bql:EnumFilter instance
   * @returns {Promise} Promise resolved with a new instance of a EnumFilter
   */
  EnumFilterEditor.prototype.doRead = function () {
    var filter    = baja.$('bql:EnumFilter'),
        enumType  = this.value().getEnumType(),
        enumRange = enumType.getRange();

    filter.setEnumType(enumType);

    if (enumRange === baja.EnumRange.DEFAULT) {
      return Promise.resolve(filter);
    }

    return this.$getEnumValuesEditor().read().then(function (encoding) {

      if (encoding.length) {
        var enumValues = baja.$('baja:Vector');

        _.each(encoding.split(" | "), function (tag) {
          var ordinal = enumRange.tagToOrdinal(tag),
              _enum   = enumRange.get(ordinal);

          enumValues.add({slot: "enum?", value: _enum});
        });

        filter.setEnumValues(enumValues);
      }

      return filter;
    });
  };

  /**
   * Sets the readonly state of the EnumEditor
   * @param readonly
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doReadonly = function doReadonly(readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Sets the enabled state of the EnumEditor
   * @param enabled
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doEnabled = function doEnabled(enabled) {
    return Promise.join(
      this.$getEnumValuesEditor().setEnabled(false),
      this.$getEditEnumsCmdBtn().setEnabled(enabled)
    );
  };

  /**
   * Saves the current encoding of selected enums to the loaded filter
   * @param {bql:EnumFilter} filter bql:EnumFilter instance returned by EnumFilterEditor#read()
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doSave = function (filter, params) {
    return save(this, filter, params);
  };

  /**
   * Delegates destruction to the child editors
   * @param params
   * @returns {Promise}
   */
  EnumFilterEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  return EnumFilterEditor;
});
