/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/AbsTimeFilterEditor
 */
define([
  'baja!',
  'baja!bql:AbsTimeFilter,bql:DynamicTimeRange,bql:DynamicTimeRangeType',
  'lex!workbench',
  'jquery',
  'underscore',
  'Promise',
  'bajaux/events',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/bql/rc/fe/DynamicTimeRangeEditor',
  'nmodule/webEditors/rc/fe/fe',
  'bajaux/mixin/batchSaveMixin',
  'nmodule/bql/rc/fe/filter/saveUtil'
], function (baja, types, lexs, $, _, Promise, events, BaseEditor, DynamicTimeRangeEditor, fe, batchSaveMixin, saveUtil) {

  "use strict";

  var TIME_RANGE_EDITOR_CLASS    = 'timeRange',
      TIME_RANGE_EDITOR_SELECTOR = "." + TIME_RANGE_EDITOR_CLASS,
      ENABLE_EVENT               = events.ENABLE_EVENT,
      DISABLE_EVENT              = events.DISABLE_EVENT,
      MODIFY_EVENT               = events.MODIFY_EVENT,
      INITIALIZE_EVENT           = events.INITIALIZE_EVENT,
      LOAD_EVENT                 = events.LOAD_EVENT,
      READONLY_EVENT             = events.READONLY_EVENT,
      WRITABLE_EVENT             = events.WRITABLE_EVENT,
      DESTROY_EVENT              = events.DESTROY_EVENT,
      save                       = saveUtil.save;

  /**
   * Initializes the DOM with a DynamicTimeRangeEditor
   * @param dom jQuery DOM instance
   * @returns {Promise}
   * @see module:nmodule/bql/rc/fe/DynamicTimeRangeEditor
   */
  function initTimeRangeEditor(dom) {
    var editorDom = $("<div/>").addClass(TIME_RANGE_EDITOR_CLASS).appendTo(dom);
    return fe.buildFor({
      dom: editorDom,
      value: baja.$('bql:DynamicTimeRange', 'today'),
      type: DynamicTimeRangeEditor
    });
  }

  /**
   * AbsTimeFilterEditor is a field editor for a bql:AbsTimeFilter instance
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/AbsTimeFilterEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var AbsTimeFilterEditor = function () {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
  };

  AbsTimeFilterEditor.prototype = Object.create(BaseEditor.prototype);
  AbsTimeFilterEditor.prototype.constructor = AbsTimeFilterEditor;

  /**
   * Returns a reference to the internal DynamicTimeRangeEditor
   * @returns {DynamicTimeRangeEditor}
   * @private
   * @see module:nmodule/bql/rc/fe/DynamicTimeRangeEditor
   */
  AbsTimeFilterEditor.prototype.$getTimeRangeEditor = function () {
    return this.jq().find(TIME_RANGE_EDITOR_SELECTOR).data('widget');
  };

  /**
   * Delegates editor initialization to the DynamicTimeRangeEditor
   * @param dom jQuery DOM instance
   * @returns {Promise}
   * @see module:nmodule/bql/rc/fe/DynamicTimeRangeEditor
   */
  AbsTimeFilterEditor.prototype.doInitialize = function (dom) {

    var that = this;

    //prevent DynamicTimeRangeEditor from bubbling up events
    dom.on([ENABLE_EVENT, DISABLE_EVENT,
      READONLY_EVENT, WRITABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '), TIME_RANGE_EDITOR_SELECTOR, false);

    dom.on(MODIFY_EVENT, TIME_RANGE_EDITOR_SELECTOR, function () {
      that.setModified(true);
      return false;
    });

    return initTimeRangeEditor(dom);
  };

  /**
   * Loads the editor with the filter's 'startTime', 'endTime' and 'timeRangeType' properties by delegating
   * to the internal DynamicTimeRangeEditor load() method
   * @param {baja.AbsTime} filter An instance of baja:AbsTimeFilter
   * @returns {Promise} Promise that rejects if the argument is not a bql:AbsTimeFilter or resolves
   * when the DynamicTimeRangeEditor has loaded
   * @see module:nmodule/bql/rc/fe/DynamicTimeRangeEditor
   */
  AbsTimeFilterEditor.prototype.doLoad = function (filter) {

    function encodeTimeRangeType(timeRangeType, startTime, endTime) {
      return timeRangeType + (timeRangeType === 'timeRange' ?
        ":startTime=" + startTime.encodeToString() +
        ";endTime=" + endTime.encodeToString() : "");
    }

    if (!baja.hasType(filter, 'bql:AbsTimeFilter')) {
      return Promise.reject('editor must be loaded with bql:AbsTimeFilter');
    }

    var that = this;
    return Promise.resolve(filter.isMounted() && filter.lease()).then(function () {
      var startTime     = filter.getStartTime(),
          endTime       = filter.getEndTime(),
          timeRangeType = filter.getTimeRangeType().getTag(),
          encoding      = encodeTimeRangeType(timeRangeType, startTime, endTime);

      return that.$getTimeRangeEditor().load(baja.$('bql:DynamicTimeRange', encoding));
    });
  };

  /**
   * Reads the fields from the internal DynamicTimeRangeEditor into a new baja:AbsTimeFilter instance
   * and returns a Promise resolved with the filter instance
   * @returns {Promise} Promise resolved with a new baja:AbsTime filter instance configured with the
   * result from DynamicTimeRangeEditor#read()
   */
  AbsTimeFilterEditor.prototype.doRead = function () {

    var timeRangeEd   = this.$getTimeRangeEditor(),
        working       = timeRangeEd.$working,
        startTime     = working.get('startTime'),
        endTime       = working.get('endTime'),
        minEnabled    = working.get('startFixed'),
        maxEnabled    = working.get('endFixed'),
        timeRangeType = working.get('period');

    var filter = baja.$('bql:AbsTimeFilter', {
      startTime: startTime,
      minEnabled: minEnabled,
      endTime: endTime,
      maxEnabled: maxEnabled,
      timeRangeType: timeRangeType
    });

    return filter;


    /*return timeRangeEd.read().then(function (timeRange) {

     var encodedTimeRange = timeRange.encodeToString(),
     startTime, endTime, minEnabled, maxEnabled;

     //decode the start and end times for the 'timeRange' selection
     if (encodedTimeRange.indexOf('timeRange') === 0) {

     var decodedTimes = decodeTimes(encodedTimeRange);
     startTime = decodedTimes[0];
     endTime = decodedTimes[1];
     encodedTimeRange = 'timeRange';
     minEnabled = timeRangeEd.$working.get('startFixed');
     maxEnabled = timeRangeEd.$working.get('endFixed');
     }

     var filter = baja.$('bql:AbsTimeFilter', {
     startTime: startTime || AbsTime.DEFAULT,
     minEnabled: minEnabled,
     endTime: endTime || AbsTime.DEFAULT,
     maxEnabled: maxEnabled,
     timeRangeType: baja.$('bql:DynamicTimeRangeType').make(encodedTimeRange)
     });

     return filter;
     });*/
  };

  /**
   * Saves the filter returned by read() to the loaded filter instance represented by this.value()
   * @param {baja.Component} filter Instance of `baja:AbsTimeFilter` returned by read()
   * @param params
   * @returns {Promise} Promise resolved once the filter has been saved
   */
  AbsTimeFilterEditor.prototype.doSave = function (filter, params) {
    return save(this, filter, params);
  };

  /**
   * Sets the readonly state of all child editors
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  AbsTimeFilterEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Sets the enabled state of all child editors
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  AbsTimeFilterEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Destroys all child editors
   * @param params
   * @returns {Promise}
   */
  AbsTimeFilterEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  return AbsTimeFilterEditor;
});
