/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/DynamicTimeRangeEditor
 */
define(['baja!',
  'jquery',
  'Promise',
  'bajaux/events',
  'bajaux/commands/Command',
  'bajaux/util/CommandButton',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/FrozenEnumEditor',
  'nmodule/webEditors/rc/fe/feDialogs',
  'bajaScript/baja/obj/dateTimeUtil',
  'nmodule/bql/rc/fe/DynamicTimeRangeDialogEditor',
  'lex!webEditors',
  'baja!bql:DynamicTimeRange,' +
  'bql:DynamicTimeRangeType'], function (baja,
                                         $,
                                         Promise,
                                         events,
                                         Command,
                                         CommandButton,
                                         fe,
                                         BaseEditor,
                                         FrozenEnumEditor,
                                         feDialogs,
                                         dateTimeUtil,
                                         DynamicTimeRangeDialogEditor,
                                         lexs) {

  'use strict';

  var ENABLE_EVENT     = events.ENABLE_EVENT,
      DISABLE_EVENT    = events.DISABLE_EVENT,
      MODIFY_EVENT     = events.MODIFY_EVENT,
      INITIALIZE_EVENT = events.INITIALIZE_EVENT,
      LOAD_EVENT       = events.LOAD_EVENT,
      READONLY_EVENT   = events.READONLY_EVENT,
      WRITABLE_EVENT   = events.WRITABLE_EVENT,
      DESTROY_EVENT    = events.DESTROY_EVENT,
      webEditorsLex    = lexs[0],
      TIMEZONE_PROPERTY = "TimeZone",
      EPOCH            = webEditorsLex.get("DynamicTimeRangeEditor.epoch");

  /**
   * Add the necessary facets to the custom Component used for editing in a
   * dialog (note that this will run synchronously since it is not a mounted
   * Component).
   *
   * @inner
   * @param {baja.Component} comp
   */
  function addComplexFacets(comp) {
    var startLex = webEditorsLex.get("DynamicTimeRangeEditor.startTime"),
        endLex   = webEditorsLex.get("DynamicTimeRangeEditor.endTime");

    comp.setFacets({
      slot: "startFixed",
      facets: baja.Facets.make(["trueText", "falseText"], [startLex, startLex])
    });

    comp.setFacets({
      slot: "endFixed",
      facets: baja.Facets.make(["trueText", "falseText"], [endLex, endLex])
    });
  }

  /**
   * A field editor for working with bql:DynamicTimeRange.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/bql/DynamicTimeRangeEditor
   * @param params
   */
  var DynamicTimeRangeEditor = function DynamicTimeRangeEditor(params) {
    var that = this;
    BaseEditor.apply(that, arguments);
    that.$periodEditor = new FrozenEnumEditor(params);
    that.$encoding = null;
    that.$working = null;
  };

  DynamicTimeRangeEditor.prototype = Object.create(BaseEditor.prototype);
  DynamicTimeRangeEditor.prototype.constructor = DynamicTimeRangeEditor;

  /**
   * Get the CommandButton
   * @private
   * @returns {module:bajaux/util/CommandButton}
   */
  DynamicTimeRangeEditor.prototype.$getCommandButton = function () {
    var that = this,
        jq   = that.jq();
    return jq.children(".details").children("button").data("widget");
  };

  /**
   * Initializes `periodEditor` and displays the details of the period if the
   * period is set to timeRange.
   *
   * @param {JQuery} dom
   * @returns {Promise} promise to be resolved when Widget is initialized.
   */
  DynamicTimeRangeEditor.prototype.doInitialize = function (dom) {
    var that                        = this,
        buildCommandButton,
        periodDiv                   = $('<span class="period"/>').appendTo(dom),
        detailsDiv                  = $('<span class="details"/>').appendTo(dom),
        button                      = $("<button type='button' class='ux-btn-tb'/>").appendTo(detailsDiv),
        editDynamicTimeRangeCommand = new Command({
          module: 'webEditors',
          lex: 'commands.editDynamicTimeRange',
          func: function () {
            return that.$launchDialog();
          }
        });

    $("<span class='display'/>").appendTo(detailsDiv);

    detailsDiv.hide();

    /*
     prevent some events from our subeditors from bubbling up
     */
    dom.on([ENABLE_EVENT, DISABLE_EVENT,
      READONLY_EVENT, WRITABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '), '.editor', false);

    dom.on([ENABLE_EVENT, DISABLE_EVENT,
      READONLY_EVENT, WRITABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '), 'button', false);

    dom.on(MODIFY_EVENT, '.period', function () {
      that.$periodEditor.read()
        .then(function (period) {
          var working = that.$working;

          working.set({
            slot: "period",
            value: period
          });
          that.$encoding = that.$complexToEncoding(working);
          dom.children(".details").toggle(period.getTag() === "timeRange");
          that.setModified(true);
        })
        .catch(baja.error);

      return false;
    });

    buildCommandButton = fe.buildFor({
      dom: button,
      type: CommandButton,
      value: editDynamicTimeRangeCommand
    });

    return Promise.join(that.$periodEditor.initialize(periodDiv), buildCommandButton);
  };

  /**
   * Launch a dialog for editing the timeRange.
   * @private
   * @return {Promise}
   */
  DynamicTimeRangeEditor.prototype.$launchDialog = function () {
    var that    = this,
        working = that.$working;

    return feDialogs.showFor({
      title: webEditorsLex.get('commands.editDynamicTimeRange.description'),
      value: working,
      type: DynamicTimeRangeDialogEditor,
      properties: {
        TimeZone: that.properties().getValue(TIMEZONE_PROPERTY)
      }
    })
      .then(function (result) {

        if (!result) {
          return;
        }

        //edit by reference, so now ignore 'result'
        that.$encoding = that.$complexToEncoding(working);
        return that.$updateDisplayText(working)
          .then(function () {
            //don't re-send modified until encoding and display changes are complete
            that.setModified(true);
          });

      });
  };

  /**
   * Convert the DynamicTimeRange to a Complex that can be easily edited via a CompositeEditor.
   * @private
   * @param {baja.Simple} simpleValue a `bql:DynamicTimeRange` instance
   * @returns {baja.Complex}
   */
  DynamicTimeRangeEditor.prototype.$simpleToComplex = function (simpleValue) {
    var that = this,
        comp,
        period     = baja.$("bql:DynamicTimeRangeType", "today"),
        startFixed = false,
        endFixed   = false,
        encodedStartTime,
        encodedEndTime,
        timezoneString =  that.properties().getValue(TIMEZONE_PROPERTY),
        timezone   = timezoneString && baja.TimeZone.DEFAULT.decodeFromString(timezoneString),
        offsetIfNoTimeZone = new Date().getTimezoneOffset() * -60000,
        now = baja.AbsTime.now(),
        startTime  = baja.AbsTime.make({
          date: now.getDate(),
          timeZone: timezone,
          offset: offsetIfNoTimeZone
        }),
        endTime    = baja.AbsTime.make({
          date: now.getDate(),
          time: now.getTime(),
          timeZone: timezone,
          offset: offsetIfNoTimeZone
        }),
        encoding   = simpleValue.encodeToString(),
        colonSplit,
        semiSplit;


    if (encoding.indexOf(":") === -1) {
      period = baja.$("bql:DynamicTimeRangeType", encoding);
    } else {
      colonSplit = encoding.split(":");
      period = baja.$("bql:DynamicTimeRangeType", colonSplit[0]);
      encoding = encoding.substring("timeRange:".length);
      semiSplit = encoding.split(";");
      encodedStartTime = baja.AbsTime.DEFAULT.decodeFromString(semiSplit[0].split("=")[1]);
      encodedEndTime = baja.AbsTime.DEFAULT.decodeFromString(semiSplit[1].split("=")[1]);

      if (!encodedStartTime.equals(baja.AbsTime.DEFAULT)) {
        startTime = encodedStartTime;
        startFixed = true;
      }

      if (!encodedEndTime.equals(baja.AbsTime.DEFAULT)) {
        endTime = encodedEndTime;
        endFixed = true;
      }
    }

    comp = new baja.$('baja:Component', {
      period: period,
      startFixed: startFixed,
      startTime: startTime,
      endFixed: endFixed,
      endTime: endTime
    });

    addComplexFacets(comp);
    return comp;

  };

  /**
   * Get the startTime and EndTime Display
   * @private
   * @param {baja.Complex} complex
   * @returns {Promise}
   */
  DynamicTimeRangeEditor.prototype.$updateDisplayText = function (complex) {
    var that       = this,
        start      = complex.get("startTime"),
        end        = complex.get("endTime"),
        startFixed = complex.get("startFixed"),
        endFixed   = complex.get("endFixed"),
        jq         = that.jq(),
        details    = jq.children(".details"),
        showParam  = baja.TimeFormat.SHOW_DATE | baja.TimeFormat.SHOW_TIME | baja.TimeFormat.SHOW_ZONE,
        timezoneString =  that.properties().getValue(TIMEZONE_PROPERTY),
        timezone   = timezoneString && baja.TimeZone.DEFAULT.decodeFromString(timezoneString);

    return baja.TimeZoneDatabase.get()
      .then(function (database) {
        var fullzone = timezone && database.getTimeZone(timezone.getId());
        return Promise.join(
          startFixed ? start.toDateTimeString({ show: showParam, TimeZone: fullzone }) : EPOCH,
          endFixed ? end.toDateTimeString({ show: showParam, TimeZone: fullzone }) : EPOCH
        );
      }).spread(function (startDisplay, endDisplay) {
        var displayText = webEditorsLex.get("DynamicTimeRangeEditor.details", startDisplay, endDisplay);
        details.children("span.display").text(displayText);
      });
  };

  /**
   * Get the encoding of the custom Component
   * @private
   * @param {baja.Complex} complex
   * @returns {String}
   */
  DynamicTimeRangeEditor.prototype.$complexToEncoding = function (complex) {
    var period    = complex.get("period"),
        periodTag = period ? period.getTag() : "timeRange",
        startTime,
        endTime,
        startFixed,
        endFixed;

    if (periodTag !== "timeRange") {
      return periodTag;
    }

    startTime = complex.get("startTime");
    endTime = complex.get("endTime");
    startFixed = complex.get("startFixed");
    endFixed = complex.get("endFixed");

    if (!startFixed) {
      startTime = baja.AbsTime.DEFAULT;
    }

    if (!endFixed) {
      endTime = baja.AbsTime.DEFAULT;
    }
    return periodTag + ":startTime=" + startTime.encodeToString() + ";endTime=" + endTime.encodeToString();
  };

  /**
   * Loads the period, startTime and endTime of the `DynamicTimeRange` into the
   * editors.
   *
   * @param {baja.Simple} value a `bql:DynamicTimeRange`
   * @returns {Promise} promise to be resolved once the
   * editors have finished loading
   */
  DynamicTimeRangeEditor.prototype.doLoad = function (value) {
    var that    = this,
        working = that.$simpleToComplex(value),
        period  = working.get("period"),
        jq      = that.jq(),
        details = jq.children(".details");

    that.$working = working;

    details.toggle(period.getTag() === "timeRange");
    that.$encoding = value.encodeToString();
    return Promise.join(
      that.$periodEditor.load(period),
      that.$updateDisplayText(working)
    );
  };

  /**
   * Reads the values from the last editor that has been modified and constructs a DynamicTimeRange
   *
   * @returns {baja.Value} a `bql:DynamicTimeRange` instance
   */
  DynamicTimeRangeEditor.prototype.doRead = function () {
    return baja.$("bql:DynamicTimeRange", this.$encoding);
  };

  /**
   * Enables/disables both the period and button
   *
   * @param {Boolean} enabled
   */
  DynamicTimeRangeEditor.prototype.doEnabled = function (enabled) {
    var that = this;
    return Promise.join(
      that.$periodEditor.setEnabled(enabled),
      that.$getCommandButton().setEnabled(!that.isReadonly() && enabled)
    );
  };

  /**
   * Set Readonly or not for period and button
   *
   * @param {Boolean} readonly
   */
  DynamicTimeRangeEditor.prototype.doReadonly = function (readonly) {
    var that = this;
    return Promise.join(
      that.$periodEditor.setReadonly(readonly),
      that.$getCommandButton().setEnabled(that.isEnabled() && !readonly)
    );
  };

  /**
   * Destroy the period and CommandButton

   * @return {Promise}
   */
  DynamicTimeRangeEditor.prototype.doDestroy = function () {
    var that = this;
    that.$encoding = null;
    that.$working = null;
    return Promise.join(
      that.$periodEditor.destroy(),
      that.$getCommandButton().destroy()
    );
  };

  return (DynamicTimeRangeEditor);
});
