/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/DynamicTimeRangeDialogEditor
 */
define(['jquery',
  'Promise',
  'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor',
  'bajaux/events',
  'bajaux/Properties',
  'nmodule/webEditors/rc/fe/config/ComplexCompositeBuilder',
  'hbs!nmodule/bql/rc/fe/dynamicTimeRangeDialogEditorStructure',
  'lex!webEditors',
  'nmodule/webEditors/rc/fe/baja/AbsTimeRangeEditor',
  'nmodule/webEditors/rc/fe/baseEditorSet',
  'baja!' +
  'baja:AbsTime,' +
  'baja:AbsTimeRange,' +
  'baja:TimeRange,' +
  'baja:Time,' +
  'baja:Weekday,' +
  'bql:DynamicTimeRangeType'], function ($,
                                         Promise,
                                         ComplexCompositeEditor,
                                         events,
                                         Properties,
                                         ComplexCompositeBuilder,
                                         dynamicTimeRangeDialogEditorStructure,
                                         lexs) {

  "use strict";

  var MODIFY_EVENT = events.MODIFY_EVENT,
      TIMEZONE_PROPERTY = "TimeZone";

  /**
   * A DynamicTimeRangeDialogEditor is the editor for bql:DynamicTimeRange.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/bql/DynamicTimeRangeDialogEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   */

  var DynamicTimeRangeDialogEditor = function () {
    var that = this;
    ComplexCompositeEditor.apply(this, arguments);

    that.validators().add(function (diff) {
      var comp = that.value().newCopy();
      return diff.apply(comp)
        .then(function () {
          return startBeforeEnd(comp);
        });
    });
  };
  DynamicTimeRangeDialogEditor.prototype = Object.create(ComplexCompositeEditor.prototype);
  DynamicTimeRangeDialogEditor.prototype.constructor = DynamicTimeRangeDialogEditor;

  /**
   * Return all the properties of the Custom Complex that allows editing of DynamicTimeRange
   * @returns {string[]}
   */
  DynamicTimeRangeDialogEditor.prototype.getSlotFilter = function () {
    return ['startFixed', 'startTime', 'endFixed', 'endTime'];
  };

  /**
   * Build the sub editor dom
   * @private
   * @returns {jQuery}
   */
  DynamicTimeRangeDialogEditor.prototype.$buildSubEditorDom = function (key) {
    return $('.js-' + key, this.jq());
  };

  /**
   * initialize the dom and register all event handlers
   * @param {jQuery} dom
   * @returns {Promise}
   */
  DynamicTimeRangeDialogEditor.prototype.doInitialize = function (dom) {
    var that = this,
        div  = $("<div></div>");

    dom.on(MODIFY_EVENT, '.slot-startFixed, .slot-endFixed', function () {
      that.$checkTimeEditors();
    });

    div.html(dynamicTimeRangeDialogEditorStructure());
    that.jq().append(div);
    //doing this last allows squelching of the premature modify event
    return ComplexCompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Load the current value and setup any dependant slot interactions
   * @returns {Promise}
   * @param value {baja.Value}
   */
  DynamicTimeRangeDialogEditor.prototype.doLoad = function () {
    var that = this;
    return ComplexCompositeEditor.prototype.doLoad.apply(this, arguments)
      .then(function () {
        return that.$checkTimeEditors();
      });
  };

  /**
   * Ensure that the value is valid time range
   * @inner
   * @param {baja.Component} value
   */
  function startBeforeEnd(value) {
    var startFixed = value.get("startFixed"),
        endFixed   = value.get("endFixed"),
        startTime  = value.get('startTime'),
        endTime    = value.get('endTime');

    if (startFixed && endFixed && startTime.getJsDate().getTime() > endTime.getJsDate().getTime()) {
      throw new Error(lexs[0].get("AbsTimeRangeEditor.startAfterEnd"));
    }
  }

  /**
   * Return the Start Time element.
   * @private
   * @returns {jquery}
   */
  DynamicTimeRangeDialogEditor.prototype.$getStartTimeEditor = function () {
    return this.jq().find('td.slot-startTime').data('widget');
  };

  /**
   * Return the End Time  element.
   * @private
   * @returns {jquery}
   */
  DynamicTimeRangeDialogEditor.prototype.$getEndTimeEditor = function () {
    return this.jq().find('td.slot-endTime').data('widget');
  };

  /**
   * Return the Start Fixed element.
   * @private
   * @returns {jquery}
   */
  DynamicTimeRangeDialogEditor.prototype.$getStartFixedEditor = function () {
    return this.jq().find('td.slot-startFixed').data('widget');
  };

  /**
   * Return the End Fixed element.
   * @private
   * @returns {jquery}
   */
  DynamicTimeRangeDialogEditor.prototype.$getEndFixedEditor = function () {
    return this.jq().find('td.slot-endFixed').data('widget');
  };

  DynamicTimeRangeDialogEditor.prototype.makeBuilder = function () {
    var that    = this,
      builder = ComplexCompositeEditor.prototype.makeBuilder.call(that);
    builder.getConfigFor = function (key) {
      return ComplexCompositeBuilder.prototype.getConfigFor.call(builder, key)
        .then(function (config) {
          if (key === "startTime" || key === "endTime") {
            config.properties = Properties.extend(config.properties, {
              TimeZone: that.properties().getValue(TIMEZONE_PROPERTY)
            });
          }
          return config;
        });
    };
    return builder;
  };

  /**
   * Setup any interdependency between the checkboxes and the other editors
   * @returns {Promise}
   */
  DynamicTimeRangeDialogEditor.prototype.$checkTimeEditors = function () {
    var that             = this,
        startTimeEditor  = that.$getStartTimeEditor(),
        endTimeEditor    = that.$getEndTimeEditor(),
        startFixedEditor = that.$getStartFixedEditor(),
        endFixedEditor   = that.$getEndFixedEditor();

    return Promise.join(startFixedEditor.read(), endFixedEditor.read())
      .spread(function (startEnabled, endEnabled) {
        return Promise.join(
          startTimeEditor.setEnabled(startEnabled),
          endTimeEditor.setEnabled(endEnabled)
        );
      });
  };

  return DynamicTimeRangeDialogEditor;
});
