/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/fe/WebHistoryTable
 */
define([ 'baja!',
  'baja!bql:FilterSet,bql:DynamicTimeRange',
  'lex!history',
  'log!nmodule.history.rc.fe.WebHistoryTable',
  'jquery',
  'Promise',
  'underscore',
  'bajaux/events',
  'bajaux/commands/ToggleCommand',
  'bajaux/mixin/responsiveMixIn',
  'bajaux/mixin/subscriberMixIn',
  'bajaux/util/CommandButton',
  'nmodule/bql/rc/fe/DynamicTimeRangeEditor',
  'nmodule/history/rc/fe/HistoryTablePaginationModel',
  'nmodule/history/rc/fe/HistoryTableStateHandler',
  'nmodule/history/rc/historyUtil',
  'nmodule/history/rc/TableFilter',
  'nmodule/js/rc/switchboard/switchboard',
  'nmodule/webEditors/rc/fe/BaseWidget',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/wb/PropertySheet',
  'nmodule/webEditors/rc/wb/table/Table',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu',
  'nmodule/webEditors/rc/wb/table/pagination/PaginationWidget',
  'nmodule/webEditors/rc/util/htmlUtils',
  'hbs!nmodule/history/rc/fe/templates/historyTable',
  'css!nmodule/history/rc/fe/historyEditors' ], function (
  baja,
  types,
  lexs,
  log,
  $,
  Promise,
  _,
  events,
  ToggleCommand,
  responsiveMixin,
  subscriberMixIn,
  CommandButton,
  DynamicTimeRangeEditor,
  HistoryTablePaginationModel,
  HistoryTableStateHandler,
  historyUtil,
  FilterCommand,
  switchboard,
  BaseWidget,
  fe,
  PropertySheet,
  Table,
  feDialogs,
  DefaultTableContextMenu,
  PaginationWidget,
  htmlUtils,
  template) {

  'use strict';

  var logSevere = log.severe.bind(log),
    historyLex = lexs[0],
    MODIFY_EVENT = events.MODIFY_EVENT,
    PAGE_SIZE_PROP_NAME = 'pageSize',
    LIVE_PROP_NAME = 'live',
    DELTA_PROP_NAME = 'delta',
    SORT_COLUMN_PROP_NAME = 'sortColumn',
    SORT_DESCENDING_PROP_NAME = 'sortDescending',
    DASHBOARD_PROP_NAME = 'dashboard',
    MAX_PAGESIZE = 10000,
    DEFAULT_PAGESIZE = 500;

  function widgetDefaults() {
    return {
      properties: {
        pageSize: { value: 500, typeSpec: 'baja:Integer', dashboard: true },
        live: { value: false, hidden: true, dashboard: false },
        delta: { value: false, dashboard: true },
        sortColumn: { value: '', dashboard: true },
        sortDescending: { value: false, dashboard: true },
        dashboard: { value: false, hidden: true, transient: true, readonly: true },
        defaultTimeRange: {
          value: dynamicTimeRange().encodeToString(),
          typeSpec: 'bql:DynamicTimeRange',
          dashboard: true
        }
      }
    };
  }

  /**
   * HistoryTable derived from Table to override the sort implementation.
   *
   * @class
   * @alias module:nmodule/history/rc/fe/WebHistoryTable
   * @extends module:nmodule/webEditors/rc/wb/table/Table
   * @param {object} params
   */
  var HistoryTable = function HistoryTable(params) {
    Table.apply(this, arguments);
  };
  HistoryTable.prototype = Object.create(Table.prototype);
  HistoryTable.prototype.constructor = HistoryTable;

  /**
   * Override the sort ordering
   * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
   * @param {boolean} desc
   * @returns {Promise}
   */
  HistoryTable.prototype.sort = function (column, desc) {
    return this.$paginationModel.sort(column, desc);
  };

  /**
   * HistoryTable widget for displaying history tables.
   *
   * @private
   * @class
   * @alias module:nmodule/history/rc/fe/WebHistoryTable
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   */
  var WebHistoryTable = function WebHistoryTable(params) {
    BaseWidget.call(this, { params: params, defaults: widgetDefaults() });
    subscriberMixIn(this);

    switchboard(this, {
      $updateQueryParams: { allow: 'oneAtATime', onRepeat: 'returnLast' }
    });

    responsiveMixin(this);
  };
  WebHistoryTable.prototype = Object.create(BaseWidget.prototype);
  WebHistoryTable.prototype.constructor = WebHistoryTable;

  /**
   * Set up the DOM for this HistoryTable
   * @param {JQuery} dom - The DOM element into which this `Widget` is loaded.
   */
  WebHistoryTable.prototype.doInitialize = function (dom) {
    var that = this;
    var props = that.properties();

    dom.addClass('WebHistoryTable').html(template());

    that.$delta = props.getValue(WebHistoryTable.DELTA_PROP_NAME, false);
    that.$live = props.getValue(WebHistoryTable.LIVE_PROP_NAME, false);

    var pageSize = Math.max(props.getValue(WebHistoryTable.PAGE_SIZE_PROP_NAME), 1);
    var sortColumn = props.getValue(WebHistoryTable.SORT_COLUMN_PROP_NAME);
    var sortDirection = props.getValue(WebHistoryTable.SORT_DESCENDING_PROP_NAME) ? 'desc' : 'asc';

    // Create the pagination widget
    var paginationWidget = that.$paginationWidget = new PaginationWidget({
      properties: {
        maxPageSize: WebHistoryTable.MAX_PAGESIZE
      }
    });
    var paginationDom = dom.find('.paginationWidget');
    var paginationModel = that.$paginationModel = new HistoryTablePaginationModel({
      rowsPerPage: pageSize,
      subscriber: that.getSubscriber(),
      sortColumn: sortColumn,
      sortDirection: sortDirection,
      sortingAllowed: true
    });

    // Build the DynamicTimeRangeEditor, 
    //  Live command button, 
    //  Delta command button,
    return Promise.all([ fe.buildFor({
      formFactor: 'mini',
      type: DynamicTimeRangeEditor, 
      dom: $('.timeRangeEditor', dom),
      value: that.getTimeRange(),
      properties: {
        TimeZone: null
      }
    }), fe.buildFor({
      formFactor: 'mini',
      dom: $('.liveCommand', dom),
      type: CommandButton,
      value: that.$getLiveCommand()
    }), fe.buildFor({
      formFactor: 'mini',
      dom: $('.deltaCommand', dom),
      type: CommandButton,
      value: that.$getDeltaCommand()
    }) ])
      .spread(function (timeRangeEditor, liveCommandButton, deltaCommandButton) {
        that.$timeRangeEditor = timeRangeEditor;
        that.$liveCommandButton = liveCommandButton;
        that.$deltaCommandButton = deltaCommandButton;

        // Subscribe to the modify event
        dom.on(MODIFY_EVENT, function () {
          timeRangeEditor.read()
            .then(function (timeRange) {
              return that.$updateTimeRange(timeRange);
            }).catch(logSevere);
        });

        // Build the history table
        return fe.makeFor({
          type: HistoryTable,
          properties: {
            fixedHeaders: true,
            paginationModel: that.$paginationModel
          }
        })
        .then(function (table) {
          that.$table = table;
          table.on('sorted', function (sortColumn, sortDirection) {
            props.setValue(WebHistoryTable.SORT_COLUMN_PROP_NAME, sortColumn);
            props.setValue(WebHistoryTable.SORT_DESCENDING_PROP_NAME, sortDirection === 'desc');
          });
          return table.initialize(that.jq().find('.historyTable'));
        });        
      })
      .then(function () {
        // Build the pagination widget
        return fe.buildFor({
            dom: paginationDom,
            value: paginationModel,
            formFactor: 'mini'
          },
          paginationWidget
        );
      })
      .then(function () {
        that.$defaultTableMenu = new DefaultTableContextMenu(that.$table);
        that.$defaultTableMenu.arm(that.jq(), '.showHideMenu');

        that.$addDialogHandlers();

        // Live currently only works when page 1 is being viewed (otherwise disable the button)
        paginationModel.on('changed', function (name, value) {
          if (name === 'currentPage') {
            that.$updateLiveCommand(value);
            that.$saveState();
          } else if (name === 'rowsPerPage') {
            that.$saveState();
          }
        });

        // Build the Filter command
        that.$filterCommand = FilterCommand.makeFor(that);
        return fe.buildFor({
          formFactor: 'mini',
          type: CommandButton,
          dom: $('.filterCommand', dom),
          value: that.$filterCommand
        }).then(function () {
          that.$filterCommand.on('filterChanged', that.$handleFilterChanged.bind(that));
        });
      });
  };

  /**
   * Update the time range
   *
   * @private
   * @param {baja.Simple} timeRange `bql:DynamicTimeRange`
   * @returns {Promise} resolves when the query with the updated time range has completed
   */
  WebHistoryTable.prototype.$updateTimeRange = function (timeRange) {
    this.setTimeRange(timeRange);
    this.$saveState();
    return this.$updateQueryParams();
  };

  WebHistoryTable.prototype.$saveState = function () {
    if (this.$stateHandler) {
      return this.$stateHandler.save(this);
    }
  };

  /**
   * Resolve the data to something useful.
   * 
   * This implementation sets up the models (pagination model, etc) and
   * then the framework calls doLoad, which will update the pagination widget.
   * 
   * It also calls doChanged to update the command buttons based on the restored
   * state.
   * 
   * The filter widget is updated each time the filter dialog is opened, so
   * there's no need to do anything special to handle it.
   *
   * @param {*|String|baja.Ord} data Specifies some data used to resolve a
   * load value so `load(value)` can be called on the widget.
   * @param {Object} [resolveParams] An Object Literal used for ORD resolution.
   * This parameter is designed to be used internally by bajaux and
   * shouldn't be used by developers.
   * @returns {Promise} a promise to be resolved with the value resolved from
   * the given data object
   */
  WebHistoryTable.prototype.resolve = function (data, resolveParams) {
    var that = this;
    return baja.Ord.make(data).get()
      .then(function (value) {
        return that.$resolveHistoryOrd(data, value);
      })
      .then(function (ord) {
        return that.$resolveHistory(ord, resolveParams);
      });
  };

  /**
   * Resolves the history ord for the table to use. This will first attempt to
   * get the history ord from a history extension. If not pointing to a history
   * extension, this will look for a history tag. If no history tag, then it
   * will fallback to whatever was passed in for "data."
   *
   * @private
   *
   * @param {*|String|baja.Ord} data
   * @param {baja.coll.Table|baja.Component} value where baja.Component is a
   * history:HistoryExt
   * @returns {Promise.<baja.Ord|baja.Component|baja.coll.Table>}
   */
  WebHistoryTable.prototype.$resolveHistoryOrd = function (data, value) {
    if (!baja.hasType(value, 'baja:Component')) { return Promise.resolve(data); }
    return resolveHistoryId(value)
      .then(function (id) {
        return id ? baja.Ord.make('history:' + id) : data;
      });
  };

  /**
   * Resolves the history we want to load
   *
   * @private
   *
   * @param {*|String|baja.Ord} data
   * @param {Object} resolveParams
   * @returns {Promise.<baja.coll.Table>} resolves with the history
   */
  WebHistoryTable.prototype.$resolveHistory = function (data, resolveParams) {
    var that = this;
    var returnValue;


    that.$ordBody = getHistoryOrdBody(data);

    // Call the base class
    var returnPromise = BaseWidget.prototype.resolve.apply(that, arguments);
    if (!that.$isOnDashboard()) {
      return returnPromise
        .then(function (value) {
          returnValue = value;
          // Create the state handler
          return HistoryTableStateHandler.make();
        })
        .then(function (handler) {
          // Restore the values from session storage, which requires the ordBody
          // as part of the key.
          that.$stateHandler = handler;

          return that.$stateHandler.restore(that.$ordBody);
        })
        .then(function (state) {
          that.$state = state;

          var props = that.properties(),
            sortColumnProp = props.getValue(WebHistoryTable.SORT_COLUMN_PROP_NAME),
            timeRange = state.timeRange;

          // if dashboard properties are defined, they should override whatever
          // is stored in state.
          if (sortColumnProp) {
            state.sortColumn = sortColumnProp;
            state.sortDirection =
              props.getValue(WebHistoryTable.SORT_DESCENDING_PROP_NAME) ? 'desc' : 'asc';
          }

          // Apply the restored the filter if one was set
          if (state.filterSet) {
            that.$filterCommand.setFilterSet(state.filterSet);
          }

          // Restore the rowsPerPage, currentPage, and live flag
          that.$paginationModel.initialize(_.pick(state, 
            'rowsPerPage', 'currentPage', 'live', 'sortColumn', 'sortDirection'));

          // For delta, set this.$delta
          that.$delta = !!state.delta;

          props.setValue(WebHistoryTable.LIVE_PROP_NAME, !!state.live);
          props.setValue(WebHistoryTable.DELTA_PROP_NAME, !!state.delta);

          // Apply the restored timeRange if one was set
          if (timeRange) {
            that.setTimeRange(timeRange);
            return that.$getTimeRangeEditor().load(timeRange);
          }
        })
        .then(function () {
          return returnValue;
        });
    } else {
      return returnPromise;
    }
  };

  /**
   * Load a history table
   *
   * @param {baja.coll.Table} value
   *
   * @returns {Promise}
   */
  WebHistoryTable.prototype.doLoad = function (value) {
    var that = this;

    return value.toConfig()
      .then(function (config) {
        var recordType = config.get('recordType');
        that.$config = config;
        that.$getTimeRangeEditor().properties().setValue('TimeZone', config.getTimeZone().encodeToString());
        return Promise.all([
          that.$updateQueryParams(),
          recordType && baja.importTypes([ recordType.encodeToString() ]),
          that.$getTimeRangeEditor().load(that.getTimeRange())
        ]);
      })
      .spread(function (queryParams, recordTypes) {
        //show the deltaCommand for NumericTrendRecord only
        that.jq().find('.deltaCommand').toggle(recordTypes[0].is('history:NumericTrendRecord'));
        return that.$paginationModel.$resolveLastTime(value);
      })
      .then(function () {
        return that.$table.load(that.$paginationModel);
      })
      .then(function () {
        return that.$paginationModel.$setLive(that.properties().getValue(WebHistoryTable.LIVE_PROP_NAME, false));
      });
  };

  /**
   * Get the current page being displayed
   * 
   * @returns {Number} current page number
   */
  WebHistoryTable.prototype.getCurrentPage = function () {
    return this.$paginationModel.getCurrentPage();
  };

  /**
   * Get the number of rows per page being displayed
   */
  WebHistoryTable.prototype.getRowsPerPage = function () {
    return this.$paginationModel.getRowsPerPage();
  };

  /**
   * Get the current filter, which may be undefined
   * 
   * @returns {baja.Component|undefined} `bql:FilterSet`
   */
  WebHistoryTable.prototype.getFilterSet = function () {
    return this.$filterCommand.getFilterSet();
  };

  /**
   * Set the current filter
   * 
   * @param {baja.Component|undefined} filterSet `bql:FilterSet`
   */
  WebHistoryTable.prototype.setFilterSet = function (filterSet) {
    this.$filterCommand.setFilterSet(filterSet);
  };

  /**
   * Get the delta flag
   * @returns {Boolean} delta flag
   */
  WebHistoryTable.prototype.getDelta = function () {
    return this.$delta;
  };

  /**
   * Get the live flag
   * @returns {Boolean} live flag
   */
  WebHistoryTable.prototype.getLive = function () {
    return this.$paginationModel.getLive();
  };

  /**
   * @returns {Promise}
   */
  WebHistoryTable.prototype.doDestroy = function () {
    this.jq().removeClass('WebHistoryTable');
    return Promise.all([
      this.getChildWidgets().destroyAll(),
      this.$paginationModel.$setLive(false),
      this.$defaultTableMenu.destroy()
    ]);
  };

  /**
   * handler for show details command that displays a details dialog box
   * @private
   * @returns {Promise} a promise that resolves with an instance of the editor in the dialog.
   */
  WebHistoryTable.prototype.$showDetails = function () {
    var selected = this.$getTable().getSelectedRows()[0];
    if (selected) {
      return feDialogs.showFor({
        type: PropertySheet,
        value: selected.getSubject(),
        formFactor: 'compact',
        readonly: true,
        title: historyLex.get('WebHistoryTable.details.title'),
        properties: {
          displayOnly: true,
          showHeader: false
        }
      });
    } else {
      return Promise.reject(new Error('No selected rows'));
    }
  };

  /**
   * arms the rows of the table with the details dialog
   * @private
   */
  WebHistoryTable.prototype.$addDialogHandlers = function () {
    var that = this;
    htmlUtils.onLongPress(that.jq(), function (event) {
      $(event.target).click(); //click to select a row
      that.$showDetails().catch(logSevere);
    }, { selector: 'tr', fireClick: true });

    that.jq().on('dblclick', 'tr', function () {
      that.$showDetails().catch(logSevere);
      return false;
    });
  };

  /**
   * @returns {baja.Simple} `bql:DynamicTimeRange`
   */
  WebHistoryTable.prototype.getTimeRange = function () {
    return dynamicTimeRange(this.properties().getValue('defaultTimeRange'));
  };

  WebHistoryTable.prototype.setTimeRange = function (timeRange) {
    this.properties().setValue('defaultTimeRange', timeRange.encodeToString());
  };

  /**
   * Prevent the TableFilter from adding the timestamp to the filter because its already present in the command bar
   * @returns {Array.<module:nmodule/webEditors/rc/wb/table/model/Column>}
   */
  WebHistoryTable.prototype.getFilterColumns = function () {
    var columns = this.value().getColumns();
    return columns.filter(function (col) {
      return col.getName() !== 'timestamp';
    });
  };

  /**
   * @private
   * @param {number} currentPage
   */
  WebHistoryTable.prototype.$updateLiveCommand = function (currentPage) {
    if (historyUtil.isLive(this.getTimeRange()) && currentPage === 1) {
      this.$getLiveCommand().setEnabled(true);
    } else {
      this.$getLiveCommand().setSelected(false);
      this.$getLiveCommand().setEnabled(false);
    }
  };

  /**
   * @private
   * @param {boolean} resetPage set to true if the page should be reset to 1
   * 
   * @returns {Promise} resolves after the query parameters have been updated
   * and the current page has been reloaded.
   */
  WebHistoryTable.prototype.$updateQueryParams = function (resetPage) {
    this.$updateLiveCommand(this.getCurrentPage());
    return this.$paginationModel.setQueryParams({
      ordBody: this.getOrdBody(),
      filter: this.getFilterSet(),
      delta: this.$delta,
      config: this.$config,
      timeRange: this.getTimeRange(),
      resetPage: resetPage
    });
  };

  /**
   * Get the ord body for the ord that is currently being edited.
   * 
   * @returns {String}
   */
  WebHistoryTable.prototype.getOrdBody = function () {
    return this.$ordBody;
  };

  /**
   * Handle the 'filterChanged' event, called when the user changes t
   * he filter.
   * 
   * @private
   * @returns {Promise}
   */
  WebHistoryTable.prototype.$handleFilterChanged = function () {
    this.$saveState();
    return this.$updateQueryParams(true);
  };

  /**
   * @private
   * @returns {module:nmodule/bql/rc/fe/DynamicTimeRangeEditor}
   */
  WebHistoryTable.prototype.$getTimeRangeEditor = function () {
    return this.$timeRangeEditor;
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/table/Table}
   */
  WebHistoryTable.prototype.$getTable = function () {
    return this.$table;
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/table/pagination/PaginationWidget}
   */
  WebHistoryTable.prototype.$getPaginationWidget = function () {
    return this.$paginationWidget;
  };

  /**
   * @private
   * @returns {module:bajaux/commands/ToggleCommand}
   */
  WebHistoryTable.prototype.$getLiveCommand = function () {
    var that = this;
    if (!that.$liveCommand) {
      that.$liveCommand = new ToggleCommand({
        module: 'history',
        lex: 'commands.toggleLiveCmd',
        selected: that.properties().getValue(WebHistoryTable.LIVE_PROP_NAME)
      });

      that.$liveCommand.on(events.command.SELECTION_EVENT, function () {
        that.$paginationModel.$setLive(that.$liveCommand.isSelected())
          .then(function () {
            that.$saveState();
          })
          .catch(baja.error);        
      });
    }

    return that.$liveCommand;
  };

  /**
   * @private
   * @returns {module:bajaux/commands/ToggleCommand}
   */
  WebHistoryTable.prototype.$getDeltaCommand = function () {
    var that = this;
    if (!that.$deltaCommand) {
      that.$deltaCommand = new ToggleCommand({
        module: 'history',
        lex: 'commands.toggleDeltaCmd',
        selected: that.properties().getValue(WebHistoryTable.DELTA_PROP_NAME)
      });

      that.$deltaCommand.on(events.command.SELECTION_EVENT, function () {
        that.$setDelta(that.$deltaCommand.isSelected())
          .then(function () {
            that.$saveState();
          })
          .catch(baja.error);        
      });
    }

    return that.$deltaCommand;
  };

  /**
   * @private
   * @param {boolean} delta
   * @returns {Promise}
   */
  WebHistoryTable.prototype.$setDelta = function (delta) {
    if (delta === this.$delta) {
      return Promise.resolve();
    }

    this.$delta = delta;

    return this.$updateQueryParams();
  };

  /**
   * @returns {boolean} true if this widget is on a dashboard.
   */
  WebHistoryTable.prototype.$isOnDashboard = function () {
    return this.properties().getValue(WebHistoryTable.DASHBOARD_PROP_NAME);
  };

  /**
   * @param  {String} name The name of the Property that's changed.
   * @param  {*} value The new Property value.
   * @return {Promise}
   */
  WebHistoryTable.prototype.doChanged = function (name, value) {
    var that = this;
    if (name === WebHistoryTable.LIVE_PROP_NAME) {
      return Promise.resolve(that.$liveCommand.setSelected(value));
    } else if (name === WebHistoryTable.DELTA_PROP_NAME) {
      return Promise.resolve(that.$deltaCommand.setSelected(value));
    } else if (name === WebHistoryTable.PAGE_SIZE_PROP_NAME) {
      return that.$paginationModel.setRowsPerPage(Math.max(value, 1));
    }
    return Promise.resolve();
  };

  /**
   * Return the last history scheme's body.
   * @inner
   * @param {*|String|baja.Ord} ordString
   * @return {String}
   */
  function getHistoryOrdBody(ordString) {
    var queries = baja.Ord.make(String(ordString)).normalize().parse(),
      i, schemeName, query;

    for (i = queries.size() - 1; i >= 0; i--) {
      query = queries.get(i);
      schemeName = query.getSchemeName();
      if (schemeName === "history") {
        return query.getBody();
      }
    }

    return "";
  }

  /**
   * The name of the page size property.
   *
   * @type {string}
   */
  WebHistoryTable.PAGE_SIZE_PROP_NAME = PAGE_SIZE_PROP_NAME;

  /**
   * The name of the live property.
   *
   * @type {string}
   */
  WebHistoryTable.LIVE_PROP_NAME = LIVE_PROP_NAME;

  /**
   * The name of the delta property.
   *
   * @type {string}
   */
  WebHistoryTable.DELTA_PROP_NAME = DELTA_PROP_NAME;

  /**
   * The name of the sort column property.
   *
   * @type {string}
   */
  WebHistoryTable.SORT_COLUMN_PROP_NAME = SORT_COLUMN_PROP_NAME;

  /**
   * The name of the sort descending property.
   *
   * @type {string}
   */
  WebHistoryTable.SORT_DESCENDING_PROP_NAME = SORT_DESCENDING_PROP_NAME;

  /**
   * The name of the dashboard property.
   * 
   * @type {string} The property name.
   */
  WebHistoryTable.DASHBOARD_PROP_NAME = DASHBOARD_PROP_NAME;

  /**
   * Max Page Size for history tables in rows per page
   */
  WebHistoryTable.MAX_PAGESIZE = MAX_PAGESIZE;

  /**
   * Default Page Size for history tables in rows per page
   */
  WebHistoryTable.DEFAULT_PAGESIZE = DEFAULT_PAGESIZE;

  function dynamicTimeRange(enc) {
    return enc ? baja.$('bql:DynamicTimeRange', enc) : baja.$('bql:DynamicTimeRange');
  }

  /**
   * Resolves the history id for a given component. If no history id is available,
   * the returned promise will resolve to null.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {Promise<String|null>}
   */
  function resolveHistoryId(comp) {
    return resolveHistoryIdFromHistoryExtension(comp)
      .then(function (id) {
        if (id) { return id; }
        return resolveHistoryIdFromHistoryTags(comp);
      });
  }

  /**
   * Resolves a history id on a component based upon the history tag if
   * available.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {Promise<baja.Ord|null>}
   */
  function resolveHistoryIdFromHistoryTags(comp) {
    if (baja.hasType(comp, "niagaraVirtual:NiagaraVirtualComponent") || baja.hasType(comp, "niagaraDriver:NiagaraProxyExt")) {
      return comp.rpc("fetchRemoteTags", [ "n:history" ])
        .then(function (tagMap) {
          var tag = tagMap && tagMap[ "n:history" ];
          return tag;
        });
    }

    return comp.tags().then(function (tags) {
      var tag = tags.get('n:history');
      return tag;
    });
  }

  /**
   * Resolves the history id for a given component if available from a history
   * extension.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {Promise.<String|null>}
   */
  function resolveHistoryIdFromHistoryExtension(comp) {
    if (!baja.hasType(comp, "niagaraVirtual:NiagaraVirtualComponent")) {
      if (!baja.hasType(comp, 'history:HistoryExt')) { return Promise.resolve(null); }

      var historyConfig = comp.getHistoryConfig();
      return historyConfig.lease()
        .then(function () {
          var historyId = historyConfig.getId();
          return historyId;
        });
    }

    var navOrd = comp.getNavOrd();

    if (!navOrd) { return Promise.resolve(null); }

    return baja.rpc("type:niagaraVirtual:NiagaraVirtualComponent", "findDescendantHistories", navOrd.toString(), 1)
      .then(function (historyIds) {
        if (historyIds && historyIds.length) { return historyIds[0]; }
        return null;
      })
      .catch(function (err) {
        logSevere(err);
        return null;
      });
  }


  return WebHistoryTable;
});
