/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/fe/HistoryTableStateHandler
 */
define([ 'baja!',
  'baja!bql:FilterSet,bql:DynamicTimeRange',
  'Promise',
  'underscore',
  'nmodule/webEditors/rc/util/SyncedSessionStorage' ], function (
    baja,
    types,
    Promise,
    _,
    SyncedSessionStorage) {
  'use strict';

  var CACHE_STORAGE_KEY = 'niagara.webhistorytable.cache';

  /**
   * History Table State Handler
   * 
   * Handles state for WebHistoryTable.  This allows WebHistoryTable to 
   * persist (to session storage) its state, so that when a user navigates
   * away from the history table for a given ord, and then navigates back
   * to that order later, its state is restored.
   * 
   * Specifically, the types of things that are stored and restored should be:
   * - filter
   * - sort
   * - page row count
   * - page number
   * 
   * Constructor not to be called directly. Call `.make()` instead.
   * 
   * @class
   * @alias module:nmodule/history/rc/fe/HistoryTableStateHandler
   */
  var HistoryTableStateHandler = function HistoryTableStateHandler() {    
  };

  /**
   * Create a HistoryTableStateHandler
   * 
   * @param {Storage} [storage]
   * @returns {Promise<HistoryTableStateHandler>} resolves to a HistoryTableStateHandler
   */
  HistoryTableStateHandler.make = function (storage) {
    return Promise.resolve(storage || SyncedSessionStorage.getInstance())
    .then(function (storage) {
      var handler = new HistoryTableStateHandler();
      handler.$storage = storage;
      return handler;
    });
  };

  /**
   * Save the state for the specified WebHistoryTable
   * 
   * @param {module:nmodule/history/rc/fe/WebHistoryTable} historyTable 
   *  for which to save state
   */
  HistoryTableStateHandler.prototype.save = function (historyTable) {
    var state = {};
    var key = this.getKeyForSessionStorage();

    this.doSave(historyTable, state);

    // Get the current value for all history tables
    var allStates = JSON.parse(this.$storage.getItem(key) || '{}');

    // Update the value for the current ord
    allStates[historyTable.getOrdBody()] = state;

    // Save the value in session storage
    this.$storage.setItem(key, JSON.stringify(allStates));
  };

  /**
   * Restore the state from session storage for the specified ordBody
   * 
   * @param {String} ordBody
   * @returns {Promise.<Object>} restored state
   */
  HistoryTableStateHandler.prototype.restore = function (ordBody) {
    var key = this.getKeyForSessionStorage();
    var allStates = JSON.parse(this.$storage.getItem(key));
    if (allStates && allStates.hasOwnProperty(ordBody)) {
      var encoded = allStates[ordBody];
      return this.doDecode(encoded);
    } else {
      return Promise.resolve({});
    }
  };

  /**
   * Get the key for the session storage
   * 
   * @returns {String}
   */
  HistoryTableStateHandler.prototype.getKeyForSessionStorage = function () {
    return baja.getUserName() + '.' + CACHE_STORAGE_KEY;
  };

  /**
   * Apply the state to the WebHistoryTable
   * 
   * Typically the API user should call `HistoryTableStateHandler#save` instead
   * of calling `doSave`.
   * 
   * @param {module:nmodule/history/rc/fe/WebHistoryTable} historyTable 
   *  for which to save state
   * @param {Object} state to be populated with the WebHistoryTable state, 
   * in a JSON serializable form (baja components have been bson encoded).
   */
  HistoryTableStateHandler.prototype.doSave = function (historyTable, state) {
    state.currentPage = historyTable.getCurrentPage();
    var filterSet = historyTable.getFilterSet();
    if (filterSet) {
      state.filterSet = baja.bson.encodeValue(historyTable.getFilterSet());
    }
    state.rowsPerPage = historyTable.getRowsPerPage();
    state.timeRange = baja.bson.encodeValue(historyTable.getTimeRange());
    state.delta = historyTable.getDelta();
    state.live = historyTable.getLive();

    var tableModel = historyTable.$getTable().getModel(),
      sortColumn = tableModel && tableModel.$getSortColumn();

    if (sortColumn) {
      state.sortColumn = sortColumn;
      state.sortDirection = tableModel.$getSortDirection(sortColumn);
    }
  };

  /**
   * Decode the state
   * 
   * Typically the API user should call `HistoryTableStateHandler#restore` instead
   * of calling `doDecode`.
   * 
   * @returns {Promise.<Object>} decoded state
   */
  HistoryTableStateHandler.prototype.doDecode = function (state) {
    return Promise.all([
      state.filterSet && baja.bson.decodeAsync(state.filterSet),
      baja.bson.decodeAsync(state.timeRange)
    ])
      .spread(function (filterSet, timeRange) {
        var decodedState = _.pick(state, 'currentPage', 'rowsPerPage', 'delta',
          'live', 'sortColumn', 'sortDirection');
        if (filterSet) {
          decodedState.filterSet = filterSet;
        }
        decodedState.timeRange = timeRange;
        return decodedState;
      });
  };

  return HistoryTableStateHandler;
});
