/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/baja/CollectionInterval
 */
define(['baja!',
        'bajaScript/baja/obj/objUtil'], function (
         baja,
         objUtil) {

  'use strict';

  function relTime(millis) { return baja.RelTime.make(millis); }

  var Simple = baja.Simple,
      cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode,

      STR_REGEX = /([^:]+):([\+\-]?\d+)/,

      MILLIS_IN_MINUTE = baja.RelTime.MILLIS_IN_MINUTE,
      MINUTE = relTime(MILLIS_IN_MINUTE);

  /**
   * BajaScript representation of a `history:CollectionInterval`.
   *
   * @class
   * @extends baja.Simple
   * @alias nmodule/history/rc/baja/CollectionInterval
   */
  var CollectionInterval = function CollectionInterval() {
    Simple.apply(this, arguments);
  };
  CollectionInterval.prototype = Object.create(Simple.prototype);
  CollectionInterval.prototype.constructor = CollectionInterval;

  /**
   * Make a new regular `CollectionInterval` instance.
   *
   * @param {baja.RelTime} interval
   * @returns {module:nmodule/history/rc/baja/CollectionInterval}
   */
  CollectionInterval.make = function (interval) {
    if (!baja.hasType(interval, 'baja:RelTime')) {
      throw new Error('baja:RelTime required');
    }
    var ci = new CollectionInterval();
    ci.$irregular = false;
    ci.$interval = interval;
    return ci;
  };

  /**
   * Make a new regular `CollectionInterval` instance.
   * @see module:nmodule/history/rc/baja/CollectionInterval.make
   */
  CollectionInterval.prototype.make = function () {
    return CollectionInterval.make.apply(CollectionInterval, arguments);
  };

  CollectionInterval.prototype.decodeFromString = cacheDecode(function (str) {
    var result = STR_REGEX.exec(str);
    if (!result) {
      throw new Error('invalid history:CollectionInterval encoding: ' + str);
    }
    var interval = baja.RelTime.make(parseInt(result[2], 10)),
        ci = CollectionInterval.make(interval);

    if (result[1] === 'true') {
      ci.$irregular = true;
    }

    return ci;
  });

  CollectionInterval.prototype.encodeToString = cacheEncode(function () {
    return this.isIrregular() + ':' + this.getInterval().encodeToString();
  });

  /**
   * Get the collection interval.
   *
   * @returns {baja.RelTime}
   */
  CollectionInterval.prototype.getInterval = function () {
    return this.$interval;
  };

  /**
   * Return true if this interval is irregular.
   *
   * @returns {boolean}
   */
  CollectionInterval.prototype.isIrregular = function () {
    return this.$irregular;
  };

  CollectionInterval.DEFAULT = CollectionInterval.make(MINUTE);

  CollectionInterval.IRREGULAR = CollectionInterval.make(MINUTE);
  CollectionInterval.IRREGULAR.$irregular = true;

  CollectionInterval.REGULAR_1_MIN = CollectionInterval.DEFAULT;
  CollectionInterval.REGULAR_5_MIN = CollectionInterval.make(relTime(MILLIS_IN_MINUTE * 5));
  CollectionInterval.REGULAR_15_MIN = CollectionInterval.make(relTime(MILLIS_IN_MINUTE * 15));
  CollectionInterval.REGULAR_30_MIN = CollectionInterval.make(relTime(MILLIS_IN_MINUTE * 30));
  CollectionInterval.REGULAR_1_HOUR = CollectionInterval.make(relTime(MILLIS_IN_MINUTE * 60));

  baja.registerType('history:CollectionInterval', function () {
    return CollectionInterval;
  });

  return CollectionInterval;
});
