/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Stephen Wiley
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/TableFilter
 */
define([ 'baja!',
  'baja!bql:FilterEntry,bql:StringFilter,bql:AbsTimeFilter,bql:FilterSet,bql:IntegerFilter',
  'lex!history',
  'underscore',
  'bajaux/commands/Command',
  'Promise',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/webEditors/rc/servlets/registry',
  'nmodule/webEditors/rc/wb/table/model/Column' ], function (
  baja,
  types,
  lex,
  _,
  Command,
  Promise,
  feDialogs,
  registry,
  Column) {
  'use strict';

  var historyLex = lex[0];

  /**
   * @class
   * @alias module:nmodule/history/rc/TableFilter
   * @extends module:bajaux/commands/Command
   * @param view
   * @param params
   */
  var TableFilter = function (view, params) {
    Command.call(this, params);
    this.$view = view;
  };
  TableFilter.prototype = Object.create(Command.prototype);
  TableFilter.prototype.constructor = TableFilter;

  /**
   * @param view must have loaded value that implements BITable
   * @returns {module:nmodule/history/rc/TableFilter}
   */
  TableFilter.makeFor = function (view) {
    return new TableFilter(view, {
      module: 'history',
      lex: 'commands.filter',
      func: function () {
        return this.$showDialog();
      }
    });
  };

  /**
   * Indicates if this filter is active.
   *
   * @returns {boolean}
   */
  TableFilter.prototype.isActive = function () {
    if (!this.$filterSet) {
      return false;
    }
    var comps = this.$filterSet.getSlots().toValueArray();
    return _.any(comps, function (filterEntry) {
      return filterEntry.get('active');
    });
  };

  /**
   * @param {baja.Component|undefined} filterSet of type `bql:FilterSet`
   */
  TableFilter.prototype.setFilterSet = function (filterSet) {
    this.$filterSet = filterSet;
    this.$updateIcon();
  };

  /**
   * @returns {baja.Component} filterSet of type `bql:FilterSet`
   */
  TableFilter.prototype.getFilterSet = function () {
    return this.$filterSet;
  };

  /**
   * @private
   * @returns {Promise}
   */
  TableFilter.prototype.$showDialog = function () {
    var that = this;
    return Promise.resolve(this.$filterSet || that.$genFilterSet(that.$getColumnsToFilter()))
      .then(function (filterSet) {
        return feDialogs.showFor({
          value: filterSet,
          title: historyLex.get('commands.filter.displayName'),
          formFactor: 'mini'
        });
      })
      .then(function (newFilterSet) {
        if (newFilterSet) {
          that.$filterSetChanged(newFilterSet);
        }
      });
  };

  /**
   * Get columns that can be filtered on. To specify which columns to use, you
   * can add a `getFilterColumns` function to your view.
   * @returns {Array.<module:baja/coll/Table.TableColumn>}
   */
  TableFilter.prototype.$getColumnsToFilter = function () {
    var view = this.$view;

    return _.result(view, 'getFilterColumns') || view.value().getColumns();
  };

  /**
   * Generate a FilterSet based on the columns available to filter on.
   *
   * @param {Array.<module:baja/coll/Table.TableColumn>} or
   * {Array.<module:nmodule/webEditors/rc/wb/table/model/Column>} columns
   * @returns {Promise.<baja.Component>} which resolves to a `bql:FilterSet`
   */
  TableFilter.prototype.$genFilterSet = function (columns) {
    return getFilterTypeSpecs(columns)
      .then(function (filterTypes) {
        var filterSet = baja.$('bql:FilterSet');
        columns.forEach(function (column, i) {
          var filterType = filterTypes[i];
          if (filterType) {
            filterSet.add({
              slot: column.getName(),
              value: baja.$('bql:FilterEntry', {
                label: getColumnDisplayName(column),
                filter: initFilter(baja.$(filterType), getColumnDefaultValue(column))
              })
            });
          }
        });
        return filterSet;
      });
  };

  function getColumnDisplayName(column) {
    var displayName;
    if (column instanceof Column) {
      displayName = column.$displayName || column.getName();
    } else {
      displayName = column.getDisplayName();
    }
    return displayName;
  }

  /**
   * @param {Array.<module:baja/coll/Table.TableColumn>} columns
   * @returns {Array.<String>} a sparse array of the `bql:IBqlFilter` types to
   * instantiate to filter these columns. If a column has no available filter
   * type that index will be undefined.
   */
  function getFilterTypeSpecs(columns) {
    var batch = new baja.comm.Batch();
    return batch.commit(Promise.all(columns.map(function (column) {
      var typeSpec = baja.$('baja:TypeSpec', 'baja:String');
      if (column.getType) {
        typeSpec = column.getType().getTypeSpec();
      }
      return registry.getAgentOnInfo(typeSpec, {
        is: 'bql:IBqlFilter',
        batch: batch
      })
        .then(function (infos) {
          return _.result(_.first(infos), 'type');
        });
    })));
  }

  /**
   * @param {module:baja/coll/Table.TableColumn} column
   * @returns {baja.Value} a default value derived from the column, used to
   * initialize the filter for the column
   */
  function getColumnDefaultValue(column) {
    var columnType = baja.lt('baja:String');
    if (column.getType) {
      columnType = column.getType();
    }
    if (columnType.is('baja:Enum')) {
      var range = column.getFacets().get('range');
      if (range) {
        return baja.DynamicEnum.make({ range: range });
      }
    }
    return baja.$(columnType);
  }

  // TODO: AbsTimeFilter?
  // TODO: would this be better served as type extensions? c.f. BIBqlFilter#init
  /**
   * @param {baja.Complex} filter a `bql:IBqlFilter` instance
   * @param {baja.Value} value the value to use to initialize the filter
   * @returns {baja.Complex} the initialized filter
   */
  function initFilter(filter, value) {
    var filterType = filter.getType();
    if (filterType.is('bql:BitStringFilter')) {
      filter.set({ slot: 'include', value: value });
      filter.set({ slot: 'exclude', value: value });
    } else if (filterType.is('bql:EnumFilter')) {
      filter.set({ slot: 'enumType', value: value });
    }
    return filter;
  }

  /**
   * @private
   */
  TableFilter.prototype.$updateIcon = function () {
    if (this.isActive()) {
      this.setIcon(historyLex.get('commands.filtered.icon'));
    } else {
      this.setIcon(historyLex.get('commands.filter.icon'));
    }
  };

  /**
   * @private
   * @param {baja.Component} filterSet
   */
  TableFilter.prototype.$filterSetChanged = function (filterSet) {
    this.setFilterSet(filterSet);
    this.trigger('filterChanged', filterSet);
  };

  return TableFilter;
});
