/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/DatabaseMaintenance
 */
define([
  'baja!',
  'baja!history:History,history:HistorySpace',
  'lex!history',
  'jquery',
  'underscore',
  'Promise',
  'dialogs',
  'bajaux/events',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/wb/tree/NavTree',
  'nmodule/webEditors/rc/wb/tree/TreeNode',
  'nmodule/webEditors/rc/wb/tree/BajaNavTreeNode',
  'nmodule/webEditors/rc/util/htmlUtils',
  'bajaux/commands/Command',
  'bajaux/util/CommandButton',
  'nmodule/history/rc/maintenance',
  'hbs!nmodule/history/rc/templates/view',
  'hbs!nmodule/history/rc/templates/dialog',
  'css!nmodule/history/rc/fe/historyEditors'
], function (baja, types, lexs, $, _, Promise, dialogs, events, fe, BaseEditor,
             NavTree, TreeNode, BajaNavTreeNode, htmlUtils,
             Command, CommandButton,
             maintenance,
             viewTemplate, dialogTemplate) {

  'use strict';

  var lex = lexs[0],
      escapeHtml = htmlUtils.escapeHtml,
      DATABASE_MAINTENANCE_CLASS = 'dbMaintenance',
      HISTORY_NAV_TREE_CLASS = 'historyTree',
      HISTORY_NAV_TREE_SELECTOR = '.' + HISTORY_NAV_TREE_CLASS,
      HISTORY_LIST_SELECTOR = '.historyList > ul',
      DATE_EDITOR_CLASS = 'dateEditor',
      DATE_EDITOR_SELECTOR = '.' + DATE_EDITOR_CLASS,
      RUN_COMMAND_BUTTON_CLASS = 'run',
      RUN_COMMAND_BUTTON_SELECTOR = '.' + RUN_COMMAND_BUTTON_CLASS,

      SELECT_ALL_HISTORIES_MSG = lex.get('history.db.selectAllDisplayName'),
      SELECT_ALL_HISTORIES_ICON = lex.get('history.db.selectAllIcon'),
      DESELECT_ALL_HISTORIES_MSG = lex.get('history.db.deselectAllDisplayName'),
      DESELECT_ALL_HISTORIES_ICON = lex.get('history.db.deselectAllIcon'),

      CLEAR_OLD_HISTORIES_LABEL = lex.get('history.db.clearOld'),
      CLEAR_ALL_HISTORIES_LABEL = lex.get('history.db.clearAll'),
      DELETE_ALL_HISTORIES_LABEL = lex.get('history.db.deleteAll'),

      HISTORY_SELECTED_CLASS = 'history-selected',
      HISTORY_SELECTED_SELECTOR = '.' + HISTORY_SELECTED_CLASS,
      DELETE_ICON = baja.Icon.make(lex.getSafe('history.db.deleteIcon')).getImageUris()[0],
      DELETE_ENTRY_COMMAND_CLASS = 'deleteEntryCmd',

      HISTORY_REGEX = /history:\/[^\/]+\/.+/,
      HISTORY_DEVICE_REGEX = /history:\/[^\/].+/,
      GROUP_REGEX = /.*\/\/.*\/.*/,
      GROUP_HISTORY_REGEX = /.*\/\/.*\/\/.*/,
      DEFAULT_REGEX = /.*\/\/\//,
      WARNING_ICON = lex.getSafe('history.db.warningIcon'),

      CLEAR_ALL_OP_COMPLETE_TITLE = lex.get('history.db.dialog.clearAll.title'),
      CLEAR_OLD_COMPLETE_TITLE = lex.get('history.db.dialog.clearOld.title'),
      DELETE_ALL_COMPLETE_TITLE = lex.get('history.db.dialog.delete.title'),
      INVALID_TIME_TITLE = lex.get('history.db.dialog.invalidTime.title'),
      SELECT_HISTORY_MSG = lex.getSafe('history.db.selectMessage'),
      CONFIRM_CLEAR_MSG = lex.getSafe('history.db.dialog.confirmClear'),
      CONFIRM_DELETE_MSG = lex.getSafe('history.db.dialog.confirmDelete'),
      COMMAND_SUCCESS_MSG = lex.getSafe('history.db.successMessage'),
      COMMAND_ERROR_MSG = lex.getSafe('history.db.errorMessage'),

      OPERATIONS = {
        clearOld: 'clearOld',
        clearAll: 'clearAll',
        deleteAll: 'deleteAll'
      };

  /**
   * DatabaseMaintenance allows for running maintenance operations on
   * histories (clearing old histories, deleting histories etc.)
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/history/rc/DatabaseMaintenance
   */
  var DatabaseMaintenance = function DatabaseMaintenance() {
    BaseEditor.call(this, {moduleName: 'history', keyName: 'DatabaseMaintenance'});
  };

  //extend and set up prototype chain
  DatabaseMaintenance.prototype = Object.create(BaseEditor.prototype);
  DatabaseMaintenance.prototype.constructor = DatabaseMaintenance;


  function initCommandGroup(view) {
    var list = view.$getList();

    function selectAllHistories() {
      return view.$getHistories().then(function (histories) {
        appendNodesToList(histories, list);
      }).catch(function(err){
        dialogs.showOk({
          content: COMMAND_ERROR_MSG + "<br/>(" + escapeHtml(err) + ")"
        });
      });
    }

    function deselectAllHistories() {
      list.empty();
      view.jq().find(HISTORY_SELECTED_SELECTOR).removeClass(HISTORY_SELECTED_CLASS);
    }

    var cmdGroup = view.getCommandGroup(),
        selectAllCmd = new Command({
          displayName: SELECT_ALL_HISTORIES_MSG,
          description: SELECT_ALL_HISTORIES_MSG,
          icon: SELECT_ALL_HISTORIES_ICON,
          func: selectAllHistories
        }),

        deselectallCmd = new Command({
          displayName: DESELECT_ALL_HISTORIES_MSG,
          description: DESELECT_ALL_HISTORIES_MSG,
          icon: DESELECT_ALL_HISTORIES_ICON,
          func: deselectAllHistories
        });

    cmdGroup.add(selectAllCmd);
    cmdGroup.add(deselectallCmd);
  }

  function initNavTreeEditor(view) {
    return baja.Ord.make('history:').get().then(function (space) {
      return fe.buildFor({
        dom: $(HISTORY_NAV_TREE_SELECTOR, view.jq()),
        value: new BajaNavTreeNode(space),
        type: NavTree
      }).then(function (tree) {

        function getParent(tree) {
          var parent = tree.jq().parent().parent().data('widget');
          return parent instanceof NavTree && parent || getParent(parent);
        }

        tree.expand();

        //register event handler for double-clicking a tree node
        tree.jq().on(NavTree.ACTIVATED_EVENT, null, view, nodeDblClicked);

        //register batch select handler
        var anchor, //stores the anchor node in a shift + click operation
            node;//stores the currently selected node,

        tree.jq().on(NavTree.SELECTED_EVENT, function (e, root, modified) { //single click
          node = root; //cache the clicked node for use in 'mouseup'
        }).on('mouseup', function (e) { //batch selection using shift + click
          if (e.shiftKey && anchor && node && anchor !== node) {
            var anchorParent = getParent(anchor),
                nodeParent = getParent(node);

            //only permit batch selection for nodes with a common parent
            if (anchorParent && anchorParent === nodeParent) {
              var kids = anchorParent.$getKids(),
                  anchorIndex, nodeIndex, i,
                  startIndex, endIndex;

              for (i = 0; i < kids.length; i++) {

                //locate the anchor index
                if (kids[i] === anchor) {
                  anchorIndex = i;
                }

                //locate the node index
                else if (kids[i] === node) {
                  nodeIndex = i;
                }

                //break if both indices have been found
                if (anchorIndex && nodeIndex) {
                  break;
                }
              }

              startIndex = anchorIndex < nodeIndex ? anchorIndex : nodeIndex;
              endIndex = anchorIndex > nodeIndex ? anchorIndex : nodeIndex;
              _.each(kids.slice(startIndex, endIndex + 1), function (kid) {
                kid.setSelected(true, true);
              });
            }
          }

          else {
            anchor = node;
          }
        });

        //arm drag/drop handler
        var list = view.$getList();
        list.on('dragover', function () {
          return false;
        }).on('drop', function (e) {
          getNodes(tree, NavTree.prototype.isSelected).then(function (nodes) {
            appendNodesToList(nodes, list);
          });
          return false;
        });
      });
    });
  }

  function nodeDblClicked(event, node) {
    var view = event.data;
    appendNodesToList([node], view.$getList());
  }

  //node must be an instance of NavTree
  function isValid(node, list) {
    //a node is valid if it or one of it's ancestor's isn't already selected and
    //is either a history, history device, group or group history
    return !(isDefault(node) || ancestorSelected(node) || inList(node, list)) &&
      (isHistory(node) || isHistoryDevice(node) || isGroup(node) || isGroupHistory(node));
  }

  function getSelectedOrds(list){
    return list.children().map(function () {
      return $(this).data('ord');
    }).get();
  }

  function inList(node, list){
    var srcOrd = node.value().value().getNavOrd().relativizeToSession().toString().substring('history:'.length),
        targetOrds = getSelectedOrds(list),
        lastDblSlash = srcOrd.lastIndexOf('//');

    //get portion of ord after last double slash (for history groups)
    if(lastDblSlash !== -1){
      srcOrd = srcOrd.substring(lastDblSlash + 1);
    }

    return _.some(_.map(targetOrds, function(ord){
      return ord.toString().substring('history:'.length);
    }), function(targetOrd){
      lastDblSlash = targetOrd.lastIndexOf('//');
      if(lastDblSlash !== -1){
        targetOrd = targetOrd.substring(lastDblSlash + 1);
      }
      return srcOrd === targetOrd;
    });
  }

  function ancestorSelected(node){
    return node.jq().closest(HISTORY_SELECTED_SELECTOR).length;
  }

  function isDefault(node){
    var ord = node.value().value().getNavOrd().toString();
    return DEFAULT_REGEX.test(ord);
  }

  function isHistory(node) {
    var ord = node.value().value().getNavOrd().toString();
    return HISTORY_REGEX.test(ord);
  }

  function isHistoryDevice(node) {
    var ord = node.value().value().getNavOrd().toString();
    return HISTORY_DEVICE_REGEX.test(ord);
  }

  function isGroup(node){
    var ord = node.value().value().getNavOrd().toString();
    return GROUP_REGEX.test(ord);
  }

  function isGroupHistory(node) {
    var ord = node.value().value().getNavOrd().toString();
    return GROUP_HISTORY_REGEX.test(ord);
  }

  function getHistories(root) {
    return getNodes(root, isHistory);
  }

  function getHistoryGroups(root) {
    return getNodes(root, isGroupHistory);
  }

  function getHistoryDevices(root) {
    return getNodes(root, isHistoryDevice);
  }

  //DFS search of nodes from the specified root satisfying the filter
  function getNodes(root, filter) {
    if (filter.call(root, root)) {
      return Promise.resolve([root]);
    }

    return root.$setLoaded(true).then(function () {
      var kids = root.$getKids(),
          kidLoadingPromises = _.map(kids, function (kid) {
            return kid.$setLoaded(true);
          });
      return Promise.all(kidLoadingPromises).then(function () {
        var kidRetrievalPromises = _.map(kids, function (kid) {
          return getNodes(kid, filter);
        });
        return Promise.all(kidRetrievalPromises);
      });
    }).then(_.flatten);
  }

  //nodes must be NavTrees
  function appendNodesToList(nodes, list) {
    var parent = list.parent();
    list.detach();
    _.each(nodes, function (node) {
      if (isValid(node, list)) {
        var treeNode = node.value(),
            navNode = treeNode.value(),
            navOrd = navNode.getNavOrd().relativizeToSession(),
            text = baja.SlotPath.unescape(navOrd.toString().substring('history:'.length)),
            listItem;

        node.jq().addClass(HISTORY_SELECTED_CLASS);

        //append list item
        listItem = $("<li><div class='historyId'>" + text + "</div></li>")
          .data('ord', navOrd)
          .appendTo(list);

        //append delete img
        $('<img>').attr('src', DELETE_ICON)
          .addClass(DELETE_ENTRY_COMMAND_CLASS)
          .prependTo(listItem)
          .click(function () {
            //unmark the selected node
            node.jq().removeClass(HISTORY_SELECTED_CLASS);
            listItem.remove();
          });
      }
    });

    parent.append(list);
  }

  function initDateEditor(view) {
    var today = new Date(),
        year  = today.getFullYear() - 5,
        month = today.getMonth(),
        day   = today.getDate();

    return fe.buildFor({
      dom: $(DATE_EDITOR_SELECTOR, view.jq()),
      value: baja.AbsTime.make({jsDate: new Date(year, month, day)}), //NCCB-14801
      formFactor: 'mini'
    }).then(function (editor) {
      editor.jq().on(events.MODIFY_EVENT, function () {
        return false;
      });
    });
  }

  function initRunCommand(view) {
    return fe.buildFor({
      dom: $(RUN_COMMAND_BUTTON_SELECTOR, view.jq()),
      type: CommandButton,
      value: new Command({
        displayName: lex.get('history.db.run.displayName'),
        description: lex.get('history.db.run.description'),
        func: function () {
          return runMaintenance(view);
        }
      }),
      formFactor: 'mini'
    });
  }

  function runMaintenance(view) {

    function makeDialogContent(date) {
      var obj = {
            title: CONFIRM_CLEAR_MSG,
            completionTitle: CLEAR_OLD_COMPLETE_TITLE
          },
          warningIcon = baja.Icon.make(WARNING_ICON).getImageUris()[0],
          op = view.$getSelectedOperation(),
          content = {icon: warningIcon},
          ords = view.$getSelectedOrds(),
          args = [ords];

      if (!ords.length) {
        obj.title = 'No Records Selected';
        obj.content = SELECT_HISTORY_MSG;
      }

      else {
        switch (op) {
          case OPERATIONS.clearOld:
            obj.completionTitle = CLEAR_OLD_COMPLETE_TITLE;
            //content.time = date.getJsDate();
            content.operation = lex.getSafe({ key: "confirmClear.operation.clearOld", args: [ date.getJsDate() ] });
            args.push(date);
            break;
          case OPERATIONS.clearAll:
            obj.completionTitle = CLEAR_ALL_OP_COMPLETE_TITLE;
            content.operation = lex.getSafe("confirmClear.operation.clearAll");
            break;
          case OPERATIONS.deleteAll:
            obj.title = CONFIRM_DELETE_MSG;
            obj.completionTitle = DELETE_ALL_COMPLETE_TITLE;
            content.delete = 'delete';
            content.operation = lex.getSafe("confirmClear.operation.delete");
            break;
        }
        content.confirm = lex.getSafe("confirmClear.confirm");
        content.question = lex.getSafe("confirmClear.question");

        obj.content = dialogTemplate(content);
        obj.func = function () {
          return maintenance[op].apply(null, args).then(function () {
            return Promise.resolve(op === OPERATIONS.deleteAll &&
            view.$getDeselectAllCommand().invoke());
          }).then(function () {
            dialogs.showOk({
              title: obj.completionTitle,
              content: COMMAND_SUCCESS_MSG
            });
          }).catch(function (err) {
            dialogs.showOk({
              title: obj.completionTitle,
              content: COMMAND_ERROR_MSG + "<br/>(" + escapeHtml(err) + ")"
            });
          });
        };
      }
      return obj;
    }

    return view.$getDateEditor().read().then(function (date) {
      var obj = makeDialogContent(date);
      if (!obj.func) {
        dialogs.showOk({
          title: obj.title,
          content: obj.content
        });
      }

      else {
        var dlg = dialogs.showYesNo({
          title: obj.title,
          content: obj.content,
          yes: function(){
            dlg.close();
            dialogs.showLoading(0, obj.func());
          }
        });
      }
    }).catch(function(err){
      dialogs.showOk({
        title: INVALID_TIME_TITLE,
        content: err
      });
    });
  }

  function resize(view) {
    var containerHeight = $('.bajaux-container').outerHeight(true),
        maintenanceDivHeight = $('.maintenance', view.jq()).outerHeight(true),
        toolbarHeight = $('.bajaux-toolbar-outer').outerHeight(true),
        historyChooserDivHeight = containerHeight - maintenanceDivHeight - toolbarHeight - 30;
    $('.historyChooser', view.jq()).outerHeight(historyChooserDivHeight);
  }

  /**
   * Initializes the DOM with the HistorySpace tree, history selection list and options
   * for running maintenance
   * @param dom
   * @returns {Promise} Promise resolved once initializtion is complete
   */
  DatabaseMaintenance.prototype.doInitialize = function (dom) {

    var that = this;

    dom.addClass(DATABASE_MAINTENANCE_CLASS)
      .html(viewTemplate({
        clearOldId: this.generateId(),
        clearAllId: this.generateId(),
        deleteAllId: this.generateId(),
        clearOld: CLEAR_OLD_HISTORIES_LABEL,
        clearOldValue: OPERATIONS.clearOld,
        clearAll: CLEAR_ALL_HISTORIES_LABEL,
        clearAllValue: OPERATIONS.clearAll,
        deleteAll: DELETE_ALL_HISTORIES_LABEL,
        deleteAllValue: OPERATIONS.deleteAll
      }));

    this.$selectOperation(OPERATIONS.clearOld);

    /*global window*/
    $(window).resize(function () {
      resize(that);
    });

    initCommandGroup(this);

    return Promise.join(
      initNavTreeEditor(this),
      initDateEditor(this),
      initRunCommand(this)
    );
  };

  /**
   * Resizes the view once it's been initialized
   */
  DatabaseMaintenance.prototype.doLoad = function () {
    resize(this);
  };

  /**
   * Destroys all child editors and clears the history selection list
   * @param params
   * @returns {Promise}
   */
  DatabaseMaintenance.prototype.doDestroy = function (params) {
    this.$getList().empty();
    return this.getChildEditors().destroyAll();
  };

  /**
   * Returns a reference to the HistorySpace NavTree editor
   * @private
   * @returns {NavTree}
   */
  DatabaseMaintenance.prototype.$getNavTree = function () {
    return this.jq().find(HISTORY_NAV_TREE_SELECTOR).data('widget');
  };

  /**
   * Returns an array of all NavTree editors representing HistoryGroup nodes
   * @private
   * @returns {Promise} Promise resolved with array of NavTree nodes
   */
  DatabaseMaintenance.prototype.$getHistoryGroups = function () {
    return getHistoryGroups(this.$getNavTree());
  };

  /**
   * Returns an array of all NavTree editors representing HistoryDevice nodes
   * @private
   * @returns {Promise} Promise resolved with array of NavTree nodes
   */
  DatabaseMaintenance.prototype.$getHistoryDevices = function () {
    return getHistoryDevices(this.$getNavTree());
  };

  /**
   * Returns an array of all NavTree editors representing History nodes
   * @private
   * @returns {Promise} Promise resolved with array of NavTree nodes
   */
  DatabaseMaintenance.prototype.$getHistories = function () {
    return getHistories(this.$getNavTree());
  };

  /**
   * Returns a reference to the list containing the histories marked for maintenance
   * @private
   * @returns {jQuery}
   */
  DatabaseMaintenance.prototype.$getList = function () {
    return this.jq().find(HISTORY_LIST_SELECTOR);
  };

  /**
   * Returns a reference to the CompositeAbsTimeEditor for selecting dates
   * @private
   * @returns {CompositeAbsTimeEditor}
   */
  DatabaseMaintenance.prototype.$getDateEditor = function () {
    return this.jq().find(DATE_EDITOR_SELECTOR).data('widget');
  };

  /**
   * Returns a reference to the CommandButton editor for running maintenance
   * @private
   * @returns {CommandButton}
   */
  DatabaseMaintenance.prototype.$getRunCommandButton = function () {
    return this.jq().find(RUN_COMMAND_BUTTON_SELECTOR).data('widget');
  };

  /**
   * Returns a reference to the Command for selecting all histories in the dataase*
   * @private
   * @returns {Command}
   */
  DatabaseMaintenance.prototype.$getSelectAllCommand = function () {
    return this.getCommandGroup().get(0);
  };

  /**
   * Returns a reference to the Command for deselecting all histories from the selection list
   * @private
   * @returns {Command}
   */
  DatabaseMaintenance.prototype.$getDeselectAllCommand = function () {
    return this.getCommandGroup().get(1);
  };

  /**
   * Returns the value for the currently selected maintenance operation
   * @private
   * @returns {String} Value for current maintenance operation
   */
  DatabaseMaintenance.prototype.$getSelectedOperation = function () {
    return this.jq().find('input[type=radio]:checked').val();
  };

  /**
   * Selects the maintenance operation to run
   * @private
   * @param operation One of 'clearOld', 'clearAll', or 'deleteAll'
   */
  DatabaseMaintenance.prototype.$selectOperation = function (operation) {
    this.jq().find('input[type=radio]:checked').prop('checked', false);
    this.jq().find('input[type=radio][value=' + operation + ']').prop('checked', true);
  };

  /**
   * Returns an arrray of baja.Ord for the currently selected histories marked for maintenance
   * @private
   * @returns {Array} Array of baja.Ord
   */
  DatabaseMaintenance.prototype.$getSelectedOrds = function () {
    return getSelectedOrds(this.$getList());
  };

  return DatabaseMaintenance;
});
