/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/email/rc/mgr/EmailAccountManager
 */
define(['baja!', 'baja!email:EmailAccount,email:IncomingAccount,email:OutgoingAccount', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/Manager', 'nmodule/email/rc/mgr/EmailAccountManagerModel', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn', 'nmodule/email/rc/mgr/EmailAuthenticationColumn', 'nmodule/email/rc/mgr/EmailProtocolColumn', 'nmodule/email/rc/mgr/EmailPasswordColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mgr/commands/NewCommand', 'nmodule/webEditors/rc/wb/mgr/commands/EditCommand', 'nmodule/webEditors/rc/wb/mgr/commands/MgrDuplicateCommand', 'nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/mgr/componentStatusUtils'], function (baja, types, Promise, _, mgrUtils, Manager, EmailAccountManagerModel, MgrTypeInfo, ContainerComponentSource, IconMgrColumn, NameMgrColumn, PropertyMgrColumn, EmailAuthenticationColumn, EmailProtocolColumn, EmailPasswordColumn, Column, DepthSubscriber, NewCommand, EditCommand, DuplicateCommand, RemoveCommand, addContextMenuSupport, componentStatusUtils) {
  'use strict';

  var IN_ACCOUNT_TYPE = baja.lt("email:IncomingAccount"),
      OUT_ACCOUNT_TYPE = baja.lt("email:OutgoingAccount"),
      INBOX_ICON = baja.Icon.make('module://icons/x16/inbox.png'),
      OUTBOX_ICON = baja.Icon.make('module://icons/x16/outbox.png'),
      EDITABLE = Column.flags.EDITABLE,
      UNSEEN = Column.flags.UNSEEN,
      READONLY = Column.flags.READONLY,
      EDITABLE_UNSEEN = EDITABLE | UNSEEN;

  function makeColumns() {
    function toColumn(slot, params) {
      params = params || {};
      var Ctor = params.ctor || PropertyMgrColumn,
          flags = params.flags || EDITABLE,
          type = params.type || IN_ACCOUNT_TYPE;
      return new Ctor(slot, {
        type: type,
        flags: flags
      });
    } //READONLY


    var icon = new IconMgrColumn(),
        status = toColumn('status', {
      flags: READONLY
    }),
        //EDITABLE
    name = new NameMgrColumn({
      flags: EDITABLE
    }),
        hostName = toColumn('hostname'),
        account = toColumn('account'),
        pollrate = toColumn('pollrate'),
        transport = toColumn('transport', {
      ctor: EmailProtocolColumn,
      type: OUT_ACCOUNT_TYPE
    }),
        auth = toColumn('useAuthentication', {
      ctor: EmailAuthenticationColumn,
      type: OUT_ACCOUNT_TYPE
    }),
        //EDITABLE + UNSEEN
    enabled = toColumn('enabled', {
      flags: EDITABLE_UNSEEN
    }),
        port = toColumn('port', {
      flags: EDITABLE_UNSEEN
    }),
        password = toColumn('password', {
      ctor: EmailPasswordColumn,
      flags: EDITABLE_UNSEEN
    }),
        useSsl = toColumn('useSsl', {
      flags: EDITABLE_UNSEEN
    });

    icon.getValueFor = function (row) {
      var subject = row.getSubject();
      return baja.hasType(subject, IN_ACCOUNT_TYPE) ? INBOX_ICON : baja.hasType(subject, OUT_ACCOUNT_TYPE) ? OUTBOX_ICON : row.getIcon();
    };

    return [icon, name, enabled, transport, hostName, port, auth, account, password, pollrate, status, useSsl];
  }
  /**
   * EmailAccountManager
   * @class
   * @extends module:nmodule/webEditors/rc/wb/mgr/Manager
   * @alias module:nmodule/history/rc/EmailAccountManager
   */


  var EmailAccountManager = function EmailAccountManager(params) {
    Manager.call(this, _.extend({
      moduleName: 'email',
      keyName: 'EmailAccountUxManager'
    }, params));
    this.getCommandGroup().add(new NewCommand(this), new EditCommand(this), new DuplicateCommand(this), new RemoveCommand(this));
    this.$sub = new DepthSubscriber(2);
    addContextMenuSupport(this);
  }; //extend and set up prototype chain


  EmailAccountManager.prototype = Object.create(Manager.prototype);
  EmailAccountManager.prototype.constructor = EmailAccountManager;
  /**
   * Returns a reference to the Edit command
   * @private
   * @returns {Command}
   */

  EmailAccountManager.prototype.$getEditCommand = function () {
    return this.getCommandGroup().get(1);
  };
  /**
   * Returns a reference to the Duplicate command
   * @private
   * @returns {Command}
   */


  EmailAccountManager.prototype.$getDuplicateCommand = function () {
    return this.getCommandGroup().get(2);
  };
  /**
   * Returns a reference to the Remove command
   * @private
   * @returns {Command}
   */


  EmailAccountManager.prototype.$getRemoveCommand = function () {
    return this.getCommandGroup().get(3);
  };
  /**
   * Initial DOM setup
   *
   * @param {jQuery} dom the DOM element into which to load this View
   * @return {Promise}
   */


  EmailAccountManager.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('dblclick', '.mainTable tr', function () {
      that.$getEditCommand().invoke();
    });
    return Manager.prototype.doInitialize.call(this, dom);
  };
  /**
   * Create a MgrModel instance from a email:EmailService component source
   * @private
   * @param emailService
   */


  EmailAccountManager.prototype.makeModel = function (emailService) {
    return MgrTypeInfo.make(['email:IncomingAccount', 'email:OutgoingAccount']).then(function (newTypes) {
      return new EmailAccountManagerModel({
        componentSource: new ContainerComponentSource({
          container: emailService,
          filter: ['email:IncomingAccount', 'email:OutgoingAccount']
        }),
        newTypes: newTypes,
        columns: makeColumns()
      });
    });
  };
  /**
   * Override load() method from parent to map email:EmailService
   * load value to a MgrModel instance
   * @param emailService
   * @returns {Promise}
   */


  EmailAccountManager.prototype.doLoad = function (emailService) {
    var that = this;
    return that.$sub.subscribe(emailService).then(function () {
      return Manager.prototype.doLoad.call(that, emailService);
    }).then(function () {
      var selection = mgrUtils.getMainTableSelection(that),
          editCmd = that.$getEditCommand(),
          dupCmd = that.$getDuplicateCommand();
      dupCmd.setComponent(emailService);
      selection.on('changed', function () {
        editCmd.setEnabled(!selection.isEmpty());
        dupCmd.setEnabled(!selection.isEmpty());
      });
    });
  };
  /**
   * Override to update manager row color based on the corresponding email account's status
   * @param row
   * @param dom
   * @returns {jQuery} dom element with updated status coloring
   */


  EmailAccountManager.prototype.finishMainTableRow = function (row, dom) {
    var account = row.getSubject(),
        status = baja.Status.getStatusFromIStatus(account);
    componentStatusUtils.addComponentStatusCss(status, dom);
    return dom;
  };
  /**
   * Destroy the main view table / unsubscribe
   * @returns {Promise}
   */


  EmailAccountManager.prototype.doDestroy = function () {
    return Promise.join(this.$sub.unsubscribeAll(), Manager.prototype.doDestroy.apply(this, arguments));
  }; ////////////////////////////////////////////////////////////
  //Context Menu Support
  ////////////////////////////////////////////////////////////

  /**
   * Callback from CommandGroupContextMenu. Trigger context menus specifically on table row right clicks
   * @override
   * @returns {string}
   */


  EmailAccountManager.prototype.getContextMenuSelector = function () {
    return 'tr';
  };
  /**
   * Callback from ContextMenuSupport to provide a chance to pre-process
   * the right click menu's CommandGroup prior to displaying it
   *
   * @param {module:bajaux/commands/CommandGroup} group default CommandGroup
   * consisting of subject's menu agents
   * @returns {module:bajaux/commands/CommandGroup} updated CommandGroup
   */


  EmailAccountManager.prototype.updateMenuCommandGroup = function (group) {
    if (group) {
      group.add(this.$getDuplicateCommand(), this.$getEditCommand());
    }

    return group;
  }; /////////////////////////////////////////////////////////////////


  return EmailAccountManager;
});
