/**
 * @copyright 2016, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 * @since Niagara 4.2
 */

/**
 * API Status: **Private**
 * @module nmodule/email/rc/fe/EmailAddressListEditor
 */
define(['baja!', 'baja!email:EmailAddressList,email:EmailAddress', 'lex!email', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'bajaux/util/CommandButtonGroup', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'nmodule/webEditors/rc/fe/fe', 'bajaux/events', 'hbs!nmodule/email/rc/fe/templates/EmailAddressListEditor', 'css!nmodule/email/rc/email'], function (baja, types, lexs, $, Promise, _, BaseEditor, CommandButtonGroup, Command, CommandGroup, fe, events, template) {
  'use strict';

  var lex = lexs[0],
      ADD_CMD_DESCRIPTION = lex.get('emailAddressList.commands.add.description'),
      DELETE_CMD_DESCRIPTION = lex.get('emailAddressList.commands.delete.description'),
      ADD_CMD_ICON = 'module://icons/x16/add.png',
      DELETE_CMD_ICON = 'module://icons/x16/delete.png',
      ADD_CMD_INDEX = 0,
      DELETE_CMD_INDEX = 1,
      DEFAULT_ADDRESS = baja.$('email:EmailAddress'),
      DEFAULT_ADDRESS_LIST = baja.$('email:EmailAddressList'),
      EMAIL_ADDRESSES_CLASS = "js-emailAddresses",
      EMAIL_ADDRESS_EDITOR_CLASS = 'js-emailAddress',
      CMD_GROUP_EDITOR_CLASS = "js-cmdGroup",
      EMAIL_ADDRESSES_SELECTOR = '.' + EMAIL_ADDRESSES_CLASS,
      EMAIL_ADDRESS_EDITOR_SELECTOR = '.' + EMAIL_ADDRESS_EDITOR_CLASS,
      CMD_GROUP_EDITOR_SELECTOR = '.' + CMD_GROUP_EDITOR_CLASS,
      EMAIL_ADDRESS_ROWS_SELECTOR = EMAIL_ADDRESSES_SELECTOR + ' tr',
      DESTROY_EVENT = events.DESTROY_EVENT,
      INITIALIZE_EVENT = events.INITIALIZE_EVENT,
      LOAD_EVENT = events.LOAD_EVENT,
      ENABLE_EVENT = events.ENABLE_EVENT,
      DISABLE_EVENT = events.DISABLE_EVENT,
      MODIFY_EVENT = events.MODIFY_EVENT,
      READONLY_EVENT = events.READONLY_EVENT,
      WRITABLE_EVENT = events.WRITABLE_EVENT,
      EVENT_SELECTORS = '.editor,' + CMD_GROUP_EDITOR_SELECTOR,
      EVENTS = [DESTROY_EVENT, INITIALIZE_EVENT, LOAD_EVENT, ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' ');

  function initCommands(editor) {
    return fe.buildFor({
      dom: $(CMD_GROUP_EDITOR_SELECTOR, editor.jq()),
      type: CommandButtonGroup,
      value: new CommandGroup({
        commands: [makeAddCommand(editor), makeDeleteCommand(editor)]
      }),
      properties: {
        toolbar: true
      },
      formFactor: 'mini'
    });
  }

  function makeAddCommand(editor) {
    return new Command({
      description: ADD_CMD_DESCRIPTION,
      icon: ADD_CMD_ICON,
      func: function func() {
        return addAddress(editor).then(function () {
          editor.$getDeleteCommand().setEnabled(true);
        });
      }
    });
  }

  function makeDeleteCommand(editor) {
    return new Command({
      description: DELETE_CMD_DESCRIPTION,
      icon: DELETE_CMD_ICON,
      enabled: false,
      func: function func() {
        var editors = editor.$getEmailAddressEditors(),
            count = editors.length;

        if (count-- > 1) {
          deleteAddress(editor);
          this.setEnabled(count > 1);
        }
      }
    });
  }

  function addAddresses(editor, addresses) {
    clearAddresses(editor);
    addresses = addresses.length ? addresses : [baja.$('email:EmailAddress')];
    return Promise.all(_.map(addresses, _.partial(addAddress, editor)));
  }

  function addAddress(editor, address) {
    var container = $(EMAIL_ADDRESSES_SELECTOR, editor.jq()),
        dom = $('<td></td>').addClass(EMAIL_ADDRESS_EDITOR_CLASS);
    return fe.buildFor({
      dom: dom,
      value: address || baja.$('email:EmailAddress'),
      formFactor: 'mini'
    }).then(function () {
      $('<tr></tr>').append(dom).appendTo(container);
    });
  }

  function clearAddresses(editor) {
    $(EMAIL_ADDRESSES_SELECTOR, editor.jq()).empty();
  }

  function deleteAddress(editor) {
    $(EMAIL_ADDRESS_ROWS_SELECTOR, editor.jq()).last().remove();
  }
  /**
   * EmailAddressListEditor is a field editor for an email:EmailAddressList type
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/email/rc/fe/EmailAddressListEditor
   */


  var EmailAddressListEditor = function EmailAddressListEditor() {
    BaseEditor.apply(this, arguments);
  }; //extend and set up prototype chain


  EmailAddressListEditor.prototype = Object.create(BaseEditor.prototype);
  EmailAddressListEditor.prototype.constructor = EmailAddressListEditor;
  /**
   * Returns an Array of the current EmailAddressEditor field editors
   * @private
   * @see module:nmodule/email/rc/fe/EmailAddressEditor
   * @returns {Array}
   */

  EmailAddressListEditor.prototype.$getEmailAddressEditors = function () {
    return $(EMAIL_ADDRESS_EDITOR_SELECTOR, this.jq()).map(function () {
      return $(this).data('widget');
    }).get();
  };
  /**
   * Returns the CommandButtonGroup containing the Add and Delete commands
   * @private
   * @returns {CommandButtonGroup}
   */


  EmailAddressListEditor.prototype.$getCommandButtonGroup = function () {
    return $(CMD_GROUP_EDITOR_SELECTOR, this.jq()).data('widget');
  };
  /**
   * Returns a reference to the Add command
   * @private
   * @returns {Command}
   */


  EmailAddressListEditor.prototype.$getAddCommand = function () {
    return this.$getCommandButtonGroup().value().get(ADD_CMD_INDEX);
  };
  /**
   * Returns a reference to the Delete command
   * @private
   * @returns {Command}
   */


  EmailAddressListEditor.prototype.$getDeleteCommand = function () {
    return this.$getCommandButtonGroup().value().get(DELETE_CMD_INDEX);
  };
  /**
   * Initializes the editor DOM
   *
   * @param {jQuery} dom element in which the widget will be initialized
   * @returns {Promise} Promise resolved when editor is initialized
   */


  EmailAddressListEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('EmailAddressListEditor').html(template({
      emailAddresses: EMAIL_ADDRESSES_CLASS,
      cmdGroup: CMD_GROUP_EDITOR_CLASS
    })).on(EVENTS, EVENT_SELECTORS, false).on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    return initCommands(this);
  };
  /**
   * Loads the editor with a value of type 'email:EmailAddressList
   *
   * @param addressList instance of type email:EmailAddressList to load
   * @returns {Promise} Promise resolved when editor is loaded or rejected when value
   * of an incorrect type is loaded
   */


  EmailAddressListEditor.prototype.doLoad = function (addressList) {
    var encodedAddresses, addresses;

    if (!baja.hasType(addressList, 'email:EmailAddressList')) {
      return Promise.reject(new Error('Expected type email:EmailAddressList'));
    }

    encodedAddresses = addressList.encodeToString().split(';');
    addresses = _.map(encodedAddresses, function (encodedAddress) {
      return DEFAULT_ADDRESS.decodeFromString(encodedAddress);
    });
    return addAddresses(this, addresses);
  };
  /**
   * Reads the current value of the editor
   *
   * @returns {Promise} promise to be resolved with a value of type email:EmailAddressList
   */


  EmailAddressListEditor.prototype.doRead = function () {
    var addressEditors = this.$getEmailAddressEditors();
    return Promise.all(_.map(addressEditors, function (addressEditor) {
      return addressEditor.read().then(function (address) {
        return address.encodeToString();
      });
    })).then(function (array) {
      return DEFAULT_ADDRESS_LIST.decodeFromString(array.join(';'));
    });
  };
  /**
   * Enables or disables the editor.
   *
   * @param {Boolean} enabled Flag indicating the enabled/disabled status
   * @returns {Promise} Promise resolved when editor has been enabled/disabled
   */


  EmailAddressListEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };
  /**
   * Set the editor's readonly status
   *
   * @param {Boolean} readonly Flag indicating readonly status
   * @returns {Promise} Promise resolved when editor has been marked readable/writable
   */


  EmailAddressListEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };
  /**
   * Destroys the editor
   *
   * @returns {Promise} Promise resolved when editor has been destroyed
   */


  EmailAddressListEditor.prototype.doDestroy = function () {
    this.jq().removeClass("EmailAddressListEditor");
    return this.getChildWidgets().destroyAll();
  };

  return EmailAddressListEditor;
});
