/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * API Status: **Private**
 * @module nmodule/saml/rc/util/UserCOTPrototypesEditor
 */
define(['baja!', 'bajaux/Widget', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'nmodule/webEditors/rc/fe/baja/util/subscriberUtils', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'css!nmodule/saml/rc/saml'], function (baja, Widget, $, Promise, _, CompositeEditor, spaceUtils, subscriberUtils, CompositeBuilder) {
  'use strict';

  var resolveService = spaceUtils.resolveService,
      leaseWhenNeeded = subscriberUtils.leaseWhenNeeded,
      getCircleOfTrustFolder = leaseWhenNeeded(function (base) {
    return resolveService('saml:SAMLIdPService', base).then(function (service) {
      return service.get('circleOfTrustFolder');
    });
  });
  /**
   * Editor for editing available circle of trust prototypes on a user.
   *
   * @class
   * @alias module:nmodule/saml/rc/util/UserCOTPrototypesEditor
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */

  var UserCOTPrototypesEditor = function UserCOTPrototypesEditor(params) {
    CompositeEditor.apply(this, arguments);
  };

  UserCOTPrototypesEditor.prototype = Object.create(CompositeEditor.prototype);
  UserCOTPrototypesEditor.prototype.constructor = UserCOTPrototypesEditor;
  /**
   * Resolve the prototypes that are available to add to a user.
   *
   * @private
   * @returns {Array.<String>|Promise.<Array.<String>>} promise to be resolved with an array
   * of prototype names
   */

  UserCOTPrototypesEditor.prototype.$getAvailablePrototypes = function (currentSlotName) {
    return this.$getPrototypesFromCurrentCircleOfTrust(currentSlotName);
  };
  /**
   * Get the prototypes from the parent Circle of Trust's properties.
   * @private
   * @returns {Array.<String>|Promise.<Array.<String>>} promise to be resolved with an array of Prototype
   * values under the Circle of Trust. The array is empty in case there is no prototype
   * or no circle of trust available.
   */


  UserCOTPrototypesEditor.prototype.$getPrototypesFromCurrentCircleOfTrust = function (currentSlotName) {
    return this.getOrdBase().then(getCircleOfTrustFolder).then(function (folder) {
      var cot = folder.get(currentSlotName);

      if (cot && baja.hasType(cot, 'saml:CircleOfTrust')) {
        return cot.lease().then(function () {
          return cot.getPrototypes().lease().then(function () {
            return cot.getPrototypes().getSlots().toValueArray();
          });
        });
      } else {
        return [];
      }
    });
  };
  /**
   * Get the circle of trust HTML container element for this editor.
   *
   * @private
   * @returns {JQuery}
   */


  UserCOTPrototypesEditor.prototype.$getCircleOfTrustContainerElement = function () {
    return this.jq().children('.circleOfTrustContainer');
  };
  /**
   * Add "UserCOTPrototypesEditor" class.
   *
   * @param {JQuery} dom
   */


  UserCOTPrototypesEditor.prototype.doInitialize = function (dom) {
    dom.addClass('UserCOTPrototypesEditor');
    dom.append('<div class="circleOfTrustContainer"/>');
  };
  /**
   * The builder for a `UserCOTPrototypesEditor` will create either
   * 1) When editing is allowed:  a boolean editor for each available prototype.
   * 2) When readonly:  a readonly string editor for each of the currently selected prototypes.
   *
   * @returns {CompositeBuilder}
   */


  UserCOTPrototypesEditor.prototype.makeBuilder = function () {
    if (this.isReadonly()) {
      return this.$makeReadonlyBuilder();
    } else {
      return this.$makeEditingBuilder();
    }
  };
  /**
   * The builder for an editing `UserCOTPrototypesEditor` will create a boolean editor for each
   * available prototype.
   *
   * @private
   * @returns {CompositeBuilder}
   */


  UserCOTPrototypesEditor.prototype.$makeEditingBuilder = function () {
    var that = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      var slot = that.getSlot();

      if (!slot) {
        return [];
      }

      return that.$getAvailablePrototypes(String(slot));
    };

    builder.getValueFor = function (key) {
      return _.contains(this.getDataSource().split(','), key);
    };

    builder.getConfigFor = function (key) {
      return {
        properties: {
          trueText: key,
          falseText: key
        },
        formFactor: 'mini'
      };
    };

    builder.getDomFor = function () {
      return $('<div/>').appendTo(that.$getCircleOfTrustContainerElement());
    };

    return builder;
  };
  /**
   * The builder for a readonly `UserCOTPrototypesEditor` will create string editors
   * displaying each of the currently selected prototypes.
   *
   * @private
   * @returns {CompositeBuilder}
   */


  UserCOTPrototypesEditor.prototype.$makeReadonlyBuilder = function () {
    var that = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return this.getDataSource().split(',');
    };

    builder.getValueFor = function (key) {
      return key;
    };

    builder.getConfigFor = function () {
      return {
        formFactor: 'mini'
      };
    };

    builder.getDomFor = function () {
      return $('<div/>').appendTo(that.$getCircleOfTrustContainerElement());
    };

    return builder;
  };
  /**
   * Read prototypes that are selected as a comma separated list.
   * 1) When editing is allowed:  Read prototypes that are selected from boolean editors, and join into a string.
   * 2) When readonly:  Read the original dataSource values joined into a string.
   *
   * @returns {Promise} promise to be resolved with a comma-separated
   * string containing what prototypes are selected
   */


  UserCOTPrototypesEditor.prototype.doRead = function () {
    var builder = this.getBuilder(),
        readonly = this.isReadonly();
    return Promise.join(builder.getKeys(), builder.readAll()).spread(function (prototypeNames, values) {
      if (readonly) {
        return values.join(',');
      } else {
        return _.filter(prototypeNames, function (prototypeName, i) {
          return values[i];
        }).join(',');
      }
    });
  };
  /**
   * Remove "UserCOTPrototypesEditor" class.
   *
   * @returns {Promise}
   */


  UserCOTPrototypesEditor.prototype.doDestroy = function () {
    this.jq().removeClass('UserCOTPrototypesEditor');
    return CompositeEditor.prototype.doDestroy.apply(this, arguments);
  };

  return UserCOTPrototypesEditor;
});
