/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/util/role/PermissionsMapMgrColumn
 */
define(['baja!', 'baja!baja:AdminRole,baja:Role,baja:PermissionsMap,baja:CategoryService', 'jquery', 'Promise', 'underscore', 'dialogs', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'lex!wbutil,webEditors'], function (baja, types, $, Promise, _, dialogs, htmlUtils, PropertyMgrColumn, MgrColumn, lexs) {
  'use strict';

  var wbutilLex = lexs[0],
      webEditorsLex = lexs[1],
      PERMISSIONS_MAP = 'baja:PermissionsMap',
      SUPER_USER = webEditorsLex.get('superUser'),
      COLUMN_NAME = wbutilLex.get('permission.permissions'),
      DEFAULT = wbutilLex.get('roleManager.noPermissions'),
      OBJECT_ICON = '/module/icons/x16/object.png';
  /**
   * Column for showing a role component's display name.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/util/role/PermissionsMapMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn
   */

  var PermissionsMapMgrColumn = function PermissionsMapMgrColumn(params) {
    var that = this;
    that.$categories = params && params.categories || [];
    PropertyMgrColumn.call(that, 'permissions', $.extend({}, params, {
      type: PERMISSIONS_MAP
    }));
  };

  PermissionsMapMgrColumn.prototype = Object.create(PropertyMgrColumn.prototype);
  PermissionsMapMgrColumn.prototype.constructor = PermissionsMapMgrColumn;
  /**
   * Returns `permission.permissions` from wbutil lexicon.
   *
   * @returns {String}
   */

  PermissionsMapMgrColumn.prototype.toDisplayName = function () {
    return COLUMN_NAME;
  };
  /**
   * Shows a display string representing the permissionsMap for this Role.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom
   * @throws {Error} if the row's value is not a baja:PermissionsMap
   */


  PermissionsMapMgrColumn.prototype.buildCell = function (row, dom) {
    var permissionsMap = this.getProposedValueFor(row);

    if (!baja.hasType(permissionsMap, PERMISSIONS_MAP)) {
      throw new Error('PermissionsMap required, got: ' + permissionsMap.getType());
    }

    dom.text(this.$getPermissionsDisplayString(permissionsMap));
  };

  PermissionsMapMgrColumn.prototype.$getCategoryName = function (index) {
    var that = this,
        category = _.find(that.$categories, function (cat) {
      return index === cat.getIndex();
    });

    if (category) {
      return category.getDisplayName();
    }

    return wbutilLex.get('category.generic', [index]);
  };

  PermissionsMapMgrColumn.prototype.$getPermissionsDisplayString = function (permissionsMap) {
    var that = this,
        i,
        p,
        displayString = ''; // SUPER_USER & DEFAULT are special cases

    if (permissionsMap === baja.PermissionsMap.SUPER_USER) {
      return SUPER_USER;
    }

    if (permissionsMap === baja.PermissionsMap.DEFAULT) {
      return DEFAULT;
    } // encode into 'categoryName=permissions;' pairs


    for (i = 1; i < permissionsMap.size(); i++) {
      p = permissionsMap.getPermissions(i); // don't encode for this index if there's no baja.Permissions, or if it's bajaPermissions.none (ie 0)

      if (!p || !p.getMask()) {
        continue;
      }

      if (displayString.length > 0) {
        displayString += '; ';
      }

      displayString += that.$getCategoryName(i) + '=' + p.encodeToString();
    }

    return displayString;
  };
  /**
   * Shows a dialog to confirm any permissions changes.
   *
   * @param {*} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} [params.editor]
   * the editor from which the value was read. If the column is not editable,
   * this parameter will be `undefined`, as no editor will have been created for
   * the value. This situation may occur when a value obtained via discovery is
   * set on row for a non-editable column.
   * @param {baja.comm.Batch} [params.batch] a batch to use to commit changes
   * up to the station
   * @param {Function} [params.progressCallback] call this with
   * `MgrColumn.COMMIT_READY` when this function is done adding network calls to
   * the batch.
   * @returns {Promise}
   */


  PermissionsMapMgrColumn.prototype.commit = function (value, row, params) {
    var that = this,
        role = row.getSubject(),
        originalPermissions = role.get('permissions'),
        newPermissions = that.getProposedValueFor(row); // A new row, no confirmation required, do commit.

    if (!role.isMounted()) {
      return PropertyMgrColumn.prototype.commit.call(that, value, row, params);
    } // Permissions are unchanged, no confirmation required, no commit.


    if (originalPermissions === newPermissions) {
      return Promise.resolve();
    } // eslint-disable-next-line promise/avoid-new


    return new Promise(function (resolve, reject) {
      dialogs.showOkCancel({
        title: wbutilLex.get('roleManager.suConfirmTitle'),
        content: htmlUtils.escapeHtml(wbutilLex.get({
          key: 'roleManager.suConfirmText',
          args: [role.getDisplayName(), that.$getPermissionsDisplayString(originalPermissions), that.$getPermissionsDisplayString(newPermissions)]
        })).replace(/\n/g, '<br/>'),
        ok: function ok() {
          resolve(PropertyMgrColumn.prototype.commit.call(that, value, row, params));
        },
        cancel: function cancel() {// Never resolve the promise. This will drop the user back at the edit screen without
          // allowing the batch to be committed.
        }
      });
    });
  };
  /**
   * Return the icon to be used for the column in the batch component editor.
   * @returns {String}
   */


  PermissionsMapMgrColumn.prototype.getColumnIcon = function () {
    return OBJECT_ICON;
  };

  return PermissionsMapMgrColumn;
});
