/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/util/AuthenticationSchemeEditor
 */
define(['baja!', 'baja!baja:AuthenticationScheme', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/OrderedMapEditor', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'nmodule/webEditors/rc/fe/baja/util/subscriberUtils'], function (baja, types, $, _, fe, BaseEditor, OrderedMapEditor, spaceUtils, subscriberUtils) {
  'use strict';

  var resolveService = spaceUtils.resolveService,
      leaseWhenNeeded = subscriberUtils.leaseWhenNeeded,
      getAuthenticationFolder = leaseWhenNeeded(function (base) {
    return resolveService('baja:AuthenticationService', base).then(function (service) {
      return service.get('authenticationSchemes');
    });
  });

  function toSlotName(slot) {
    return slot.getName();
  }

  function toMap(schemes) {
    var map = new baja.OrderedMap();

    _.each(schemes, function (scheme) {
      map.put(scheme, scheme);
    });

    return map;
  }
  /**
   * Field editor for changing the selected authentication scheme name on a
   * `baja:User`.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/util/AuthenticationSchemeEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */


  var AuthenticationSchemeEditor = function AuthenticationSchemeEditor() {
    BaseEditor.apply(this, arguments);
  };

  AuthenticationSchemeEditor.prototype = Object.create(BaseEditor.prototype);
  AuthenticationSchemeEditor.prototype.constructor = AuthenticationSchemeEditor;
  /**
   * Resolve the `AuthenticationSchemeFolder` under the `AuthenticationService`
   * and resolve an array of the slot names of all `AuthenticationSchemes`
   * found therein.
   *
   * @private
   * @returns {Promise}
   */

  AuthenticationSchemeEditor.prototype.$getAvailableSchemeNames = function () {
    return this.getOrdBase().then(getAuthenticationFolder).then(function (schemes) {
      var slots = schemes.getSlots().is('baja:AuthenticationScheme').toArray();
      return _.map(slots, toSlotName);
    })["catch"](_.constant(null));
  };
  /**
   * Get the `OrderedMapEditor` from which the user will select a scheme name.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */


  AuthenticationSchemeEditor.prototype.$getOrderedMapEditor = function () {
    return this.jq().children('.map').data('widget');
  };
  /**
   * Resolve all available `AuthenticationScheme` names and render them into
   * an `OrderedMapEditor`.
   *
   * @param {jQuery} dom
   * @returns {Promise} promise to be resolved after the
   * `OrderedMapEditor` is built, or rejected if authentication scheme names
   * could not be resolved from the station
   */


  AuthenticationSchemeEditor.prototype.doInitialize = function (dom) {
    var that = this;
    return that.$getAvailableSchemeNames().then(function (schemes) {
      return fe.buildFor({
        dom: $('<div class="map"/>').appendTo(dom),
        type: OrderedMapEditor,
        value: toMap(schemes)
      });
    });
  };
  /**
   * Set the selected scheme name in the `OrderedMapEditor`. Note that if the
   * loaded map does not contain this scheme name, it will not be selected -
   * only valid scheme names can be chosen.
   *
   * @param {String} value
   */


  AuthenticationSchemeEditor.prototype.doLoad = function (value) {
    var mapEd = this.$getOrderedMapEditor(),
        map = mapEd.value(); //if no scheme names were loaded, just show a one-item dropdown

    if (!map.getKeys().length) {
      map = new baja.OrderedMap();
      map.put(value, value);
      return mapEd.load(map);
    }

    this.$getOrderedMapEditor().setSelectedValue(value, true);
  };
  /**
   * Read the currently selected scheme name.
   *
   * @returns {Promise} promise to be resolved with the selected
   * scheme name
   */


  AuthenticationSchemeEditor.prototype.doRead = function () {
    return this.$getOrderedMapEditor().read();
  };
  /**
   * Enable/disable the dropdown.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */


  AuthenticationSchemeEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };
  /**
   * Readonly/writable the dropdown.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */


  AuthenticationSchemeEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  return AuthenticationSchemeEditor;
});
