/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson.
 */
define(["baja!", "baja!history:FoxHistory", "lex!webEditors", "Promise", "underscore", "nmodule/webEditors/rc/wb/tree/TreeNode", "nmodule/webEditors/rc/wb/tree/BajaNavTreeNode", "nmodule/webEditors/rc/wb/tree/BajaNavFileTreeNode", "nmodule/webEditors/rc/fe/fe", "nmodule/webEditors/rc/wb/tree/NavTree", "baja!hierarchy:LevelElem"], function (baja, types, lexs, Promise, _, TreeNode, BajaNavTreeNode, BajaNavFileTreeNode, fe, NavTree) {
  "use strict";

  var webEditorsLex = lexs[0];

  function fromWebEditorsLex(key) {
    return function () {
      return Promise.resolve(webEditorsLex.get(key));
    };
  } ////////////////////////////////////////////////////////////////
  // First-level nodes
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   *
   * Used for building a complete Station tree.
   *
   * The history space part of the tree isn't a full tree implementation.
   * This will require more work in the future but is perfectly adequate
   * for now.
   *
   * @exports nmodule/webEditors/rc/wb/tree/stationTree
   */


  var exports = {};
  /**
   * The name of the top-level session node generated by this module. In
   * general, this is meaningless and would be stripped out of any ORD read
   * from the session node.
   *
   * @type {string}
   */

  var SESSION_NODE_NAME = exports.SESSION_NODE_NAME = "local";
  /**
   * Makes the file nav container.
   *
   * @inner
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode} The tree node for the file space.
   */

  exports.makeFileSpaceNode = function () {
    return new baja.FileNavContainer().init().then(function (fileNavContainer) {
      var node = new BajaNavTreeNode(fileNavContainer);
      node.toDisplay = fromWebEditorsLex("stationTree.files.displayName");
      return node;
    })["catch"](_.constant(null));
  };
  /**
   * Resolves the history space and creates a tree node for it.
   *
   * @inner
   * @returns {Promise} promise to be resolved with a `TreeNode` for the
   * history space, or null if history space could not be resolved
   */


  exports.makeHistorySpaceNode = function () {
    return baja.Ord.make("history:").get().then(function (historySpace) {
      var node = new BajaNavTreeNode(historySpace);
      node.toDisplay = fromWebEditorsLex("stationTree.histories.displayName");
      return node;
    })["catch"](_.constant(null));
  };
  /**
   * Resolves the component space and creates a tree node for it.
   *
   * @inner
   * @returns {Promise} promise to be resolved with a `TreeNode` for the
   * component space
   */


  exports.makeComponentSpaceNode = function () {
    return baja.Ord.make("station:|slot:/").get({
      lease: true
    }).then(function (station) {
      var node = new BajaNavTreeNode(station); // Change the root Station space to say 'Config'

      node.toDisplay = fromWebEditorsLex("stationTree.config.displayName");
      return node;
    });
  };
  /**
   * Resolves the nav file, then builds the tree node for it.
   *
   * @inner
   * @returns {Promise} promise to be resolved with a `TreeNode` for the
   * user's nav file, or null if the user has no nav file
   */


  exports.makeNavFileNode = function () {
    return baja.nav.navfile.load().then(function () {
      var rootNode = baja.nav.navfile.getRootNode();
      return rootNode && new BajaNavFileTreeNode(rootNode);
    })["catch"](_.constant(null));
  };
  /**
   * The prefix for the top hierarchy nodes.
   *
   * @type {String}
   */


  exports.HIERARCHY_NODE_PREFIX = 'hierarchy-node-';
  /**
   * Resolves the hierarchy space, then builds the tree node for it.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/BajaNavTreeNode|null>} promise
   * to be resolved with the hierarchy space node, or `null` if the station has
   * no hierarchy service installed
   */

  exports.makeHierarchySpaceNode = function () {
    return baja.Ord.make("local:|hierarchy:").get().then(function (hierarchySpace) {
      var node = new BajaNavTreeNode(hierarchySpace);
      node.toDisplay = fromWebEditorsLex("stationTree.hierarchy.displayName");
      return node;
    })["catch"](_.constant(null));
  };
  /**
   * Resolves an array of hierarchy tree nodes.
   *
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/tree/BajaNavTreeNode>|null>} promise
   * to be resolved with the hierarchy tree nodes, or `null` if the station has
   * no hierarchy service installed
   */


  exports.makeHierarchyNodes = function () {
    return baja.Ord.make("local:|hierarchy:").get().then(function (hierarchySpace) {
      return hierarchySpace.getNavChildren();
    }).then(function (kids) {
      return kids.map(function (node) {
        var hierarchyNode = new BajaNavTreeNode(node),
            getName = hierarchyNode.getName; // Replace the 'getName' method for the top level hierarchy nodes so we can use a unique prefix.
        // This stops any name clashes with the other common spaces in the Station. For instance,
        // a Hierarchy with 'Config' won't cause any problems with the 'Config' Station Component Space.

        hierarchyNode.getName = function () {
          return exports.HIERARCHY_NODE_PREFIX + getName.apply(this, arguments);
        };

        return hierarchyNode;
      });
    })["catch"](_.constant(null));
  };
  /**
   * Make all the first-level nodes to go under the root session node in the
   * nav tree.
   *
   * @inner
   * @param {Object} params
    * @returns {Promise} promise to be resolved with an array of TreeNodes
   * to be loaded as children of the root session node
   */


  function makeRootNodes(params) {
    return Promise.all([params.configTree && exports.makeComponentSpaceNode(), params.navFileTree && exports.makeNavFileNode(), params.filesTree && exports.makeFileSpaceNode(), params.historiesTree && exports.makeHistorySpaceNode(), params.hierarchiesTree && exports.makeHierarchyNodes()]).then(function (nodes) {
      // Flatten any arrays one level down then filter out falsies.
      return _.compact(_.flatten(nodes,
      /*shallow*/
      true));
    });
  } ////////////////////////////////////////////////////////////////
  // Top-level nodes
  ////////////////////////////////////////////////////////////////

  /**
   * Create the very top-level session node to load into the nav tree. All
   * child nodes of this top-level node must already be loaded and ready to
   * pass into the constructor.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>} kids
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */


  exports.makeSessionNode = function makeSessionNode(kids) {
    var name = SESSION_NODE_NAME,
        displayName = webEditorsLex.get("stationTree.station.displayName"),
        sessionNode = new TreeNode(name, displayName, kids);

    sessionNode.getIcon = function () {
      return [webEditorsLex.get("stationTree.station.icon")];
    };

    return sessionNode;
  };
  /**
   * Make the Station tree and return it.
   *
   * @param {Object} [params]
   * @param {Boolean} [params.configTree] If true, the Station Component Space is shown.
   * @param {Boolean} [params.navFileTree] If true, the Nav File Tree is shown.
   * @param {Boolean} [params.filesTree] If true, the File System Tree is shown.
   * @param {Boolean} [params.historiesTree] If true, Station's history tree is shown.
   * @param {Boolean} [params.hierarchiesTree] If true, the Station hierarchy tree is shown.
   * @returns {Promise} promise to be resolved with a TreeNode for
   * the root of a Station Tree.
   */


  exports.makeRootNode = function makeRootNode(params) {
    return makeRootNodes(params || {}).then(exports.makeSessionNode);
  };
  /**
   * Create a nav tree that can be used for showing Station tree nodes.
   *
   * If dom (and value) are passed in, the widget will be initialized and loaded
   * as well. If not, the nav tree is just created.
   *
   * @param {Object} params parameters for constructing the nav tree.
   * @param [params.dom] If specified, the DOM used to initialize the nav tree.
   * @param [params.value] If specified, the value to load into the nav tree.
   * @returns {Promise} A promise that resolves to the nav tree.
   */


  exports.createNavTree = function (params) {
    var newParams = _.extend({
      moduleName: 'webEditors',
      keyName: 'navTree',
      type: NavTree,
      properties: {
        hideRoot: true
      }
    }, params),
        make = newParams.dom ? fe.buildFor : fe.makeFor;

    return make.call(fe, newParams);
  };

  return exports;
});
