/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint-env browser */

/*jshint browser: true */

/*global niagara: false */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/tree/NavTree
 */
define(['baja!', 'log!nmodule.webEditors.rc.wb.tree.NavTree', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/dragdrop/dragDropUtils', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/IconEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/util/Switchboard', 'nmodule/webEditors/rc/wb/mixin/TransferSupport', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'hbs!nmodule/webEditors/rc/wb/template/NavTree', 'css!nmodule/webEditors/rc/wb/tree/NavTreeStyle'], function (baja, log, $, Promise, _, events, dragDropUtils, fe, feDialogs, BaseEditor, IconEditor, htmlUtils, Switchboard, TransferSupport, TreeNode, tplNavTree) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
      LOAD_EVENT = events.LOAD_EVENT,
      MODIFY_EVENT = events.MODIFY_EVENT,
      contextMenuOnLongPress = htmlUtils.contextMenuOnLongPress,
      EXPAND_SPINNER_DELAY = 250,
      HOVER_PRELOAD_DELAY = 200,
      logError = log.severe.bind(log);
  /*
   * TODO: maybe be smart about which node is most likely to be visited first?
   * TODO: limit and configure preload depth
   * TODO: turn red or something if kids fail to load
   */

  function isWb() {
    return typeof niagara !== 'undefined' && niagara.env && niagara.env.type === 'wb';
  } //TODO: fix drag highlighting in Workbench (NCCB-9185)
  //purposely break styling in Workbench when dragging, since we never get the
  //dragleave event to remove the styling


  var DROP_TARGET_CLASS = isWb() ? 'wbDropTarget' : 'dropTarget'; ////////////////////////////////////////////////////////////////
  // Helper methods
  ////////////////////////////////////////////////////////////////

  function resolve(val) {
    return Promise.resolve(val);
  }
  /** @param {String} val */


  function reject(val) {
    return Promise.reject(new Error(val));
  }

  function unloadKids(tree) {
    var kidsList = tree.$getKidsList(),
        kidEditors = tree.$getKids();
    return kidEditors.destroyAll().then(function () {
      kidsList.empty();
    });
  }

  function buildKid(tree, kid) {
    return tree.buildChildFor({
      value: kid,
      dom: $('<li/>').data('node', kid).appendTo(tree.$getKidsList()),
      type: tree.constructor,
      //TODO: be smarter about this
      //TODO: these should be in properties or data
      expanded: false,
      loadKids: false,
      enableHoverPreload: tree.$enableHoverPreload,
      displayFilter: tree.$displayFilter
    });
  }

  function loadKids(tree, batch) {
    var node = tree.value();
    return node.getKids({
      batch: batch
    }).then(function (kids) {
      return Promise.all(_.map(kids, function (kid) {
        var displayFilter = tree.getDisplayFilter();

        if (displayFilter) {
          if (displayFilter(node, kid)) {
            return buildKid(tree, kid);
          } else {
            return null;
          }
        } else {
          return buildKid(tree, kid);
        }
      })).then(function (kidTrees) {
        return _.compact(kidTrees);
      });
    });
  }

  function setSelectedEditor(tree, selectedEd) {
    if (tree !== selectedEd) {
      tree.setSelected(false);
    }

    _.each(tree.$getKids(), function (kidEd) {
      setSelectedEditor(kidEd, selectedEd);
    });
  }

  function kidName(kid) {
    return kid.getName();
  }

  function toValue(node) {
    return node.value();
  }

  function isNavNode(value) {
    return baja.hasType(value, 'baja:INavNode');
  }

  function isTreeNode(value) {
    return value instanceof TreeNode;
  }

  function toNavOrd(navNode) {
    return navNode.getNavOrd();
  } ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Allows for displaying, expanding, and collapsing a single node in a tree.
   *
   * A single `NavTree` instance will show a node's icon and display value,
   * an expand/collapse button, and a list containing the node's children.
   *
   * The kid list will initially be collapsed, but when the `NavTree` first
   * loads, it will immediately kick off loading the children's values behind
   * the scenes. This should allow the list to be immediately expanded when the
   * user does get around to clicking the expand button to show the list. (If
   * the list is not finished loading when the user clicks, there will just
   * be a delay until it does finish.)
   *
   * Note that each individual node in the tree will receive its own `NavTree`
   * instance. With some clever overriding of `buildChildFor`, this should allow
   * for some granularity when expanding out your tree. Inline field editors?
   * Who knows.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/tree/NavTree
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @param {Object} [params]
   * @param {Boolean} [params.expanded=false] set to true if this tree should
   * be immediately expanded on first load
   * @param {Boolean} [params.loadKids=true] set to false if this tree should
   * not attempt to load its children before it is expanded
   * @param {Boolean} [params.enableHoverPreload=false] set to true if child
   * nodes should start to preload when the mouse hovers over the expand
   * button (to shave a bit of time off child node expansion)
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter} [params.displayFilter]
   */


  var NavTree = function NavTree(params) {
    //TODO: _.extend on params
    var that = this,
        expanded = params && params.expanded,
        loadKids = params && params.loadKids,
        enableHoverPreload = params && params.enableHoverPreload;
    BaseEditor.apply(that, arguments);
    this.$expanded = expanded;
    this.$loadKids = loadKids !== false;
    this.$selected = false;
    this.$enableHoverPreload = enableHoverPreload;

    if (params && params.displayFilter) {
      that.setDisplayFilter(params.displayFilter);
    }

    new Switchboard(this).allow('$setLoaded').oneAtATime().keyedOn(function (loaded) {
      return loaded;
    }).allow('$addKid').notWhile('$removeKid').keyedOn(kidName).allow('$removeKid').notWhile('$addKid').keyedOn(kidName).allow('$reorderKids').notWhile('$addKid').notWhile('$removeKid');
    TransferSupport(this);
  };

  NavTree.prototype = Object.create(BaseEditor.prototype);
  NavTree.prototype.constructor = NavTree;
  NavTree.ACTIVATED_EVENT = 'navTree:activated';
  NavTree.COLLAPSED_EVENT = 'navTree:collapsed';
  NavTree.DESELECTED_EVENT = 'navTree:deselected';
  NavTree.EXPANDED_EVENT = 'navTree:expanded';
  NavTree.SELECTED_EVENT = 'navTree:selected';
  /**
   * Return the element for the expand/collapse button.
   * @private
   * @returns {jQuery}
   */

  NavTree.prototype.$getButton = function () {
    return this.jq().children('button');
  };
  /**
   * Return the element to load the node's display value.
   * @private
   * @returns {jQuery}
   */


  NavTree.prototype.$getDisplayElement = function () {
    return this.jq().children('.display').children('.displayName');
  };
  /**
   * Return the element to load the node's icon.
   * @private
   * @returns {jQuery}
   */


  NavTree.prototype.$getIconElement = function () {
    return this.jq().children('.display').children('.icon');
  };
  /**
   * Return the <ul> element that holds the list items to hold `NavTree`s for
   * the child nodes.
   * @private
   * @returns {jQuery}
   */


  NavTree.prototype.$getKidsList = function () {
    return this.jq().children('ul');
  };
  /**
   * @private
   * @returns {Array.<module:nmodule/webEditors/rc/wb/tree/NavTree>}
   */


  NavTree.prototype.$getKids = function () {
    return this.getChildEditors({
      dom: this.$getKidsList(),
      type: NavTree
    });
  };
  /**
   * Given a kid node, find the `NavTree` that currently has that node loaded,
   * if any.
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} node
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree}
   */


  NavTree.prototype.$getKidEditorFor = function (node) {
    var kidDoms = this.$getKidsList().children(),
        ed,
        i;

    for (i = 0; i < kidDoms.length; i++) {
      ed = $(kidDoms[i]).data('widget');

      if (ed && ed.value().equals(node)) {
        return ed;
      }
    }
  };
  /**
   * The amount of time to wait before automatically unloading children of
   * a collapsed node.
   * @private
   * @returns {number} milliseconds
   */


  NavTree.prototype.$getUnloadTimeout = function () {
    return 10000;
  };
  /**
   * Return whether the tree is currently in expanded mode or not.
   * @private
   * @returns {Boolean}
   */


  NavTree.prototype.$isExpanded = function () {
    return !!this.$expanded;
  };
  /**
   * Set the expanded status of the nav tree.
   *
   * If expanding, the child nodes of this node will be loaded if they are
   * not already, and then the kid list will be shown. After the kid list is
   * shown, kids of kids will start preloading immediately to speed up
   * subsequent drilldowns.
   *
   * If collapsing, the kid list will simply be hidden. Any kid editors will
   * not necessarily unload just from collapsing the tree.
   *
   * @private
   * @param {Boolean} expanded
   * @returns {Promise} promise to be resolved when the tree has fully
   * expanded or collapsed.
   */


  NavTree.prototype.$setExpanded = function (expanded) {
    var that = this,
        kidsList = that.$getKidsList(),
        button = that.$getButton(),
        prom,
        pending = true,
        spinnerTicket,
        event = NavTree[expanded ? 'EXPANDED_EVENT' : 'COLLAPSED_EVENT'];

    if (expanded === that.$isExpanded()) {
      return resolve();
    }

    that.$expanded = expanded;

    if (expanded) {
      spinnerTicket = setTimeout(function () {
        if (pending) {
          button.addClass('loading');
        }
      }, EXPAND_SPINNER_DELAY);
      prom = that.$setLoaded(true).then(function () {
        kidsList.show();

        if (that.$loadKids) {
          return that.$setKidsLoaded(true);
        }
      })["finally"](function () {
        button.removeClass('loading');
        pending = false;
        clearTimeout(spinnerTicket);
      });
    } else {
      kidsList.hide();
      setSelectedEditor(that, null);
      prom = resolve();
    }

    return prom.then(function () {
      that.trigger(event);
      return that.$updateButton();
    });
  };
  /**
   * Are my current children loaded and ready to expand?
   * @private
   * @returns {Boolean}
   */


  NavTree.prototype.$isLoaded = function () {
    return this.$loaded;
  };
  /**
   * Load up my children so that I can expand without delay. Or, unload them
   * to save memory. Note that if unloading kids, this tree will also be
   * collapsed (an expanded tree with zero children makes no sense). Kids
   * will be automatically re-loaded if the tree is expanded again, but
   * hopefully they will have been loaded well in advance of the expand button
   * being clicked (for quick response).
   *
   * @private
   * @param {Boolean} loaded
   * @param {baja.comm.Batch} [batch]
   * @returns {Promise} promise to be resolved when I have loaded or
   * unloaded all of my kids.
   */


  NavTree.prototype.$setLoaded = function (loaded, batch) {
    var that = this,
        prom; //TODO: this is Switchboard behavior. come back and do it right

    if (loaded === that.$loaded) {
      return that.$$slProm || resolve();
    }

    if (that.$getKidsList().is(':visible')) {
      logError('warning: $setLoaded should never happen when list element ' + 'is visible');
    }

    that.$loaded = loaded;
    prom = that.$$slProm = (loaded ? loadKids(that, batch) : that.collapse().then(function () {
      return unloadKids(that);
    })).then(function () {
      delete that.$$slProm;
    });
    return prom;
  };
  /**
   * Are all of my kids in the loaded state? Or: are my grandkids loaded?
   * @private
   * @returns {Boolean}
   */


  NavTree.prototype.$isKidsLoaded = function () {
    return this.$kidsLoaded;
  };
  /**
   * Set the loaded state of all my children. Or: load or unload my grandkids.
   * This will typically be called when the tree is expanded, to try to keep
   * one step ahead of the user as he or she drills down through the nav tree.
   *
   * @private
   * @param {Boolean} loaded
   * @returns {Promise} promise to be resolved when all of my grandkids
   * are loaded or unloaded
   */


  NavTree.prototype.$setKidsLoaded = function (loaded) {
    var that = this,
        kids = that.$getKids();

    if (that.$kidsLoaded === loaded) {
      return resolve();
    }

    that.$kidsLoaded = loaded;
    var batch = new baja.comm.Batch(),
        prom = Promise.all(_.map(kids, function (kid) {
      return kid.$setLoaded(loaded, batch);
    }));
    batch.commit();
    return prom;
  };
  /**
   * Builds a new nav tree for the added node and appends it to the current list
   * of kids. Will be called when the backing TreeNode gets a new kid added
   * (the `added` event is emitted), and we must create a new tree in the DOM
   * to reflect the change. Note that the expand/collapse button will be enabled
   * if this is the first kid to be added.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} kid
   * @returns {Promise} promise to be resolved when the new editor is
   * added
   */


  NavTree.prototype.$addKid = function (kid) {
    var that = this;
    return buildKid(that, kid).then(function (kidTree) {
      that.$updateButton()["catch"](logError);
      return kidTree.$setLoaded(that.$isExpanded());
    });
  };
  /**
   * Removes the nav tree editor from the DOM when a kid node is removed. Will
   * be called when the backing TreeNode gets a kid removed (the `removed`
   * event is emitted), and we must remove the associated tree from the DOM
   * to reflect the change. Note that the expand/collapse button will be
   * disabled if this is the last kid to be removed.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} kid the removed
   * node (has already been removed from the tree when this method is called)
   * @returns {Promise} promise to be resolved when the nav tree editor
   * has been removed from the DOM
   */


  NavTree.prototype.$removeKid = function (kid) {
    var that = this,
        kidEd = that.$getKidEditorFor(kid);

    if (kidEd) {
      var li = kidEd.jq();
      return kidEd.destroy().then(function () {
        li.remove();
        that.$updateButton()["catch"](logError);
      });
    } else {
      return resolve();
    }
  };
  /**
   * Updates the display element of the renamed kid. Note that the referenced
   * kid has already been updated in the tree; we're just updating the DOM to
   * reflect that.
   *
   * @private
   * @param {String} newName
   * @param {String} oldName
   * @returns {Promise} promise to be resolved when the editor for the
   * renamed kid has had its display element updated
   */


  NavTree.prototype.$renameKid = function (newName, oldName) {
    var that = this,
        node = that.value();
    return that.value().getKid(newName).then(function (kid) {
      var isDisplayed = !that.$displayFilter || that.$displayFilter(node, kid);

      if (isDisplayed) {
        var kidEd = that.$getKidEditorFor(kid);

        if (kidEd) {
          return kid.toDisplay().then(function (display) {
            return kidEd.$getDisplayElement().text(display);
          });
        } else {
          return that.$addKid(kid).then(function () {
            return that.$reorderKids();
          });
        }
      } else {
        return that.$removeKid(kid);
      }
    });
  };
  /**
   * Shuffles around the order of child trees in the DOM to reflect changes in
   * nav child order.
   *
   * @private
   * @returns {Promise} promise to be resolved when the reordering is
   * complete
   */


  NavTree.prototype.$reorderKids = function () {
    var that = this,
        node = this.value(),
        kidsList = this.$getKidsList(),
        kidDoms = kidsList.children();
    return node.getKids().then(function (kids) {
      //kids are newly ordered at this point. for each kid, find the existing
      //(out of order) DOM element for that kid, and map it into a correctly
      //ordered array of DOM elements.
      var orderedDoms = _.map(kids, function (kid) {
        var isDisplayed = !that.$displayFilter || that.$displayFilter(that, kid);

        if (isDisplayed) {
          for (var i = 0; i < kidDoms.length; i++) {
            if (kid.equals($(kidDoms[i]).data('widget').value())) {
              return kidDoms[i];
            }
          }
        }
      });

      orderedDoms = _.compact(orderedDoms); //detach the out-of-order elements and pop them back in in order.

      kidDoms.detach();
      kidsList.html($(orderedDoms));
    });
  };
  /**
   * Updates the CSS classes and disabled status of the expand/collapse button
   * to reflect whether the tree can be expanded or collapsed.
   *
   * @private
   * @returns {Promise}
   */


  NavTree.prototype.$updateButton = function () {
    var button = this.$getButton(),
        node = this.value(),
        loadKids = this.$loadKids,
        expanded = this.$isExpanded();

    function doUpdate(disabled) {
      button.prop('disabled', disabled).toggleClass('expanded', !!expanded).toggleClass('collapsed', !expanded);
    }

    if (node) {
      if (node.$kidsLoaded || loadKids) {
        return node.getKids().then(function (kids) {
          doUpdate(!kids.length);
        });
      } else {
        return Promise.resolve(doUpdate(false));
      }
    } else {
      return Promise.resolve(doUpdate(true));
    }
  };
  /**
   * Determine if a node should be displayed.
   * @callback module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} parent
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} child
   * @returns {boolean} true if to be displayed, false other wise
   *
   */

  /**
   * Returns the display filter if one is registered.
   *
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter}
   */


  NavTree.prototype.getDisplayFilter = function () {
    return this.$displayFilter;
  };
  /**
   * Sets the display filter
   * @param {module:nmodule/webEditors/rc/wb/tree/NavTree~displayFilter} filter
   */


  NavTree.prototype.setDisplayFilter = function (filter) {
    this.$displayFilter = filter;
  };
  /**
   * Return true if this nav tree is currently selected/highlighted by the user.
   *
   * @returns {Boolean}
   */


  NavTree.prototype.isSelected = function () {
    return this.$selected;
  }; //TODO: select a swath via shift?

  /**
   * Sets the nav tree's selected status. A second modified parameter allows
   * you to specify whether we are modifying an existing selection (e.g. by
   * holding the CTRL key) or making a brand new selection. The event triggered
   * will pass this as a parameter to the event handler.
   *
   * If setting selected to `false`, and the `NavTree` is already unselected,
   * no DOM change will occur and no event will be fired (why should one care?)
   * However setting selected to `true` when the node is already selected, it
   * will still fire an event. (Consider when you have several nodes selected
   * with `Ctrl` and click on one of the nodes already selected: we need to
   * fire the selected event so that we know to clear the rest of the
   * selection.)
   *
   * @param {Boolean} selected
   * @param {object} [params]
   * @param {Boolean} [params.modified] set to true if this should modify an
   * existing selection instead of starting a new one.
   * @param {Boolean} [params.silent] set to true if this should not fire a
   * `SELECTED_EVENT`.
   *
   */


  NavTree.prototype.setSelected = function (selected, params) {
    var that = this,
        jq = that.jq(),
        eventName = selected ? NavTree.SELECTED_EVENT : NavTree.DESELECTED_EVENT,
        modified = params && params.modified,
        silent = params && params.silent;

    if (!selected && !that.$selected) {
      return;
    }

    that.$selected = selected;
    jq.toggleClass('selected', selected);

    if (!silent) {
      jq.trigger(eventName, [that, !!modified]);
    }
  };
  /**
   * Indicates that this node has been "activated" by the user, say by double-
   * clicking it. In contrast with "selected", say by a single click. Triggers
   * `NavTree.ACTIVATED_EVENT`.
   */


  NavTree.prototype.activate = function () {
    this.trigger(NavTree.ACTIVATED_EVENT);
  };
  /**
   * Performs a depth-first search through the nav tree and assembles all
   * selected nodes into an array.
   *
   * @returns {Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>} an array
   * of selected nodes
   */


  NavTree.prototype.getSelectedNodes = function () {
    var selectedNodes = this.isSelected() ? [this.value()] : [];

    _.each(this.$getKids(), function (kidEd) {
      var selectedKids = kidEd.getSelectedNodes();

      if (selectedKids) {
        selectedNodes = selectedNodes.concat(selectedKids);
      }
    });

    return selectedNodes;
  };
  /**
   * Get an array of nodes that are currently visible, i.e., expanded, within
   * this tree. If the tree is collapsed, only the one currently loaded node
   * will be resolved.
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>>}
   */


  NavTree.prototype.getVisibleNodes = function () {
    var nodes = [this.value()];

    if (!this.$isExpanded()) {
      return Promise.resolve(nodes);
    } else {
      return Promise.all(_.map(this.$getKids(), function (kid) {
        return kid.getVisibleNodes();
      })).then(function (nodeArrays) {
        return nodes.concat(_.flatten(nodeArrays));
      });
    }
  };
  /**
   * Implementation for `NavMonitorSupport` - get the nav ORDs of all visible
   * tree nodes.
   * @see module:nmodule/webEditors/rc/wb/mixin/NavMonitorSupport
   * @returns {Promise.<Array.<baja.Ord>>}
   */


  NavTree.prototype.getTouchableOrds = function () {
    return this.getVisibleNodes().then(function (nodes) {
      return _.chain(nodes).map(toValue).filter(isNavNode).map(toNavOrd).value();
    });
  };
  /**
   * Expands out the tree, recursing down through the sub-nodes until the
   * specified node is found and selected, or selecting the parent node when
   * reaching an unrecognized node.
   *
   * The given path should be an array of node names. The first name in the path
   * should match the name of the node loaded into this tree, and the second
   * name in the path (if provided) should match one of the child tree nodes. It
   * will then recurse down until a match is found, or an unrecognized node is
   * reached. Each referenced tree node will have all of its child nodes loaded.
   *
   * @param {Array.<String>} path - the path of node names to select
   * @param {Object} [params]
   * @param {Boolean} [params.modified] set to true if this should modify an
   * existing selection instead of starting a new one.
   * @param {Boolean} [params.silent] set to true if this should not fire a
   * `SELECTED_EVENT`.
   *
   * @returns {Promise} promise to be resolved when a match or unrecognized node
   * has been been found and tree nodes leading to it have been loaded and
   * expanded.
   */


  NavTree.prototype.setSelectedPath = function (path, params) {
    if (!_.isArray(path)) {
      return reject('Array required');
    }

    if (!path.length) {
      return resolve();
    }

    var that = this,
        restOfPath = path.slice(),
        head = restOfPath.shift(),
        next = restOfPath[0],
        node = that.value(),
        modified = params && params.modified;

    if (node.getName() !== head) {
      return reject(); // Node not found
    }

    if (!next) {
      //no more kids to check. it's me!
      that.setSelected(true, params);

      if (!modified) {
        setSelectedEditor(getRoot(that), that);
      }

      return resolve();
    }

    return that.$setLoaded(true).then(function () {
      return node.getKid(next);
    }).then(function (kid) {
      var kidEd = that.$getKidEditorFor(kid);

      if (!kidEd) {
        return reject(that.setSelected(true, params)); // Node not found
      }

      return kidEd.setSelectedPath(restOfPath, params);
    }) // eslint-disable-next-line promise/no-return-in-finally
    ["finally"](function () {
      return that.expand();
    });
  };
  /**
   * Collapses the tree.
   *
   * @returns {Promise}
   */


  NavTree.prototype.collapse = function () {
    return this.$setExpanded(false);
  };
  /**
   * Expands the tree.
   *
   * @returns {Promise}
   */


  NavTree.prototype.expand = function () {
    return this.$setExpanded(true);
  };
  /**
   * Adds event listeners to the backing `TreeNode`. Whenever the node has its
   * children renamed, reordered, added to or removed from, the corresponding
   * method on the `NavTree` will be called to keep the DOM up to date.
   *
   * This will automatically be called as soon as the node is loaded into the
   * `NavTree`.
   *
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$addKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$removeKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$renameKid
   * @see module:nmodule/webEditors/rc/wb/tree/NavTree#$reorderKids
   */


  NavTree.prototype.subscribe = function () {
    var that = this,
        node = that.value();
    node.on('added', that.$addedHandler = function (kid) {
      if (!that.$displayFilter || that.$displayFilter(node, kid)) {
        that.$addKid(kid)["catch"](logError);
      }
    }).on('removed', that.$removedHandler = function (kid) {
      that.$removeKid(kid)["catch"](logError);
    }).on('renamed', that.$renamedHandler = function (newName, oldName) {
      that.$renameKid(newName, oldName)["catch"](logError);
    }).on('reordered', that.$reorderedHandler = function () {
      that.$reorderKids(node)["catch"](logError);
    });
  };
  /**
   * Removes event listeners added in `subscribe()`. Will be called when the
   * `NavTree` is destroyed, or when an old node is unloaded.
   */


  NavTree.prototype.unsubscribe = function () {
    var that = this,
        node = that.value();

    if (node) {
      node.removeListener('added', that.$addedHandler).removeListener('removed', that.$removedHandler).removeListener('renamed', that.$renamedHandler).removeListener('reordered', that.$reorderedHandler);
    }
  };

  function getRoot(tree) {
    var parent = tree.jq().parent().parent().data('widget');
    return parent && parent instanceof NavTree ? getRoot(parent) : tree;
  }
  /**
   * Sets up initial HTML for the nav tree node.
   * @param {JQuery} dom
   */


  NavTree.prototype.doInitialize = function (dom) {
    var that = this,
        enableHoverPreload = that.$enableHoverPreload,
        preloadTicket,
        touchdown = false,
        mouseupCancelled = false;
    dom.html(tplNavTree()).addClass('NavTree').toggleClass('hideRoot', !!that.properties().get('hideRoot'));
    dom.on([DESTROY_EVENT, LOAD_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    }); //new tree node selected by the user.

    dom.on(NavTree.SELECTED_EVENT, function (e, ed, modified) {
      if (!modified) {
        //we're not just modifying an existing selection - so go up to the root
        //and wipe out all existing selections except for the one we just made.
        setSelectedEditor(getRoot(that), ed);
      }
    });

    function cancelMouseup() {
      function uncancel() {
        mouseupCancelled = false;
        $(document).off('mouseup touchend', uncancel);
      }

      if (!mouseupCancelled) {
        $(document).on('mouseup touchend', uncancel);
      }

      mouseupCancelled = true;
    }

    function handleMouseSelection(e) {
      if (that.value().isSelectable()) {
        that.setSelected(!e.ctrlKey || !that.isSelected(), {
          modified: !!e.ctrlKey
        });
      }
    }

    contextMenuOnLongPress(dom.children('.display'));
    dom.children('.display').on('touchstart', function (e) {
      var touches = e.originalEvent.touches;

      if (touches.length > 1) {
        //don't select/activate on a pinch zoom
        cancelMouseup();
      } else {
        touchdown = true;
      }
    }).on('touchmove', function () {
      cancelMouseup();
    }).on('mousedown contextmenu', function (e) {
      if (!(that.isSelected() && !e.ctrlKey) && !mouseupCancelled) {
        handleMouseSelection(e);
        cancelMouseup();
      }
    }).on('touchend', function (e) {
      var multitouch = e.originalEvent.touches.length > 1;

      if (!mouseupCancelled && !multitouch && touchdown) {
        handleMouseSelection(e);
        that.trigger(NavTree.ACTIVATED_EVENT, that);
        touchdown = false;
        return false;
      }
    }).on('dblclick', function () {
      if (that.value().isSelectable()) {
        that.trigger(NavTree.ACTIVATED_EVENT, that);
        return false;
      }
    }).on('dragstart', function (e) {
      var node = that.value(),
          values = _.map(getRoot(that).getSelectedNodes(), function (node) {
        return node.value();
      });

      if (node.isDraggable()) {
        var clipboard = e.originalEvent.dataTransfer,
            mime = 'niagara/navnodes';
        dragDropUtils.toClipboard(clipboard, mime, values)["catch"](logError);
      }
    }).on('dragover dragenter', function (e) {
      if (that.value().isDropTarget()) {
        dom.children('.display').addClass(DROP_TARGET_CLASS);
        e.preventDefault();
      }
    }).on('dragend dragleave drop', function () {
      if (that.value().isDropTarget()) {
        dom.children('.display').removeClass(DROP_TARGET_CLASS);
      }
    }).on('drop', function (e) {
      var node = that.value();

      if (node.isDropTarget()) {
        var clipboard = e.originalEvent.dataTransfer;
        dragDropUtils.fromClipboard(clipboard).then(function (envelope) {
          return envelope.toValues();
        }).then(function (values) {
          return node.doDrop(values);
        })["catch"](feDialogs.error);
        return false;
      }
    });
    dom.children('button').on('click', function () {
      var p;

      if (that.$isExpanded()) {
        that.$disableTicket = setTimeout(function () {
          that.$setKidsLoaded(false)["catch"](logError);
        }, that.$getUnloadTimeout());
        p = that.collapse();
      } else {
        clearTimeout(that.$disableTicket);
        p = that.expand();
      }

      p["catch"](logError);
    });

    if (enableHoverPreload) {
      dom.children('button').on('mouseenter', function () {
        preloadTicket = setTimeout(function () {
          that.$setLoaded(true)["catch"](logError);
        }, HOVER_PRELOAD_DELAY);
      }).on('mouseleave', function () {
        clearTimeout(preloadTicket);
      });
    }
  };
  /**
   * Loads in a new `TreeNode`.
   *
   * *Important*: This node will be destroyed when this editor is destroyed.
   * Don't pass in `TreeNode`s you want to hold onto after you are finished
   * with the `NavTree`.
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} value
   * @returns {Promise}
   */


  NavTree.prototype.doLoad = function (value) {
    if (!isTreeNode(value)) {
      return reject('TreeNode required');
    }

    var that = this,
        dom = that.jq(),
        displayElement = that.$getDisplayElement(),
        oldNode = that.$oldNode,
        iconElement = that.$getIconElement(),
        oldIconEditor = iconElement.data('widget'),
        iconURIs = value.getIcon();
    that.unsubscribe();
    that.$oldNode = value;
    that.$updateButton()["catch"](logError);
    that.subscribe();
    return Promise.all([oldNode && oldNode !== value && oldNode.destroy(), oldIconEditor && oldIconEditor.destroy()]).then(function () {
      dom.children('.spacer').remove();
      dom.prepend(_.map(value.getFullPath().slice(1), function () {
        return '<div class="spacer"></div>';
      }).join(''));
      return Promise.all([value.activate(), value.toDisplay().then(function (display) {
        displayElement.text(display);
      }), fe.buildFor({
        dom: iconElement,
        value: iconURIs,
        type: IconEditor
      })]);
    }).then(function () {
      if (that.$loadKids) {
        return that.$setLoaded(true);
      }
    }).then(function () {
      dom.children('.display').prop('draggable', !!value.isDraggable());
      return that.$setExpanded(that.properties().get('hideRoot'));
    });
  };

  NavTree.prototype.getSubject = function () {
    return _.map(this.getSelectedNodes(), function (node) {
      return node.value();
    });
  };
  /**
   * Removes the `NavTree` CSS class, unsubscribes for events from the
   * backing `TreeNode`, and destroys all child editors. Essentially, wipes
   * out everything from here down.
   * @returns {*}
   */


  NavTree.prototype.doDestroy = function () {
    var that = this,
        dom = that.jq(),
        node = that.value();
    dom.removeClass('NavTree');
    dom.children('.expand, .display').off();
    that.unsubscribe();
    return that.getChildEditors().destroyAll().then(function () {
      return node && node.destroy();
    });
  };

  return NavTree;
});
