/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/tree/CommandTreeNode
 */
define(['Promise', 'underscore', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'nmodule/webEditors/rc/wb/tree/TreeNode'], function (Promise, _, Command, CommandGroup, TreeNode) {
  'use strict';
  /**
   * A tree node backed by `bajaux` `Command`s. Allows traversal through a
   * `CommandGroup`.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/tree/CommandTreeNode
   * @param {module:bajaux/commands/Command|module:bajaux/commands/CommandGroup} cmd
   * @see module:bajaux/commands/Command
   * @see module:bajaux/commands/CommandGroup
   */

  var CommandTreeNode = function CommandTreeNode(cmd) {
    if (!(cmd instanceof Command) && !(cmd instanceof CommandGroup)) {
      throw new Error('Command or CommandGroup required');
    }

    this.$cmd = cmd;
    TreeNode.call(this, cmd.getDisplayNameFormat());
  };

  CommandTreeNode.prototype = Object.create(TreeNode.prototype);
  CommandTreeNode.prototype.constructor = CommandTreeNode;
  /**
   * Resolves the `Command` or `CommandGroup`'s display name, using lexicon
   * formatting if needed.
   *
   * @returns {Promise} promise to be resolved with the display name
   */

  CommandTreeNode.prototype.toDisplay = function () {
    return this.value().toDisplayName();
  };
  /**
   * Returns the `Command` or `CommandGroup` passed to the constructor.
   * @returns {module:bajaux/commands/Command|module:bajaux/commands/CommandGroup}
   */


  CommandTreeNode.prototype.value = function () {
    return this.$cmd;
  }; //TODO: does this actually jibe with Command#getIcon? does Command#getIcon properly handle badges?

  /**
   * Returns the URIs to the associated `Command`'s icon images. If a
   * `CommandGroup`, returns an empty array.
   *
   * @returns {Array.<String>}
   */


  CommandTreeNode.prototype.getIcon = function () {
    var cmd = this.value();
    return cmd instanceof Command ? [cmd.getIcon()] : [];
  };
  /**
   * If a `CommandGroup` is loaded, instantiates and loads `CommandTreeNode`s
   * for the group's child groups and commands.
   *
   * @returns {Promise} promise to be resolved with an array of
   * `CommandTreeNode`s
   */


  CommandTreeNode.prototype.$loadKids = function () {
    var cmd = this.value(),
        kidNodes = cmd instanceof CommandGroup ? _.map(cmd.getChildren(), function (kid) {
      return new CommandTreeNode(kid);
    }) : [];
    return Promise.resolve(kidNodes);
  };
  /**
   * @returns {boolean} true if a CommandGroup with at least one child is loaded
   */


  CommandTreeNode.prototype.mayHaveKids = function () {
    var cmd = this.value();
    return cmd instanceof CommandGroup && cmd.getChildren().length !== 0;
  };

  return CommandTreeNode;
});
