/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/table/pagination/PaginationWidget
 */
define(['hbs!nmodule/webEditors/rc/wb/template/Pagination', 'lex!webEditors', 'log!nmodule.webEditors.rc.wb.table.pagination.PaginationWidget', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'css!nmodule/webEditors/rc/fe/webEditors-structure'], function (tplPagination, lexs, log, Command, CommandButtonGroup, tinyevents, BaseWidget, fe, feDialogs) {
  'use strict';

  var logSevere = log.severe.bind(log);
  var PAGE_CLASS = '.table-widget-page-number',
      NEXT_CLASS = '.table-widget-page-next',
      PREV_CLASS = '.table-widget-page-prev',
      ENTER_KEY = 13,
      DEFAULT_MAX_PAGESIZE = 100,
      MIN_PAGESIZE = 1,
      webEditorsLex = lexs[0];
  /**
   * Command to show the settings dialog.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/webEditors/rc/wb/table/pagination/PaginationWidget} ed
   */

  var SettingsCommand = function SettingsCommand(ed) {
    Command.call(this, {
      module: 'webEditors',
      lex: "paginationWidget.commands.settings",
      func: function func() {
        var model = ed.value();
        return feDialogs.showFor({
          value: model.getRowsPerPage(),
          properties: {
            max: ed.$getMaxPageSize(),
            min: MIN_PAGESIZE,
            prelabel: webEditorsLex.get("paginationWidget.settings.dialog.label.pageSize"),
            precision: 0
          },
          formFactor: 'mini',
          title: webEditorsLex.get("paginationWidget.settings.dialog.title")
        }).then(function (pageSize) {
          if (pageSize) {
            return model.setRowsPerPage(pageSize).then(function () {
              model.writeToStorage();
              return model.setCurrentPage(1);
            });
          }
        });
      }
    });
  };

  SettingsCommand.prototype = Object.create(Command.prototype);
  SettingsCommand.prototype.constructor = SettingsCommand;
  /**
   * Pagination Widget provides a UX to set page limit and turn pages.
   * Note that this widget is not intended to be used as a "Widget" and
   * will be initialized only by a Table presently.
   *
   * The widget emits a 'pageChanged' event when a page is changed and passes in
   * the index of the currently displayed/selected page. Table listens to this
   * event and handles the page load functionality.
   *
   * This widget supports the following properties:
   *
   * - `maxPageSize`: (number) maximum page size in rows per page, defaults to 100
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   * @alias module:nmodule/webEditors/rc/wb/table/pagination/PaginationWidget
   * @mixes tinyevents
   * @see module:nmodule/webEditors/rc/wb/table/Table#doInitialize
   */

  var PaginationWidget = function PaginationWidget() {
    var that = this;
    BaseWidget.apply(this, arguments);
    tinyevents(this);
    that.getCommandGroup().add(new SettingsCommand(that));
  };

  PaginationWidget.prototype = Object.create(BaseWidget.prototype);
  PaginationWidget.prototype.constructor = PaginationWidget;
  /**
   * @param {JQuery} dom
   * @returns {Promise}
   */

  PaginationWidget.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('PaginationWidget');
    dom.html(tplPagination({
      prevText: webEditorsLex.get("paginationWidget.commands.prev"),
      nextText: webEditorsLex.get("paginationWidget.commands.next")
    })); // Query text field handler (submit search on enter key press)

    dom.on('keypress change', PAGE_CLASS, function (event) {
      if (event.type === 'change' || event.which === ENTER_KEY) {
        var pageNumber = parseInt(that.$getPageNumberInput().val(), 10);

        if (!isNaN(pageNumber)) {
          that.value().setCurrentPage(pageNumber)["catch"](logSevere);
        }

        return false; //Prevent page from reloading
      }
    });
    dom.on('click', NEXT_CLASS, function () {
      var model = that.value();
      model.setCurrentPage(model.getCurrentPage() + 1)["catch"](logSevere);
    });
    dom.on('click', PREV_CLASS, function () {
      var model = that.value();
      model.setCurrentPage(model.getCurrentPage() - 1)["catch"](logSevere);
    });
    return fe.buildFor({
      dom: that.jq().find('.settings'),
      type: CommandButtonGroup,
      value: this.getCommandGroup(),
      properties: {
        toolbar: true
      }
    });
  };
  /**
   * @param {module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel} model
   */


  PaginationWidget.prototype.doLoad = function (model) {
    var that = this;
    that.updateFromModel(model);
    model.on('changed', function () {
      that.updateFromModel(model);
    });
  };
  /**
   * Update widget parameters (say after every page changed event)
   * @param {module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel} model
   */


  PaginationWidget.prototype.updateFromModel = function (model) {
    var that = this,
        currentPage = model.getCurrentPage(),
        pageCount = model.getPageCount();
    that.$getPageNumberInput().val(currentPage).prop('disabled', pageCount <= 1).prop('min', 1).prop('max', pageCount);
    that.$getPrevButton().prop('disabled', currentPage <= 1);
    that.$getNextButton().prop('disabled', currentPage >= pageCount);
  };
  /**
   * Clean up the dom and call Widget destroy
   * @returns {*}
   */


  PaginationWidget.prototype.doDestroy = function () {
    this.jq().removeClass('PaginationWidget');
  };
  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */


  PaginationWidget.prototype.$getSettingsCommand = function () {
    return this.getCommandGroup().get(0);
  };
  /**
   * @private
   * @returns {JQuery}
   */


  PaginationWidget.prototype.$getPageNumberInput = function () {
    return this.jq().find(PAGE_CLASS);
  };
  /**
   * @private
   * @returns {JQuery}
   */


  PaginationWidget.prototype.$getNextButton = function () {
    return this.jq().find(NEXT_CLASS);
  };
  /**
   * @private
   * @returns {JQuery}
   */


  PaginationWidget.prototype.$getPrevButton = function () {
    return this.jq().find(PREV_CLASS);
  };
  /**
   * Get the maximum page size in number of rows.
   *
   * This uses maxPageSize property, but defaults to DEFAULT_MAX_PAGESIZE
   * (currently 100) if the property isn't set.
   *
   * @private
   * @returns {Number}
   */


  PaginationWidget.prototype.$getMaxPageSize = function () {
    var maxPageSize = this.properties().getValue('maxPageSize');
    return typeof maxPageSize === 'number' ? maxPageSize : DEFAULT_MAX_PAGESIZE;
  };

  return PaginationWidget;
});
