/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * @module nmodule/webEditors/rc/wb/table/model/columns/IconColumn
 */
define(['jquery', 'Promise', 'nmodule/webEditors/rc/fe/baja/IconEditor', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/table/model/Column'], function ($, Promise, IconEditor, fe, Column) {
  'use strict';
  /**
   * API Status: **Development**
   *
   * Column type used to show an icon in a specific column the table. This
   * column type will not, by default, be available in the show/hide menu, and
   * will not have the sorting functionality available. Users of this type can
   * change this behavior by calling the `#setSortable` and `#setHidable`
   * functions after construction.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/model/columns/IconColumn
   * @extends module:nmodule/webEditors/rc/wb/table/model/Column
   */

  var IconColumn = function IconColumn(name, params) {
    name = name || 'icon';
    params = params || {};
    params.flags = Column.flags.READONLY;

    if (!params.displayName) {
      params.displayName = ' '; // a single space for an empty column heading
    }

    Column.call(this, name, params);
    this.setSortable(false);
    this.setHidable(false);
    this.setExportable(false);
  };

  IconColumn.prototype = Object.create(Column.prototype);
  /**
   * Gets the icon URI for the given row. By default, this will delegate to
   * the `Row`'s getIcon() function. This function can be overridden to allow
   * the source of the icon to be more flexible. This method may return a single
   * icon, or an array of icons.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row A table row
   * @returns {baja.Icon|Array.<baja.Icon>} the icon(s) for the row, or null if the row does not
   * specify any.
   */

  IconColumn.prototype.getValueFor = function (row) {
    return row.getIcon();
  };
  /**
   * Build the dom for the cell. This will build an `IconEditor` for each
   * icon returned by the `getValueFor` function.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom
   * @returns {Promise} A promise resolved with the img element added to the dom.
   */


  IconColumn.prototype.buildCell = function (row, dom) {
    var icons = this.getValueFor(row);

    if (icons) {
      if (!Array.isArray(icons)) {
        icons = [icons];
      }

      return Promise.all(icons.map(function (icon) {
        return fe.buildFor({
          value: icon,
          type: IconEditor,
          dom: $('<span/>').appendTo(dom)
        });
      }));
    } else {
      return Promise.resolve(dom.html('&nbsp;'));
    }
  };
  /**
   * Destroy the IconEditor that was created for this cell.
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom
   * @returns {Promise}
   */


  IconColumn.prototype.destroyCell = function (row, dom) {
    var promises = [];
    dom.find('.IconEditor').each(function () {
      var fe = $(this).data('widget');

      if (fe) {
        promises.push(fe.destroy());
      }
    });
    return Promise.all(promises);
  };

  return IconColumn;
});
