/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson.
 */

/**
 * API Status: **Private**
 *
 * Loads the Sidebars for the Shell Profile.
 *
 * @module nmodule/webEditors/rc/wb/profile/profileSideBars
 */
define(["baja!", "lex!webEditors", "jquery", "profileInfo", "underscore", "nmodule/webEditors/rc/util/storageUtil", "nmodule/webEditors/rc/wb/profile/paletteSideBar", "nmodule/webEditors/rc/wb/profile/navTreeSideBar", "nmodule/webEditors/rc/wb/profile/searchSideBar", "hbs!nmodule/webEditors/rc/wb/profile/profileSideBarsTemplate", "hbs!nmodule/webEditors/rc/wb/profile/profileSideToggleBarTemplate", "nmodule/js/rc/jquery/split-pane/split-pane"], function (baja, lexes, $, profileInfo, _, storageUtil, paletteSideBar, navTreeSideBar, searchSideBar, template, toggleBarTemplate) {
  "use strict";
  /**
   * @param {Array.<SidebarOption>} options
   */

  function pluckSidebarKeys(options) {
    return _.pluck(options, 'key');
  }

  var sidebarOptions = profileInfo.getSideBarConfigOptions(),
      MAIN_SIDEBAR_NAME = 'sidebar-left',
      DEFAULT_SIDEBAR_WIDTH = 300,
      SIDEBAR_NAMES = [MAIN_SIDEBAR_NAME].concat(pluckSidebarKeys(sidebarOptions)),
      webEditorLex = lexes[0];
  var currentStates; ////////////////////////////////////////////////////////////////
  // Saving/Layout
  ////////////////////////////////////////////////////////////////

  /**
   * Get the main element for a particular sidebar.
   * @param {string} sidebarName
   * @returns {jQuery}
   */

  function getSidebar(sidebarName) {
    return $('#WebShell-' + sidebarName);
  }
  /**
   * @param {string} sidebarName
   * @returns {boolean} true if this is the main content sidebar
   */


  function isMainSidebar(sidebarName) {
    return sidebarName === MAIN_SIDEBAR_NAME;
  }
  /**
   * Get the left-hand toggle button for a particular sidebar.
   * @param {string} sidebarName
   * @returns {jQuery}
   */


  function getSidebarToggleButton(sidebarName) {
    return $('#toggle-' + sidebarName);
  }
  /**
   * When first loading the page, no sidebar layout information will be stored
   * in local storage, so generate some smart defaults to do the first page
   * layout.
   * @function
   * @returns {Object.<String, SidebarState>} sidebar name -> state
   */


  var defaultSidebarStates = _.once(function defaultSidebarStates() {
    var states = {},
        count = SIDEBAR_NAMES.length,
        i = 0;

    _.each(SIDEBAR_NAMES, function (sidebarName) {
      var offsetPercent,
          isMain = isMainSidebar(sidebarName);

      if (isMain) {
        offsetPercent = DEFAULT_SIDEBAR_WIDTH / $('#WebShell').width();
      } else {
        i++;
        offsetPercent = 1 / (count - i);
      }

      states[sidebarName] = {
        visible: false,
        orientation: isMain ? 'vertical' : 'horizontal',
        offsetPercent: offsetPercent
      };
    });

    return states;
  });
  /**
   * Read the current state of all sidebars out of the DOM.
   * @returns {Object.<String, SidebarState>} sidebar name -> state
   */


  function readSidebarStates() {
    var states = {},
        defaults = defaultSidebarStates();

    _.each(SIDEBAR_NAMES, function (sidebarName) {
      var sidebar = getSidebar(sidebarName),
          divider = sidebar.next(),
          //currently only the main content is vertical.
      horizontal = !isMainSidebar(sidebarName),
          parentWidth = sidebar.parent()[horizontal ? 'height' : 'width'](),
          offset = parseInt(divider.css(horizontal ? 'top' : 'left'), 10),
          offsetPercent = parentWidth ? offset / parentWidth : 0;

      if (offset < 0) {
        offsetPercent = defaults[sidebarName].offsetPercent;
      }

      states[sidebarName] = {
        visible: sidebar.is(':visible'),
        orientation: horizontal ? 'horizontal' : 'vertical',
        offsetPercent: offsetPercent
      };
    });

    return states;
  }

  function getVisibleSidebarsAfter(sidebarName, states) {
    var i = _.indexOf(SIDEBAR_NAMES, sidebarName);

    return _.filter(SIDEBAR_NAMES.slice(i + 1), function (after) {
      return states[after].visible;
    });
  }
  /**
   * Given sidebar state information, go and apply it to the sidebar elements
   * in the DOM.
   * @param {Object.<String, SidebarState>} states
   */


  function applySidebarStates(states) {
    var anyShown = false;

    _.each(SIDEBAR_NAMES, function (sidebarName) {
      var state = states[sidebarName],
          isMain = isMainSidebar(sidebarName),
          button = getSidebarToggleButton(sidebarName),
          sidebar = getSidebar(sidebarName),
          parent = sidebar.parent(),
          divider = parent.children('.split-pane-divider'),
          //all subsequent sidebars are contained in the same element.
      restContainer = divider.next(),
          horizontal = state.orientation === 'horizontal',
          offsetPercent = state.offsetPercent,
          posProp = horizontal ? 'top' : 'left',
          widthProp = horizontal ? 'height' : 'width',
          minProp = 'min-' + widthProp,
          offset = offsetPercent * parent[widthProp](),
          //are any sidebars visible after this one?
      visibleAfter = getVisibleSidebarsAfter(sidebarName, states),
          isFirstShown = isMain ? !!visibleAfter.length : state.visible,
          isSecondShown = isMain || !!visibleAfter.length,
          isDividerShown = isFirstShown && isSecondShown; // split-pane uses the divider width and second container min-height to
      // limit the height of the main sidebar. if these two are hidden, set them
      // to 0 to avoid having split-pane chop down the main sidebar contents.

      divider.css(widthProp, isDividerShown ? '' : 0);
      restContainer.css(minProp, isSecondShown ? '' : 0);

      if (isFirstShown) {
        if (!isMain) {
          anyShown = true;
        } else if (divider.css('visibility') === 'hidden') {
          //if the profile divider's visibility is hidden, we're in mobile mode,
          // so set the sidebar container to the default width
          offset = DEFAULT_SIDEBAR_WIDTH;
        }

        sidebar[widthProp](!isSecondShown ? '100%' : offset);
      } else {
        offset = 0;
      }

      if (isSecondShown) {
        restContainer.css(posProp, offset);

        if (!isMain) {
          //set a min-width/min-height on the rest of the sidebars to avoid
          //letting them
          var minValue = _.reduce(_.map(visibleAfter, function (sidebarName) {
            return parseInt(getSidebar(sidebarName).css(minProp), 10);
          }), function (a, b) {
            return a + b;
          }, 0);

          restContainer.css(minProp, minValue);
        }
      }

      if (isDividerShown) {
        divider.css(posProp, offset);
      }

      sidebar.toggle(isFirstShown);
      restContainer.toggle(isSecondShown);
      divider.toggle(isDividerShown);
      button.toggleClass('selected', isFirstShown);

      if (isFirstShown) {
        sidebar.trigger('sidebar-layout');
      }

      sidebar.trigger('sidebar-toggle');
    });

    $('#WebShell-sidebar-left, #WebShell-content-divider').toggle(anyShown);
  }

  function layoutPanes() {
    if (!currentStates) {
      throw new Error('sidebar states not setup');
    }

    applySidebarStates(currentStates);
  }

  function getLocalStorageKey() {
    return baja.getUserName() + '.profile.paneDimensions';
  }
  /**
   * Saves the position of the panes in the Profile.
   *
   * @inner
   * @function
   */


  var saveStates = _.throttle(function () {
    _.each(readSidebarStates(), function (state, sidebarName) {
      var currentState = currentStates[sidebarName];

      if (state.visible) {
        currentState.offsetPercent = state.offsetPercent;
      }

      currentState.visible = state.visible;
    });

    var json = {
      states: currentStates,
      sideBars: sidebarOptions
    },
        key = getLocalStorageKey(); //TODO: come back and use local storage manager module.
    //should persist stationside so it can be used during initial page render
    //otherwise you get a "pop" where the page is rendered, and then later the dimensions are set

    try {
      storageUtil.getLocalStorage().setItem(key, JSON.stringify(json));
    } catch (e) {}
  }); ////////////////////////////////////////////////////////////////
  // Loading
  ////////////////////////////////////////////////////////////////


  function toggleSidebar(sidebarName) {
    currentStates[sidebarName].visible = !currentStates[sidebarName].visible; //to ensure sidebars don't accidentally disappear after resizing and then
    //toggling, revert their sizes to equally spaced.
    //avoids this problem:
    // - show nav and search - drag search all the way down
    // - hide nav, show palette - drag palette all the way down
    // - show nav - palette now off the bottom of the screen
    //TODO: remove this so sidebars retain their size after toggling.

    _.each(currentStates, function (state, name) {
      if (!isMainSidebar(name)) {
        var visibleAfter = getVisibleSidebarsAfter(name, currentStates).length;
        currentStates[name].offsetPercent = 1 / (visibleAfter + 1);
      }
    });

    layoutPanes();
    saveStates();
  }
  /**
   * Load the Sidebars into the Toolbar.
   *
   * @inner
   * @private
   */


  function loadToolbar() {
    /**
     * Get targeted sidebar name.
     * @param {jQuery.Event} ev
     * @returns {string}
     */
    function getSidebarName(ev) {
      var jq = $(ev.target).closest('.toggleSideBar');
      return jq.data('sidebar');
    }
    /**
     * Toggle highlight status for the sidebar's display name tab.
     * @param {JQuery} sidebar
     * @param {boolean} isHighlighted
     */


    function toggleHighlight(sidebar, isHighlighted) {
      sidebar.find('.header-contents .tab').toggleClass('highlight', isHighlighted);
    }

    $("#WebShell-toolbar").html(toggleBarTemplate({
      options: sidebarOptions,
      commandsTitle: webEditorLex.get("commandsSideBar")
    }));
    $('#WebShell-maintogglebar').on('click', '.toggleSideBar', function (ev) {
      toggleSidebar(getSidebarName(ev));
    }).on('mouseenter', '.toggleSideBar', function (ev) {
      var sidebar = getSidebar(getSidebarName(ev));
      toggleHighlight(sidebar, true);
    }).on('mouseleave', '.toggleSideBar', function (ev) {
      var sidebar = getSidebar(getSidebarName(ev));
      toggleHighlight(sidebar, false);
    });
  }
  /**
   *
   * @param {Array.<Object>} sidebars
   * @returns {*}
   */


  function doTemplate(sidebars) {
    var bar = sidebars[0],
        rest = sidebars.slice(1);
    return bar && template({
      key: bar.key,
      last: rest.length === 1,
      next: doTemplate(rest)
    });
  }
  /**
   * Perform one-time setup of the profile sidebars. Will read the sidebar
   * layout information from local storage, if present, and use it to
   * reconstitute the previous sidebar layout. (This prevents the sidebars from
   * resetting to default positions on every page load.)
   *
   * @param  {Function} hyperlinkFunc The function used for hyperlinking in the
   * profile.
   */


  function loadSidebars(hyperlinkFunc) {
    var dom = $("#WebShell"),
        config = profileInfo.getConfigOptions(),
        json;
    $('#WebShell-sidebar-left').html(doTemplate(sidebarOptions));
    dom.addClass('fixed-left');

    if (config.navTreeSideBar) {
      navTreeSideBar($("#WebShell-navTreeSideBar"), hyperlinkFunc, config);
    }

    if (config.paletteSideBar) {
      paletteSideBar($("#WebShell-paletteSideBar"));
    }

    if (config.searchSideBar) {
      searchSideBar($("#WebShell-searchSideBar"));
    } // Initialize side bars


    try {
      json = storageUtil.getLocalStorage().getItem(getLocalStorageKey());
      json = json && JSON.parse(json);
    } catch (e) {} // If there's a different configuration of side bars then reset the layout.


    if (json && json.sideBars && pluckSidebarKeys(json.sideBars).join() !== pluckSidebarKeys(sidebarOptions).join()) {
      json = null;
    }

    currentStates = json && json.states || defaultSidebarStates();
    layoutPanes();
    dom.splitPane();
    dom.find('.split-pane').each(function () {
      $(this).splitPane();
    });
    dom.on('splitpaneresize', _.throttle(saveStates, 500));
  }
  /**
   * Loads the Profile's Sidebars.
   *
   * @private
   * @alias nmodule/webEditors/rc/wb/profile/profileSideBars
    * @param  {Function} hyperlinkFunc The function used for
   * hyperlinking in the profile.
   */


  return function load(hyperlinkFunc) {
    loadToolbar();
    loadSidebars(hyperlinkFunc);
  }; ////////////////////////////////////////////////////////////////
  // Typedefs
  ////////////////////////////////////////////////////////////////

  /**
   * Sidebar configuration option, sent down from the station. Generated in
   * profile.vm by BHTML5HxProfile.
   * @private
   * @typedef {Object} SidebarOption
   * @property {string} className
   * @property {string} displayName
   * @property {string} key
   */

  /**
   * Describes the current layout and visibility of a sidebar.
   *
   * @private
   * @typedef {Object} SidebarState
   * @property {boolean} visible
   * @property {string} orientation `horizontal` or `vertical` - main divider is
   * vertical, left sidebars are horizontal
   * @property {number} offsetPercent
   */
});
