/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson.
 */

/*eslint-env browser */

/*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/profile/ServletViewWidget
 */
define(['jquery', 'baja!', 'Promise', 'underscore', 'bajaux/Widget', 'nmodule/js/rc/asyncUtils/asyncUtils'], function ($, baja, Promise, _, Widget, asyncUtils) {
  "use strict";

  var doRequire = asyncUtils.doRequire;
  /**
   * A bajaux Widget for displaying Servlet View content in an iframe. This is
   * intended to be only used in conjunction with the HTML5 Hx Profile.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/profile/ServletViewWidget
   * @extends module:bajaux/Widget
   */

  var ServletViewWidget = function ServletViewWidget() {
    Widget.apply(this, arguments);
  };

  ServletViewWidget.prototype = Object.create(Widget.prototype);
  ServletViewWidget.prototype.constructor = ServletViewWidget;
  var fullScreenParamKey = ServletViewWidget.fullScreenParamKey = "fullScreen",
      fullScreenParam = ServletViewWidget.fullScreenParam = fullScreenParamKey + "=true";
  /**
   * Adds a parameter to the specified href that specifies whether the given view
   * should be rendered full screen or not.
   *
   * A full screen rendering of an Hx View renders just the view and not the profile.
   *
   * Please note, the way the ORD is computed must be kept in sync with
   * 'hx-wb/src/com/tridium/hx/BHxWebWidget#addFullScreenParam'.
   *
   * @param {String} href The href to add the full screen parameter too.
   */

  function addFullScreenParam(href) {
    if (/\/ord[/?]/.test(href)) {
      href = href.replace("/ord?", "/ord/");
      var match = /\?(.+)$/.exec(href),
          hasFullScreenParam = match && match[1].indexOf(fullScreenParamKey + "=") !== -1;

      if (hasFullScreenParam) {
        return href.replace(fullScreenParamKey + "=false", fullScreenParam);
      }

      href += "%7Cview:?fullScreen=true";
    }

    return href;
  }
  /**
   * Add a click handler to the iframe so we can intercept all click events
   * and process them internally to see if the full screen parameter needs to be
   * added.
   *
   * @private
   */


  ServletViewWidget.prototype.$addClickFrameHandler = function () {
    var that = this,
        wnd = that.$getFrameWnd();

    if (wnd) {
      that.$onClickFrame = that.$onClickFrame || function (e) {
        // Ignore this event if preventDefault has been called.
        if (e.originalEvent && e.originalEvent.defaultPrevented) {
          return;
        }

        var src = $(e.target).closest("a"),
            href = src.attr("href"),
            newHref,
            frameWnd = that.$getFrameWnd();

        if (src && href && frameWnd) {
          newHref = addFullScreenParam(href); // If the new hyperlink is different then change the source of
          // iframe.

          if (newHref !== href) {
            e.preventDefault();
            window.niagara.env.hyperlink(newHref);
          }
        }
      };

      $(wnd.document).find("body").first().on("click", that.$onClickFrame);
    }
  };
  /**
   * Load the hyperlink into the widget.
   *
   * @private
   *
   * @param  {String} uri The hyperlink to load.
   * @returns {Promise} A promise that's resolved once the page has been loaded.
   */


  ServletViewWidget.prototype.$loadFromUri = function (uri) {
    var uriWithParam = addFullScreenParam(uri),
        that = this; // eslint-disable-next-line promise/avoid-new

    return new Promise(function (resolve, reject) {
      var frame = that.$getFrame(),
          src = frame.attr("src");

      function onLoaded() {
        try {
          // eslint-disable-next-line no-unused-expressions
          that.$getFrameWnd().Object;
        } catch (err) {
          return reject(err);
        }

        that.$addClickFrameHandler();
        frame.css("visibility", "inherit");
        resolve();
      }

      if (src !== uriWithParam) {
        frame = that.$setFrameSrc(uriWithParam);
        frame.one("load", onLoaded);
      } else {
        //TODO: who's ready? should this be $getFrameWnd().$(onLoaded)?
        frame.ready(onLoaded);
      }
    });
  };
  /**
   * @private
   * @param {string} uri the new iframe URI
   * @returns {JQuery} the updated iframe
   */


  ServletViewWidget.prototype.$setFrameSrc = function (uri) {
    var frame = this.$getFrame();
    frame.replaceWith(frame.clone().attr('src', uri));
    return this.$getFrame();
  };
  /**
   * Called to resolve an ORD.
   *
   * There's no need to resolve an ORD for a Servlet View as everything is being loaded through an iframe.
   *
   * @param  data The data to resolve.
   * @returns {Promise} A promise that resolves with the data.
   */


  ServletViewWidget.prototype.resolve = function (data) {
    // Just resolve with the data. There's no need to resolve this in BajaScript as we're just loading the
    // current view anyway.
    return Promise.resolve(data);
  };
  /**
   * Called to load a new value.
   */


  ServletViewWidget.prototype.doLoad = function () {
    return this.$loadFromUri(window.location.href);
  };
  /**
   * Called when destroyed.
   */


  ServletViewWidget.prototype.doDestroy = function () {
    var that = this,
        wnd = that.$getFrameWnd(),
        onClickFrame = that.$onClickFrame;

    if (wnd && onClickFrame) {
      $(wnd.document).find("body").first().off("click", onClickFrame);
      delete that.$onClickFrame;
    }

    that.$setFrameSrc("about:blank");
    that.$getFrame().css("visibility", "hidden");
  };
  /**
   * Returns the internal iframe's window.
   *
   * @private
   *
   * @returns {Window} The internal iframe's window or null if it can't be found.
   */


  ServletViewWidget.prototype.$getFrameWnd = function () {
    var frame = this.$getFrame()[0];
    return frame && frame.contentWindow || null;
  };
  /**
   * Returns the internal iframe's root jquery function
   *
   * @since Niagara 4.6
   * @private
   *
   * @returns {jQuery}
   */


  ServletViewWidget.prototype.$getFrameJQuery = function () {
    var wnd = this.$getFrameWnd();
    return wnd ? wnd.jQuery : null;
  };
  /**
   * Returns the internal iframe's root jquery function
   *
   * @since Niagara 4.6
   * @private
   *
   * @returns {HTMLElement}
   */


  ServletViewWidget.prototype.$getFrameBody = function () {
    var frame = this.$getFrameWnd();
    return frame ? frame.document ? frame.document.body : null : null;
  };
  /**
   * @private
   * @returns {JQuery} The iframe DOM object.
   */


  ServletViewWidget.prototype.$getFrame = function () {
    return $("#servletViewWidget");
  };
  /**
   * Check the iframe to see if it's been modified in any way.
   *
   * @returns {Boolean} true if the Widget has been modified.
   */


  ServletViewWidget.prototype.isModified = function () {
    var wnd = this.$getFrameWnd(),
        res;

    if (wnd && typeof wnd.onbeforeunload === "function") {
      res = wnd.onbeforeunload();
    }

    return res !== undefined && res !== null ? true : Widget.prototype.isModified.apply(this, arguments);
  };
  /**
   * Call resize on the iframe using the iframe's jquery.
   *
   * @private
   */


  ServletViewWidget.prototype.$doResize = function () {
    var iframeJQuery = this.$getFrameJQuery(),
        iframeBody = this.$getFrameBody();

    if (iframeJQuery && iframeBody) {
      iframeJQuery(iframeBody).resize();
    }
  };
  /**
   * Notify the widget that it is being resized.
   *
   * @private
   */


  ServletViewWidget.prototype.$resize = _.debounce(function () {
    return this.$doResize();
  }, 200);
  /**
   * Indicates if the browser is running on iOS.
   *
   * @private
   *
   * @returns {Boolean}
   */

  ServletViewWidget.prototype.$isIOS = function () {
    var wnd = this.$getFrameWnd(),
        frameHx = wnd.hx;
    return frameHx && frameHx.ios;
  };
  /**
   * Called when the layout of the Widget changes.
   *
   * @returns {*}
   */


  ServletViewWidget.prototype.doLayout = function () {
    if (this.$isIOS()) {
      return this.$resize();
    }
  };
  /**
   * Called to save the widget.
   *
   * This will submit the form and wait for its response.
   *
   * @returns {Promise} A promise that's resolved once the widget is saved.
   */


  ServletViewWidget.prototype.doSave = function () {
    var wnd = this.$getFrameWnd(),
        frameHx = wnd.hx,
        savePromise = null,
        formSubmit;

    if (frameHx) {
      wnd.onbeforeunload = null; // eslint-disable-next-line promise/avoid-new

      formSubmit = new Promise(function (resolve, reject) {
        savePromise = frameHx.save(window, resolve, reject);
      });
      return Promise.all([savePromise, formSubmit]);
    }
  };
  /**
   * Returns the Ord backing this widget
   *
   * @returns {baja.Ord}
   */


  ServletViewWidget.prototype.getOrd = function () {
    var val = this.$value;
    return typeof val === 'string' ? baja.Ord.make(val) : val;
  };
  /**
   * Returns a promise that resolves to the widget's display name.
   *
   * @returns {Promise.<string>} A promise that resolves to the widget's display name.
   */


  ServletViewWidget.prototype.toDisplayName = function () {
    return doRequire('profileInfo').then(function (profileInfo) {
      var list = profileInfo.getViewInfo().list,
          item = _.findWhere(list, {
        def: true
      }) || _.first(list);

      return item.displayName;
    })["catch"](_.constant(''));
  };
  /**
   * Returns the TransformOperations of this view
   *
   * @returns {Promise.<Array.<module:nmodule/export/rc/TransformOperation>>}
   */


  ServletViewWidget.prototype.getTransformOperations = function () {
    var that = this;
    return asyncUtils.doRequire('nmodule/webEditors/rc/transform/ServletViewTransformOperationProvider').then(function (ServletViewTransformOperationProvider) {
      return new ServletViewTransformOperationProvider().getTransformOperations(that);
    });
  };

  return ServletViewWidget;
});
