/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/options/MruButton
 */
define(['log!nmodule.webEditors.rc.wb.options.MruButton', 'bajaux/commands/Command', 'bajaux/events', 'bajaux/util/CommandButton', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/wb/options/MruOptions'], function (log, Command, events, CommandButton, Promise, _, CompositeEditor, RelevantStringEditor, CompositeBuilder, MruOptions) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
      logError = log.severe.bind(log);
  /**
   * Command to show the MRU options popup.
   *
   * @inner
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:nmodule/webEditors/rc/wb/options/MruButton} mruButton
   */

  var MruCommand = function MruCommand(mruButton) {
    Command.call(this, {
      icon: 'module://icons/x16/recent.png',
      func: function func() {
        mruButton.getBuilder().getEditorFor('string').$getTextElement().trigger('show-dropdown');
      }
    });
  };

  MruCommand.prototype = Object.create(Command.prototype);
  MruCommand.prototype.constructor = MruCommand;
  /**
   * Editor for retrieving and displaying user-specific most-recently-used
   * values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/webEditors/rc/wb/options/MruButton
   */

  var MruButton = function MruButton() {
    CompositeEditor.apply(this, arguments);
  };

  MruButton.prototype = Object.create(CompositeEditor.prototype);
  MruButton.prototype.constructor = MruButton;
  /**
   * Convert the currently loaded MRU data into a datalist to be shown on screen
   * when the button is clicked.
   * @private
   * @returns {Promise.<Object>}
   */

  MruButton.prototype.$retrieveDatalist = function () {
    var that = this,
        obj = that.value(),
        toDisplayString = obj && obj.toDisplayString || _.identity,
        history = that.$mru.getHistory();
    return Promise.all(history.map(function (his) {
      return Promise["try"](function () {
        return toDisplayString(his);
      })["catch"](function (err) {
        logError(err);
        return his;
      });
    })).then(function (displayStrings) {
      var datalist = {};
      displayStrings.forEach(function (displayString, i) {
        datalist[history[i]] = displayString;
      });
      return datalist;
    });
  };
  /**
   * Build a CommandButton and a datalist popup.
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */


  MruButton.prototype.makeBuilder = function () {
    var that = this,
        builder = new CompositeBuilder({
      removeOnDestroy: false
    });

    builder.getKeys = function () {
      return ['button', 'string'];
    };

    builder.getConfigFor = function (key) {
      switch (key) {
        case 'button':
          return {
            type: CommandButton
          };

        case 'string':
          return that.$retrieveDatalist().then(function (datalist) {
            return {
              type: RelevantStringEditor,
              properties: {
                'relevant-filter': false,
                'relevant-force': true,
                datalist: datalist,
                hideText: true
              }
            };
          });
      }
    };

    builder.getValueFor = function (key) {
      switch (key) {
        case 'button':
          return new MruCommand(that);

        case 'string':
          return '';
      }
    };

    builder.getDomFor = function (key) {
      return that.jq().children('.key-' + key);
    };

    return builder;
  };
  /**
   * When a new value is selected from the datalist popup, pass it to the
   * `select` function on the loaded value.
   *
   * @param {JQuery} dom
   */


  MruButton.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('MruButton');
    dom.html('<button type="button" class="key-button ux-btn-tb"/>' + '<span class="key-string"/>');
    dom.on(MODIFY_EVENT, '.key-string', function (e, ed) {
      ed.read().then(function (str) {
        var value = that.value();

        if (typeof value.select === 'function') {
          return value.select(str);
        }
      })["catch"](logError);
    });
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  /**
   * An `MruButton` receives an object literal containing, at a minimum, the
   * MRU data to be shown, and callbacks describing how to display and handle
   * selected values.
   *
   * @param {Object} obj
   * @param {module:nmodule/webEditors/rc/wb/options/MruOptions} obj.options
   * @param {Function} [obj.select] a function to receive each string as it is
   * selected from the datalist popup
   * @param {Function} [obj.toDisplayString] a function to convert each value
   * from the MRU history into a string for display. Can return a string or a
   * Promise.
   * @returns {Promise}
   */


  MruButton.prototype.doLoad = function (obj) {
    var that = this,
        options = obj && obj.options;

    if (!(options instanceof MruOptions)) {
      throw new Error('options required');
    }

    that.$mru = obj.options;
    return CompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      //prevent BaseWidget#requestFocus from accidentally showing dropdown
      that.jq().find('input').addClass('nofocus').attr('tabindex', '-1');
    });
  };
  /**
   * Get the currently loaded MRU options.
   * @returns {module:nmodule/webEditors/rc/wb/options/MruOptions}
   */


  MruButton.prototype.getOptions = function () {
    return this.$mru;
  };
  /**
   * Set the currently loaded MRU options. This will rebuild the datalist popup
   * so it will be up to date on next click.
   * @param {module:nmodule/webEditors/rc/wb/options/MruOptions} options
   * @returns {Promise}
   */


  MruButton.prototype.setOptions = function (options) {
    this.$mru = options;
    var builder = this.getBuilder();
    return builder.destroyFor('string').then(function () {
      return builder.buildAll();
    });
  };

  return MruButton;
});
