/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/model/folderMgrModelMixin
 */
define(['baja!', 'underscore', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mgr/model/folderComponentSourceMixin', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/table/model/Row', 'nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource', 'log!nmodule.webEditors.rc.wb.mgr.model.folderMgrModelMixin', 'baja!baja:Folder'], function (baja, _, Promise, DepthSubscriber, addFolderComponentSourceMixin, mixinUtils, Row, ContainerComponentSource, log) {
  "use strict";

  var applyMixin = mixinUtils.applyMixin,
      hasMixin = mixinUtils.hasMixin,
      hasFolderComponentSource = _.partial(hasMixin, _, 'FOLDER_SOURCE'),
      hasSubscriberMixin = _.partial(hasMixin, _, 'subscriber'),
      FOLDER_TYPE = baja.lt('baja:Folder'),
      MIXIN_NAME = 'FOLDER_MGR_MODEL',
      logFine = log.fine.bind(log);

  function resolveFolderType(type) {
    return baja.importTypes([String(type)]).then(_.first);
  }
  /**
   * Test whether the component source has folder support mixed in and whether
   * the `AllDescendantsCommand` has set it to a flattened state.
   *
   * @param source
   * @returns {boolean}
   */


  function isFlattenedFolderSource(source) {
    return !!(hasFolderComponentSource(source) && source.isFlattened());
  }
  /**
   * For the subject component of a given row 'ancestorRow', test whether the the subject
   * of 'descendantRow' is a descendant of that component.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} ancestorRow
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} descendantRow
   *
   * @returns {boolean}
   */


  function isRowSubjectDescendantOf(ancestorRow, descendantRow) {
    var descendant = descendantRow.getSubject(),
        ancestor = ancestorRow.getSubject();

    while (descendant) {
      if (descendant === ancestor) {
        return true;
      }

      descendant = descendant.getParent();
    }

    return false;
  } ////////////////////////////////////////////////////////////////
  // FolderMgr Mix-In
  ////////////////////////////////////////////////////////////////

  /**
   * A mixin to add some extra folder related functionality to the model of a manager
   * view.
   *
   * @alias module:webEditors/rc/wb/mgr/model/folderMgrModelMixin
   *
   * @mixin
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} target The
   * manager model instance that will have the mixin applied to it.
   * @param {Object} params
   * @param {string|Type} params.folderType The folder type that will be created upon a call to the
   * exported `createFolder()` function. This can be a string type spec or a BajaScript Type.
   */


  var exports = function exports(target, params) {
    params = params || {};

    if (!applyMixin(target, MIXIN_NAME)) {
      return;
    }

    var folderType = params.folderType,
        source;

    if (!folderType) {
      throw new Error('Parameters must specify a type spec string or Type instance for a folder type');
    }

    source = target.getComponentSource(); // If the source is a container component source, we'll mix folder support into that
    // too, which provides the functionality that the 'all descendants' command needs to load
    // in the flat folder structure and return the flattened array of components.

    if (source instanceof ContainerComponentSource) {
      addFolderComponentSourceMixin(source, {
        folderType: folderType
      });
    }

    var superInsertRows = target.insertRows,
        superGetValueAt = target.getValueAt;
    /**
     * Creates a new folder instance with the given name. This will create a new
     * instance of the type returned by `getFolderType()`. The passed in name should
     * be unescaped.
     *
     * @param {string} name
     * @returns {Promise}
     */

    target.createFolder = function (name) {
      var that = this;
      return resolveFolderType(folderType).then(function (type) {
        if (!type.is(FOLDER_TYPE)) {
          throw new Error('Type is not a BFolder.');
        }

        return that.addInstances([baja.$(type)], [name + '?']);
      });
    };
    /**
     * Extends the target model with a function to return the folder type specified when the
     * mixin was applied.
     *
     * @returns {Type|String}
     */


    target.getFolderType = function () {
      return folderType;
    };
    /**
     * Reload the model for a change in the required folder depth. This is intended
     * to be called by the `AllDescendantsCommand` after it has been toggled.
     *
     * @private
     * @returns {Promise.<*>}
     */


    target.reloadForNewFolderDepth = function () {
      var that = this,
          currentCount = that.getRows().length,
          source = that.getComponentSource(),
          rows;
      rows = _.map(source.getComponents(), function (c) {
        return that.makeRow(c);
      });
      return (currentCount ? that.removeRows(0, currentCount) : Promise.resolve()).then(function () {
        return that.insertRows(rows, 0);
      });
    };
    /**
     * When inserting a new row in a flattened view, we need to insert the row
     * after the last descendant component of the new component's last sibling.
     * For a given sibling row, this will look through the array of rows and
     * return the row corresponding to the last descendant. If there are no
     * descendants, this will return the sibling row. The new row should be
     * inserted immediately after the row returned by this function.
     *
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} sibling - the row for the new component's sibling
     * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows - the array of rows currently in the model
     * @returns {module:nmodule/webEditors/rc/wb/table/model/Row}
     */


    function findLastDescendantRowOfSibling(sibling, rows) {
      var index = _.indexOf(rows, sibling);

      while (index + 1 < rows.length) {
        if (isRowSubjectDescendantOf(sibling, rows[index + 1])) {
          index++;
        } else {
          break;
        }
      }

      return rows[index];
    }
    /**
     * Override of the `insertRows()` function that provides the facility to insert a new row at
     * the correct position in the table, for use in the situation where the model is showing
     * a flat hierarchy of folders and leaf components. If we find that a nested folder has
     * a new child component, we will want to insert it at the end of the component's siblings
     * within that folder, not as the last item in the whole table.
     *
     * @override
     * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} toInsert The rows to be inserted.
     * @param {Number} [index] - the optional index at which to insert the new rows.
     *
     * @returns {Promise}
     */


    target.insertRows = function (toInsert, index) {
      var that = this,
          source = that.getComponentSource(); // Ensure we have Row instances if 'toInsert' is an array of Components.

      toInsert = _.map(toInsert, function (row) {
        return row instanceof Row ? row : that.makeRow(row);
      });

      if (index === undefined && isFlattenedFolderSource(source) && that.getRows().length) {
        //
        // If the model is toggled to show a flat folder hierarchy, for each
        // new component, we need to look through the array of existing rows
        // to decide where the row for the new component should be inserted.
        //
        return toInsert.reduce(function (prom, newRow) {
          return prom.then(function () {
            var index,
                insertAfterRow,
                currentRows = that.getRows(),
                newRowParent = newRow.getSubject().getParent(); // Look to see if the parent component has any children already in the
            // table. If so, we'll want to insert the new component after it's
            // last sibling *and* all of its descendants if the last sibling is
            // a folder.

            insertAfterRow = _.find(currentRows.slice().reverse(), function (row) {
              return row.getSubject().getParent() === newRowParent;
            });

            if (insertAfterRow && insertAfterRow.getSubject().getType().is(FOLDER_TYPE)) {
              // We have found the last sibling of the new component, now we need to find
              // any descendants of it, and insert it after the last one.
              insertAfterRow = findLastDescendantRowOfSibling(insertAfterRow, currentRows);
            }

            if (!insertAfterRow) {
              // If we didn't find any siblings, then we'll just want to find the
              // index of the new component's parent in the table, and insert the
              // row immediately after that.
              insertAfterRow = _.find(currentRows, function (row) {
                return row.getSubject() === newRowParent;
              });
            }

            if (insertAfterRow) {
              index = _.indexOf(currentRows, insertAfterRow) + 1;
            }

            return superInsertRows.call(that, [newRow], index);
          });
        }, Promise.resolve());
      } else {
        // Not flattened or the required index has been provided, so just call the
        // base implementation.
        return superInsertRows.call(that, toInsert, index);
      }
    };
    /**
     * Set or unset the `all descendants` functionality for the model.
     *
     * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr - the manager view using this model instance.
     * @param {Boolean} flattened - true if the 'all descendants' command is selected and the
     * model should show a flattened view of the folders, or false to indicate that the command
     * is unselected, and the model should just show the direct children of the root container.
     *
     * @returns {Promise.<*>}
     */


    target.setAllDescendants = function (mgr, flattened) {
      var source = target.getComponentSource(),
          subscriber = hasSubscriberMixin(mgr) ? mgr.getSubscriber() : null;

      if (source && hasFolderComponentSource(source)) {
        // If the manager is using a depth subscriber, notify the folder source
        // mixin code of the required depth, so it can load the deepest folders'
        // contents down to the same relative depth.
        if (subscriber instanceof DepthSubscriber) {
          source.setComponentDepth(subscriber.getDepth());
        }

        return source.setFlattened(flattened);
      }

      return Promise.reject(new Error('Component source must have the folder source mixin applied.'));
    };
    /**
     * override the target model's implementation of getValueAt.
     * returns blank for any column that errors
     * (ie the folder has no value for that column).
     *
     * @param {Number} x column index
     * @param {Number} y row index
     * @returns {Promise} promise to be resolved with the value
     */


    target.getValueAt = function (x, y) {
      var that = this;
      return Promise["try"](function () {
        return superGetValueAt.apply(that, [x, y]);
      })["catch"](function (error) {
        logFine('no value found for: ' + that.$columns[x].getName() + // column name
        ' in: ' + that.$rows[y].getSubject().getType() + // the row type
        ', error: ' + error);
        return '';
      });
    };
  };

  return exports;
});
