/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */
define(['underscore', 'Promise', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn'], function (_, Promise, MgrTypeInfo, NameMgrColumn) {
  'use strict';
  /**
   * Get the subjects from the selected rows in the given table.
   */

  function getSelectedRows(table) {
    return _.invoke(table.getSelectedRows(), 'getSubject');
  }
  /**
   * API Status: **Private**
   *
   * Utilities for working with bajaux Managers. The exported functions are currently intended
   * for internal manager subclasses only, for things that we don't necessarily want exposed on
   * the Manager API at the moment.
   *
   * @exports nmodule/webEditors/rc/wb/mgr/mgrUtils
   */


  var exports = {};
  /**
   * Return the `ListSelection` instance for the main table of the manager.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */

  exports.getMainTableSelection = function (manager) {
    return manager.getMainTable().$getSelection();
  };
  /**
   * Get an array of the currently selected subjects in the manager's main table.
   * For each of the selected rows, this will call the `Row`'s getSubject() function,
   * returning an array of the resulting subjects.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @Returns {Array.<*>}
   */


  exports.getMainTableSelectedSubjects = function (manager) {
    return getSelectedRows(manager.getMainTable());
  };
  /**
   * Return the `ListSelection` instance for the learn table of the manager.
   * The manager must have the `MgrLearn` mixin applied.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} manager
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */


  exports.getLearnTableSelection = function (manager) {
    return manager.getLearnTable().$getSelection();
  };
  /**
   * Get an array of the currently selected subjects in the manager's discovery table.
   * For each of the selected rows, this will call the `Row`'s getSubject() function,
   * returning an array of the resulting subjects.The manager must have the `MgrLearn`
   * mixin applied.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} manager
   * @Returns {Array.<*>}
   */


  exports.getLearnTableSelectedSubjects = function (manager) {
    return getSelectedRows(manager.getLearnTable());
  };
  /**
   * Find a Command in the manager's command group by its constructor.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @param {Function} ctor the constructor of the required Command.
   */


  exports.findCommand = function (manager, ctor) {
    return _.find(manager.getCommandGroup().getChildren(), function (c) {
      return c instanceof ctor;
    });
  };
  /**
   * Return an array of `MgrTypeInfo` instances for the given argument. If the
   * argument already contains one or more `MgrTypeInfo`s, these will be returned
   * unmodified. Any other type (e.g. string or baja.Type) will be passed to the
   * `MgrTypeInfo.make()` function. A non-array argument will resolve to an array
   * with a single `MgrTypeInfo` instance.
   *
   * @param {*} arg an array or single value to be converted to an array of `MgrTypeInfo`.
   *
   * @returns {Promise<Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>>}
   */


  exports.toMgrTypeInfos = function (arg) {
    if (!_.isArray(arg)) {
      arg = [arg];
    }

    return Promise.all(arg.map(function (t) {
      return t instanceof MgrTypeInfo ? t : MgrTypeInfo.make(t);
    }));
  };
  /**
   * Called when adding a discovery item into the main table, this will take
   * the newly created component and `Row`, and set the proposed initial
   * values on it. This is done by passing the discovery node to the concrete
   * manager, and letting it return an object containing the proposed values,
   * based on whatever information it stored in the learn model.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} mgr the manager
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model the manager model
   * for the main table
   * @param {*} discovery the discovery object, obtained from the selected row
   * in the table of discovered objects.
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row the newly created Row instance
   *
   * @returns {Promise}
   */


  exports.setProposedDiscoveryValues = function (mgr, model, discovery, row) {
    return Promise.resolve(mgr.getProposedValuesFromDiscovery(discovery, row.getSubject())).then(function (result) {
      var proposals = _.defaults(result, {
        values: {}
      }); // keep the proposed name if it's already been changed in the editor


      if (NameMgrColumn.hasUserDefinedValue(row)) {
        proposals.name = row.data('proposed.__name');
      }

      return Promise.all(Object.keys(proposals.values).map(function (prop) {
        var col = model.getColumn(prop);
        return col ? col.propose(proposals.values[prop], row) : Promise.reject(new Error('Unknown column ' + prop));
      })).then(function () {
        var nameCol;

        if (proposals.hasOwnProperty('name') && proposals.name !== row.getSubject().getName()) {
          nameCol = model.getColumn('__name');

          if (nameCol) {
            return nameCol.propose(proposals.name, row);
          }
        }
      });
    });
  };

  return exports;
});
