/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand
 */
define(['baja!', 'lex!webEditors', 'bajaux/commands/Command', 'dialogs', 'Promise', 'underscore'], function (baja, lexs, Command, dialogs, Promise, _) {
  'use strict';

  var webEditorsLex = lexs[0],
      CONFIRM_MSG = webEditorsLex.getSafe('commands.mgr.remove.confirm');

  function getSelection(manager) {
    return manager.getMainTable().$getSelection();
  }
  /**
   * Command for removing the currently selected rows in a Manager view.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   */


  var RemoveCommand = function RemoveCommand(manager) {
    var that = this;
    Command.call(that, {
      module: 'webEditors',
      lex: 'commands.mgr.remove',
      enabled: false,

      /**
       * Shows an "are you sure" message, then deletes the currently selected
       * rows.
       *
       * @alias module:nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        // eslint-disable-next-line promise/avoid-new
        return new Promise(function (resolve, reject) {
          dialogs.showOkCancel(CONFIRM_MSG).ok(function () {
            var selectedRows = manager.getMainTable().getSelectedRows(),
                mgrModel = manager.getModel(),
                source = mgrModel.getComponentSource();
            source.removeComponents(_.map(selectedRows, function (row) {
              return row.getSubject();
            })).then(resolve, function (err) {
              dialogs.showOk(err);
              reject(err);
            });
          }).cancel(resolve);
        });
      }
    });
    manager.on('loaded', function () {
      var selection = getSelection(manager);
      selection.on('changed', function () {
        that.setEnabled(!selection.isEmpty());
      });
    });
  };

  RemoveCommand.prototype = Object.create(Command.prototype);
  RemoveCommand.prototype.constructor = RemoveCommand;

  RemoveCommand.prototype.setEnabled = function (enabled) {
    //TODO: remove readonly checks after password rpc works offline
    return Command.prototype.setEnabled.call(this, !baja.isOffline() && enabled);
  };

  return RemoveCommand;
});
