/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/commands/NewCommand
 */
define(['baja!', 'lex!webEditors', 'bajaux/commands/Command', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/PropertySheet', 'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel', 'nmodule/webEditors/rc/wb/mgr/model/columns/TypeMgrColumn'], function (baja, lexs, Command, Promise, _, feDialogs, PropertySheet, BatchComponentEditor, MgrTypeInfo, MgrModel, TypeMgrColumn) {
  'use strict';

  var webEditorsLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * To query the user for what type of new components to create, and how
   * many, a PropertySheet will be shown in a dialog. This component contains
   * the necessary fields to load into that PropertySheet.
   *
   * @inner
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} newTypes an
   * array of `MgrTypeInfo` instances, for the types that will be presented to the
   * user to choose from.
   * @returns {baja.Component}
   */

  function makeComponent(newTypes) {
    var newTypesEnum = baja.DynamicEnum.make({
      range: baja.EnumRange.make({
        ordinals: _.map(newTypes, function (type, i) {
          return i;
        }),
        tags: _.map(_.invoke(newTypes, 'getDisplayName'), baja.SlotPath.escape)
      })
    }),
        comp = baja.$('baja:Component', {
      newTypes: newTypesEnum,
      count: baja.Integer.make(1),
      displayNames: baja.NameMap.make({
        newTypes: webEditorsLex.get('commands.mgr.add.type'),
        count: webEditorsLex.get('commands.mgr.add.count')
      })
    });
    comp.setFacets({
      slot: 'count',
      facets: baja.Facets.make({
        min: 1,
        max: 100
      })
    });
    comp.setFlags({
      slot: 'displayNames',
      flags: baja.Flags.HIDDEN
    }); // noinspection JSValidateTypes

    return comp;
  }
  /**
   * When loading up the `BatchComponentEditor`, a secondary `MgrModel` will be
   * constructed solely for the components being edited. After editing the
   * components, they will be applied back to the original `MgrModel`.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr the `Manager` to
   * which we're adding components
   * @param {Object} params
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} params.typeInfo the `MgrTypeInfo`
   * representing the type of the components we're creating
   * @param {Number} params.count how many new components to create
   * @returns {Promise} promise to be resolved with a new
   * `MgrModel` containing the new instances to edit
   */


  function makeSubMgrModel(mgr, params) {
    var mgrModel = mgr.getModel(),
        origContainer = mgr.value(),
        source = baja.$('baja:Folder'),
        typeInfo = params.typeInfo,
        uniqueName = origContainer.getUniqueName(typeInfo.toSlotName()),
        count = params.count,
        slotNames = [],
        slotName,
        slotNum = uniqueName.match(/\d*$/)[0] || 0,
        i;
    uniqueName = uniqueName.replace(/\d*$/, '');

    for (i = 0; i < count; i++) {
      while (origContainer.has(slotName = uniqueName + (slotNum || ''))) {
        slotNum++;
      }

      slotNames.push(slotName);
      slotNum++;
    }

    return Promise.all(slotNames.map(function (slotName) {
      return Promise.resolve(mgrModel.newInstance(typeInfo)).then(function (instance) {
        return source.add({
          slot: slotName,
          value: instance
        });
      });
    })).then(function () {
      var newTypes = mgrModel.getNewTypes(),
          model;
      model = new MgrModel({
        componentSource: source,
        columns: mgrModel.getEditableColumns()
      });

      _.each(model.getRows(), function (row) {
        TypeMgrColumn.setAvailableTypes(row, newTypes);
      });

      return model;
    });
  }
  /**
   * Present the user with a prompt asking what type of components to add and
   * how many.
   *
   * @inner
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} newTypes the
   * new types to choose from
   * @param {baja.Component} [ordBase]
   * @returns {Promise} promise to be resolved with an object with
   * `typeSpec` and `count` properties, or `null` if the prompt was canceled
   */


  function promptForParams(newTypes, ordBase) {
    MgrTypeInfo.markDuplicates(newTypes);
    var comp = makeComponent(newTypes);
    return feDialogs.showFor({
      value: comp,
      type: PropertySheet,
      properties: {
        showHeader: false,
        showControls: false,
        ordBase: ordBase
      }
    }).then(function (diff) {
      if (diff) {
        return diff.apply(comp).then(function () {
          return {
            typeInfo: newTypes[comp.get('newTypes').getOrdinal()],
            count: comp.get('count').valueOf()
          };
        });
      }
    });
  }
  /**
   * After determining what type of components to add and how many, show a
   * `BatchComponentEditor` allowing the user to make changes to the new
   * instances.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr
   * @param {Object} params
   * @param {String} params.typeSpec
   * @param {Number} params.count
   * @param {baja.Component} [ordBase]
   * @returns {Promise} promise to be resolved with an array of
   * new `Component` instances, or `null` if `BatchComponentEditor` dialog was
   * canceled
   */


  function promptForNewComponents(mgr, params, ordBase) {
    return makeSubMgrModel(mgr, params).then(function (model) {
      return feDialogs.showFor({
        value: model,
        properties: {
          ordBase: ordBase
        },
        type: BatchComponentEditor
      });
    }).then(function (model) {
      return model && _.map(model.getRows(), function (row) {
        return row.getSubject();
      });
    });
  } ////////////////////////////////////////////////////////////////
  // NewCommand
  ////////////////////////////////////////////////////////////////

  /**
   * Command for adding new components to a Manager view.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/mgr/commands/NewCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   */


  var NewCommand = function NewCommand(manager) {
    Command.call(this, {
      module: 'webEditors',
      lex: 'commands.mgr.new',

      /**
       * Add new components to the Manager view. First shows a dialog asking
       * for the type of component to add and how many, then a manager editor
       * allowing the components to be configured before adding.
       *
       * @alias module:nmodule/webEditors/rc/wb/mgr/commands/NewCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        var mgrModel = manager.getModel();
        return manager.getOrdBase().then(function (ordBase) {
          return promptForParams(mgrModel.getNewTypes(), ordBase).then(function (params) {
            return params && promptForNewComponents(manager, params, ordBase);
          }).then(function (comps) {
            return comps && mgrModel.addInstances(comps);
          });
        })["catch"](feDialogs.error);
      }
    });
  };

  NewCommand.prototype = Object.create(Command.prototype);
  NewCommand.prototype.constructor = NewCommand;

  NewCommand.prototype.setEnabled = function (enabled) {
    //TODO: remove readonly checks after password rpc works offline
    return Command.prototype.setEnabled.call(this, !baja.isOffline() && enabled);
  };

  return NewCommand;
});
