/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/commands/MatchCommand
 */
define(['Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel', 'nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource'], function (Promise, _, Command, feDialogs, mixinUtils, BatchComponentEditor, mgrUtils, MgrModel, ArrayComponentSource) {
  'use strict';

  var getMainTableSelection = mgrUtils.getMainTableSelectedSubjects,
      getLearnTableSelection = mgrUtils.getLearnTableSelectedSubjects,
      setProposedValues = mgrUtils.setProposedDiscoveryValues;
  /**
   * Display the `BatchComponentEditor` for the component being matched.
   * This will already have had the proposed values from the matched discovery
   * object set at the point this is called.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr - the manager
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} editorModel - the manager model for the batch editor
   */

  function editInstance(mgr, editorModel) {
    var baseComponent;
    return mgr.getOrdBase().then(function (ordBase) {
      baseComponent = ordBase;
      return feDialogs.showFor({
        value: editorModel,
        properties: {
          ordBase: baseComponent
        },
        type: BatchComponentEditor
      });
    });
  }
  /**
   * Match the discovered item to the component in the database. This will
   * set the proposed values from the discovery object in the same way as
   * the `AddCommand` does.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr - the manager
   * @param {baja.Component} comp - the existing database component
   * @param {*} discovery - the discovery item to be matched to the existing component.
   */


  function matchSelection(mgr, comp, discovery) {
    var databaseModel = mgr.getModel(),
        editModel,
        row;
    editModel = new MgrModel({
      componentSource: new ArrayComponentSource([comp]),
      columns: databaseModel.getColumns()
    });
    row = _.first(editModel.getRows());
    return setProposedValues(mgr, databaseModel, discovery, row).then(function () {
      return editInstance(mgr, editModel);
    });
  }
  /**
   * Command to match a discovered object to an existing database object. This requires
   * one subject to be selected in each of the tables.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/mgr/commands/MatchCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr
   */


  var MatchCommand = function MatchCommand(mgr) {
    Command.call(this, {
      module: 'webEditors',
      lex: 'commands.mgr.match',
      enabled: false,

      /**
       * Match the selected object in the discovery table to the selected object
       * in the database table.
       *
       * @alias module:nmodule/webEditors/rc/wb/mgr/commands/MatchCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        var mainSelection, learnSelection;

        if (!mixinUtils.hasMixin(mgr, 'MGR_LEARN')) {
          return Promise.resolve();
        }

        mainSelection = getMainTableSelection(mgr);
        learnSelection = getLearnTableSelection(mgr);

        if (mainSelection.length === 1 && learnSelection.length === 1) {
          return matchSelection(mgr, mainSelection[0], learnSelection[0])["catch"](feDialogs.error);
        } else {
          return Promise.reject(new Error('Must have one item selected in the learn and database tables'));
        }
      }
    });
    this.$mgr = mgr;
  };

  MatchCommand.prototype = Object.create(Command.prototype);
  MatchCommand.prototype.constructor = MatchCommand;
  /**
   * Function called when the selection in the main table or this discovery
   * table changes. The match command should only be enabled when one item
   * is selected in each.
   *
   * @private
   */

  MatchCommand.prototype.tableSelectionChanged = function () {
    this.setEnabled(getMainTableSelection(this.$mgr).length === 1 && getLearnTableSelection(this.$mgr).length === 1);
  };

  return MatchCommand;
});
