/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 *
 * Type providing functionality to support the learn table on a manager view. This is
 * intended to mix in functionality that is user interface specific, and should be
 * kept out of the `MgrLearn` mixin; i.e. this is intended for stuff that is private to
 * the view's implementation, and will not be part of the exposed learn API.
 *
 * This is not intended to be directly mixed-in by a manager implementation. The exported
 * functions will be applied to the manager at the same time that the `MgrLearn` mixin
 * is applied, and will be done automatically, so there's no need for concrete managers
 * to do it themselves.
 *
 * @module nmodule/webEditors/rc/wb/mgr/MgrLearnTableSupport
 * @mixin
 * @extends module:nmodule/webEditors/rc/wb/mgr/MgrLearn
 */
define(['baja!', 'lex!webEditors', 'log!nmodule.webEditors.rc.wb.mgr.MgrLearnTableSupport', 'bajaux/Widget', 'Promise', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/job/JobBar', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu', 'nmodule/webEditors/rc/wb/table/tree/TreeTable', 'hbs!nmodule/webEditors/rc/wb/mgr/template/Manager', 'nmodule/js/rc/jquery/split-pane/split-pane', 'css!nmodule/js/rc/jquery/split-pane/split-pane'], function (baja, lexs, log, Widget, Promise, $, _, fe, JobBar, mgrUtils, mixinUtils, DefaultTableContextMenu, TreeTable, tplManager) {
  'use strict';

  var webEditorsLex = lexs[0],
      logError = log.severe.bind(log),
      logFine = log.fine.bind(log),
      MIXIN_NAME = '$MGR_LEARN_TABLE';

  var exports = function exports(target, params) {
    var tableConstructor = params.tableCtor,
        superDoInitialize = target.doInitialize,
        superDoLayout = target.doLayout,
        superDoLoad = target.doLoad,
        superDoDestroy = target.doDestroy,
        getLearnTableSelection = mgrUtils.getLearnTableSelection,
        getMainTableSelection = mgrUtils.getMainTableSelection,
        SPLIT_PANE_THROTTLE_MS = 200,
        UPDATED_MODEL_THROTTLE_MS = 250;

    if (!mixinUtils.applyMixin(target, MIXIN_NAME)) {
      return;
    } ////////////////////////////////////////////////////////////////
    // Utility Functions
    ////////////////////////////////////////////////////////////////


    function isLearnModeCommandSelected(mgr) {
      var cmd = mgr.$getLearnModeCommand();
      return cmd && cmd.isSelected();
    }

    function setLearnModel(mgr, model) {
      mgr.$learnModel = model;
    }

    function getLearnModel(mgr) {
      return mgr.$learnModel;
    } ////////////////////////////////////////////////////////////////
    // Initialize / Layout / Load / Destroy
    ////////////////////////////////////////////////////////////////

    /**
     * Extend the base Manager's doInitialize() function, by adding the
     * discovery tables and the title elements (with the object counts).
     * This will also create the job bar widget that is used during discovery.
     *
     * @param {JQuery} dom
     * @param {Object} params
     * @returns {*|Promise}
     */


    target.doInitialize = function (dom, params) {
      var that = this;
      params = params || {};
      params = _.extend({
        html: exports.mgrLearnHtml()
      }, params);
      return superDoInitialize.call(that, dom, params).then(function () {
        return that.initializeJobBar(dom);
      });
    };
    /**
     * Initializes the JobBar widget.
     *
     * @param {JQuery} dom
     * @returns {Promise}
     */


    target.initializeJobBar = function (dom) {
      var jobBarEl = dom.find('.mgr-job-bar-container');

      if (!jobBarEl.length) {
        return Promise.resolve();
      }

      var jobBar = this.$jobBar = new JobBar();
      return jobBar.initialize(jobBarEl);
    };
    /**
     * Extends the base Manager type's doLayout() function to handle both the main
     * database table and the discovery table.
     *
     * @returns {Promise}
     */


    target.doLayout = function () {
      var that = this; // Call the super layout to set the bottom position of the database
      // table according to the action bar, then adjust the table heights.
      // We also adjust the position the split pane, but only if flagged to
      // do so.

      return Promise.resolve(superDoLayout.apply(that, arguments)).then(function () {
        if (that.$positionPanesOnLayout) {
          // This will resize the table heights when the positioning
          // is done.
          that.$positionSplitPane(isLearnModeCommandSelected(that));
        } else {
          resizeTableHeights(that);
        }
      });
    };
    /**
     * Initialize the event handlers on the main manager model. This will be called after
     * the user interface has been constructed.
     *
     * @private
     * @param mgr
     */


    function initializeMainModelEvents(mgr) {
      var model = mgr.getModel();
      mgr.$mainModelChanged = _.throttle(function () {
        if (!mgr.isDestroyed()) {
          disableIconsForExisting(mgr);
          updateDbObjectCount(mgr);
        }
      }, UPDATED_MODEL_THROTTLE_MS, {
        leading: false
      });
      model.on('rowsAdded', mgr.$mainModelChanged);
      model.on('rowsRemoved', mgr.$mainModelChanged);
    }
    /**
     * Initialize the event handlers on the learn model. This will be called after
     * the user interface has been constructed.
     *
     * @private
     * @param mgr
     */


    function initializeLearnModelEvents(mgr) {
      var learnModel = getLearnModel(mgr);
      mgr.$learnModelChanged = _.throttle(function () {
        if (!mgr.isDestroyed()) {
          disableIconsForExisting(mgr);
          updateDiscoveryObjectCount(mgr);
        }
      }, UPDATED_MODEL_THROTTLE_MS, {
        leading: false
      });
      learnModel.on('rowsAdded', mgr.$learnModelChanged);
      learnModel.on('rowsRemoved', mgr.$learnModelChanged);
    }
    /**
     * Set the selection change event handler on the two tables.
     * This will enable the add command when one or more items are
     * selected in the learn table. It will also notify the match
     * command of a selection change, so it can decide whether to
     * enable itself based on what is currently selected.
     *
     * @param mgr
     */


    function initializeTableSelections(mgr) {
      var mainSelection, learnSelection;

      function notifyAddCommand() {
        var addCmd = mgr.$getAddCommand();

        if (addCmd) {
          addCmd.setEnabled(!learnSelection.isEmpty());
        }
      }

      function notifyMatchCommand() {
        var matchCmd = mgr.$getMatchCommand();

        if (matchCmd) {
          matchCmd.tableSelectionChanged();
        }
      }

      learnSelection = getLearnTableSelection(mgr);
      learnSelection.on('changed', function () {
        notifyAddCommand();
        notifyMatchCommand();
      });
      mainSelection = getMainTableSelection(mgr);
      mainSelection.on('changed', function () {
        notifyMatchCommand();
      });
    }
    /**
     * Extension of the manager's load functionality. As well as loading the main table's
     * model, this will also load the model for the learn table.
     *
     * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
     * @returns {*}
     */


    target.doLoad = function (model) {
      var that = this,
          jq = that.jq();
      return Promise.resolve(that.makeLearnModel()).then(function (learnModel) {
        setLearnModel(that, learnModel);
        return superDoLoad.call(that, model);
      }).then(function () {
        return buildLearnTable(that);
      }).then(function () {
        initializeSplitPanes(that, jq);
        initializeDragAndDrop(that, jq);
        initializeMainModelEvents(that);
        initializeLearnModelEvents(that);
        initializeTableSelections(that);
        updateDbObjectCount(that);
        updateDiscoveryObjectCount(that);
        that.$toggleLearnPane(isLearnModeCommandSelected(that));
        return disableIconsForExisting(that);
      });
    };
    /**
     * Create and initialize the learn table for the manager. This will hook into the
     * table construction to allow the appropriate CSS to be set on the table. This
     * will also configure the show/hide menu for the new table.
     *
     * @private
     * @param mgr
     */


    function buildLearnTable(mgr) {
      var jq = mgr.jq(),
          tableContainer = jq.find('.discoveryTable').children('.tableContainer'),
          learnModel = mgr.getLearnModel(),
          initParams = {};
      initParams.buildCell = _.bind(target.buildDiscoveryTableCell, mgr);
      initParams.destroyCell = _.bind(target.destroyDiscoveryTableCell, mgr);
      initParams.finishBuildingRow = _.bind(target.finishDiscoveryTableRow, mgr);
      return fe.buildFor({
        dom: $('<table class="ux-table no-stripe"></table>').appendTo(tableContainer),
        type: tableConstructor,
        value: learnModel,
        initializeParams: initParams
      }).then(function (table) {
        new DefaultTableContextMenu(table).arm(jq, '.mgr-show-hide-menu-discovery');
      });
    }
    /**
     * Remove the event handlers fromm the learn model, and delete the properties
     * on the manager that relate to it.
     *
     * @private
     * @param mgr
     * @returns {Promise}
     */


    function disposeLearnModel(mgr) {
      var model = getLearnModel(mgr),
          promise;

      if (model) {
        model.removeAllListeners('columnsFlagsChanged');
        model.removeListener('rowsAdded', mgr.$learnModelChanged);
        model.removeListener('rowsRemoved', mgr.$learnModelChanged);
        promise = model.removeColumns(model.getColumns());
      } else {
        promise = Promise.resolve();
      }

      return promise.then(function () {
        if (model) {
          delete mgr.$learnModel;
        }

        delete mgr.$learnModelChanged;
      });
    }
    /**
     * Extend the manager's destroy functionality to clean up event handlers
     * relating to the discovery table and model.
     *
     * @returns {*|Promise}
     */


    target.doDestroy = function () {
      var that = this,
          jq = that.jq(),
          model = that.getModel();
      jq.find('.split-pane').off();

      if (model) {
        model.removeListener('rowsAdded', that.$mainModelChanged);
        model.removeListener('rowsRemoved', that.$mainModelChanged);
      }

      delete that.$mainModelChanged;
      return disposeLearnModel(that).then(function () {
        return superDoDestroy.apply(that, arguments);
      });
    }; ////////////////////////////////////////////////////////////////
    // Drag and Drop
    ////////////////////////////////////////////////////////////////

    /**
     * Initialize the event handling for the drag and drop events. This will call
     * functions to handle dragging rows from the learn table to the database table.
     *
     * @private
     * @param mgr the manager
     * @param dom the jquery object for the manager
     */


    function initializeDragAndDrop(mgr, dom) {
      dom.on('dragstart', '.mgr-discovery-row', handleLearnTableDragStart);
      dom.on('drop', '.mainTable', function (e) {
        return handleDrop(e, mgr);
      });
      dom.on('dragover', '.mainTable', handleDragOverMainTable);
    }
    /**
     * Event handler called when starting a drag from an item in the discovery table.
     * This will place some data on the dataTransfer object to indicate the drag operation.
     * When this is dropped, we shall invoke the AddCommand.
     *
     * @private
     * @param e the drag event
     */


    function handleLearnTableDragStart(e) {
      var row = $(e.target).data('mgr-discovered-row'),
          dataTransfer = e.originalEvent.dataTransfer; // NOTE 1: For this to work in the Workbench Java FX browser, it appears that it needs
      // to set any data on the event's dataTransfer object synchronously, before the event
      // handler returns. If it gets set asynchronously via a Promise, the data doesn't show
      // up in the drop event.
      // NOTE 2: The code for adding the items to the station is in the add command, as
      // items can be added without dragging and dropping. This means that we don't
      // actually place the discovery data in the DataTransfer object, just an indication
      // that something is being dragged. The AddCommand will pick up the discovery items
      // from the current list selection.

      if (row) {
        dataTransfer.setData('Text', JSON.stringify({
          mime: 'niagara/strings',
          data: JSON.stringify({
            kind: 'mgr-drag-op',
            source: 'mgr-learn'
          })
        }));
      }
    }
    /**
     * Event handler called when the item is over the main table.
     *
     * @param e the drag over event
     * @returns {boolean} returns false to allow a drop on to the table.
     */


    function handleDragOverMainTable(e) {
      e.originalEvent.dataTransfer.dropEffect = 'copy';
      return false;
    }
    /**
     * Event handler called when an item is dropped on the main table.
     * This will check that the source of the drag was one or more
     * rows in the discovery table, and invoke the operations to edit
     * and add those items as new rows in the main manager table.
     *
     * @private
     * @param e the drop event
     * @param mgr the manager instance
     */


    function handleDrop(e, mgr) {
      var dataTransfer = e.originalEvent.dataTransfer,
          json = dataTransfer.getData('Text'),
          dropped,
          data;

      if (json && (dropped = JSON.parse(json)) && dropped.mime === 'niagara/strings') {
        data = dropped.data && JSON.parse(dropped.data);

        if (data && data.kind === 'mgr-drag-op' && data.source === 'mgr-learn') {
          mgr.$getAddCommand().invoke()["catch"](logError);
        }
      }

      return false;
    } ////////////////////////////////////////////////////////////////
    // Table Content
    ////////////////////////////////////////////////////////////////

    /**
     * Update the object count text on one of the manager panes.
     *
     * @param model the model requiring the count to be updated; either database or discovery.
     * @param jq the jquery object for the element to contain the row count.
     */


    function updateRowCount(model, jq) {
      jq.text(webEditorsLex.get({
        key: 'mgr.titlePane.objects',
        args: [model.getRows().length]
      }));
    }
    /**
     * Update the count of objects in the title portion of the main database table pane.
     */


    function updateDbObjectCount(mgr) {
      var model = mgr.getModel();
      updateRowCount(model, mgr.jq().find('.main-table-object-count'));
    }
    /**
     * Update the count of objects in the title portion of the discovery table pane.
     */


    function updateDiscoveryObjectCount(mgr) {
      var model = mgr.getLearnModel();
      updateRowCount(model, mgr.jq().find('.discovery-table-object-count'));
    }

    function setIconOnDiscoveryRow(element, mgr, batch) {
      var dom = $(element),
          row = dom.data('mgr-discovered-row'),
          node = row && row.getSubject(),
          promise;

      if (node && (!node.isGroup || !node.isGroup())) {
        promise = mgr.getExisting(node, batch).then(function (exists) {
          return !!exists;
        });
      } else {
        promise = Promise.resolve(false);
      }

      return promise.then(function (existing) {
        dom.find('.IconEditor').toggleClass('mgr-learn-icon-disabled', existing);
      });
    }
    /**
     * Look through the rows in the learn table, and for any that match a component
     * in the station, disable the icon to show the item is already added.
     */


    function disableIconsForExisting(mgr) {
      var table = mgr.$getLearnTableElement(),
          batch = new baja.comm.Batch(),
          promises = [];
      table.find('.mgr-discovery-row').each(function (index, element) {
        promises.push(setIconOnDiscoveryRow(element, mgr, batch));
      });
      logFine('disableIconsForExisting: committing [' + promises.length + '] batched promises');
      return batch.commit(Promise.all(promises));
    } ////////////////////////////////////////////////////////////////
    // Table Layout
    ////////////////////////////////////////////////////////////////

    /**
     * Get the height of the table 'pane' area - that is the overall view height
     * minus the height of the action bar at the bottom.
     *
     * @returns {Number}
     */


    function getPaneAreaHeight(mgr) {
      var jq = mgr.jq(),
          managerHeight = jq.closest('.Manager').first().parent().outerHeight(),
          actionBarHeight = jq.find('.mgr-action-bar').outerHeight(),
          paneTop = jq.find('.mgr-pane-container').position().top;
      return managerHeight - actionBarHeight - paneTop;
    }
    /**
     * Function called to resize the divs containing the tables when the pane divider
     * is moved or the window size is changed. This needs to take into account a) the
     * divider position, b) the height of the command button container at the bottom
     * and c) the overall height of the view.
     *
     * @param mgr the `Manager` instance.
     */


    function resizeTableHeights(mgr) {
      var paneAreaHeight,
          mainTableHeight,
          discoveryTableHeight,
          jq = mgr.jq(),
          divider = jq.find('.mgr-split-pane-divider'),
          mainTable = jq.find('.mainTable'),
          discoveryTable = jq.find('.discoveryTable'),
          actionBar = jq.find('.mgr-action-bar'),
          tableContainer = jq.find('.mgr-pane-container');
      paneAreaHeight = getPaneAreaHeight(mgr);
      tableContainer.outerHeight(paneAreaHeight);
      mainTableHeight = actionBar.offset().top - mainTable.offset().top;
      mainTable.outerHeight(mainTableHeight);
      discoveryTableHeight = divider.offset().top - discoveryTable.offset().top;
      discoveryTable.outerHeight(discoveryTableHeight);
    }
    /**
     * Returns the TreeTable widget, used to display the discovered items.
     *
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#getLearnTable
     * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeTable} the discovery table
     * @since Niagara 4.6
     */


    target.getLearnTable = function () {
      // noinspection JSValidateTypes
      return Widget["in"](this.$getLearnTableElement().find('.tableContainer table'));
    };
    /**
     * Returns the jquery element for the discovery table.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$getLearnTableElement
     * @returns {JQuery}
     */


    target.$getLearnTableElement = function () {
      return this.jq().find('.discoveryTable');
    };
    /**
     * Returns the job bar widget, used when the setJob() function is called on the
     * manager.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$getJobBar
     * @returns {module:nmodule/webEditors/rc/wb/job/JobBar}
     */


    target.$getJobBar = function () {
      return this.$jobBar;
    };
    /**
     * Set up the split pane between the discovery and database tables. This will
     * resize the divs containing the tables in response to the events received when
     * the split pane divider is moved.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr
     * @param {JQuery} dom the jQuery object for the manager view's DOM.
     */


    function initializeSplitPanes(mgr, dom) {
      dom.find('.split-pane').each(function () {
        var that = this,
            pane = $(that);
        pane.splitPane();
        pane.on('dividerdragend', _.throttle(function (event) {
          if (event.target === that) {
            mgr.layout()["catch"](logError);
          }
        }, SPLIT_PANE_THROTTLE_MS));
      });
    }
    /**
     * Set the position of the split panes for the discovery and main tables. This will
     * be called in response to the `LearnModeCommand` being toggled, or the manager state
     * being restored following a load.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$positionSplitPane
     * @param {Boolean} show - true if the discovery pane is to be shown.
     */


    target.$positionSplitPane = function (show) {
      var that = this,
          jq = that.jq(),
          splitPane,
          size;

      if (jq.height() > 0) {
        // Fix the split pane position, such that when the learn table is shown, the splitter
        // divides the top and bottom tables equally. The table divs will then be resized
        // to accommodate those new heights. If the learn table is being hidden, set the height
        // of the first component to 0.
        splitPane = jq.find('.split-pane.horizontal-percent');
        size = show ? getPaneAreaHeight(that) / 2 : 0;

        _.defer(function () {
          if (!that.isDestroyed()) {
            splitPane.splitPane('firstComponentSize', size);
            resizeTableHeights(that);
          }
        });

        that.$positionPanesOnLayout = false;
      } else {
        // If we are loading the view and the $toggleLearnPane() function was called in
        // response to restoring the manager state, we might have to wait for the manager
        // widget to be laid out before we can calculate the required position for the
        // split pane. In this case, we'll set a flag to tell the layout method to call
        // this function, so it can try again when the UI is ready. The flag will be cleared
        // in the code above when doLayout() is called when the height > 0.
        that.$positionPanesOnLayout = true;
      }
    };
    /**
     * Function to show or hide the learn pane of the manager view. This involves a couple
     * of steps: first it needs to tell the split pane to collapse the first component, and
     * then needs to resize the table containers according to the new heights.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$toggleLearnPane
     * @param {Boolean} show boolean value used to show or hide the learn pane.
     */


    target.$toggleLearnPane = function (show) {
      var that = this,
          jq = that.jq(); // Hide or show both the top pane and the div used for the divider.

      jq.find('.mgr-pane-top').toggle(show);
      jq.find('.mgr-split-pane-divider').toggle(show);
      that.$positionSplitPane(show);

      _.defer(function () {
        if (!that.isDestroyed()) {
          resizeTableHeights(that);
        }
      });
    }; ////////////////////////////////////////////////////////////////
    // Cell/Row Construction & Destruction
    ////////////////////////////////////////////////////////////////

    /**
     * Return a promise to build a cell in the discovery table. This will by default
     * just delegate to the table.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#buildDiscoveryTableCell
     * @returns {Promise}
     */


    target.buildDiscoveryTableCell = function (column, row, dom) {
      return TreeTable.prototype.buildCell.call(this, column, row, dom);
    };
    /**
     * Return a promise to destroy a cell in the discovery table. This will by default
     * just delegate to the table.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#destroyDiscoveryTableCell
     * @returns {Promise}
     */


    target.destroyDiscoveryTableCell = function (column, row, dom) {
      return TreeTable.prototype.destroyCell.call(this, column, row, dom);
    };
    /**
     * Finish the discovery row before it is added to the table.
     * This is where the CSS classes and other attributes are set.
     * We need to indicate that the row should be draggable. It
     * will also set a class to indicate that the row is for the
     * discovery table, which could be useful in the drag and drop code.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#finishDiscoveryTableRow
     * @returns {Promise}
     */


    target.finishDiscoveryTableRow = function (row, dom) {
      dom.addClass('mgr-row mgr-discovery-row').prop('draggable', 'true').data('mgr-discovered-row', row);
      return Promise.resolve(dom);
    };
  };

  exports.mgrLearnHtml = function () {
    return tplManager({
      databaseTitle: webEditorsLex.get('mgr.titlePane.database'),
      discoveryTitle: webEditorsLex.get('mgr.titlePane.discovered')
    });
  };

  return exports;
});
