/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 *
 * Mixin providing folder related functions to be mixed in to a `Manager` if a
 * folder type is specified in the parameters to the constructor.
 *
 * @module nmodule/webEditors/rc/wb/mgr/MgrFolderSupport
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/MgrStateHandler', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/commands/AllDescendantsCommand'], function (baja, Promise, _, DepthSubscriber, mixinUtils, MgrStateHandler, mgrUtils, AllDescendantsCommand) {
  'use strict';

  var exports = function exports(target, params) {
    var MIXIN_NAME = 'MGR_FOLDER',
        applyMixin = mixinUtils.applyMixin,
        hasMixin = mixinUtils.hasMixin,
        findCommand = mgrUtils.findCommand;

    if (!applyMixin(target, MIXIN_NAME)) {
      return;
    }

    var hasSubscriberMixin = _.partial(hasMixin, _, 'subscriber'),
        hasFolderComponentSourceMixin = _.partial(hasMixin, _, 'FOLDER_SOURCE'),
        superBeforeMainTableLoaded = target.$beforeMainTableLoaded,
        superAfterMainTableLoaded = target.$afterMainTableLoaded;

    target.$folderType = params.folderType;
    /**
     * Get the folder type for this manager if one was specified at construction time.
     *
     * @returns {baja.Type} the folder type specified in the constructor parameters or undefined
     * if not specified.
     */

    function getFolderType() {
      return target.$folderType;
    }

    target.getFolderType = target.getFolderType || getFolderType;
    /**
     * Return a new temporary `Subscriber` for use by the `resubscribeForNewFolderDepth()`
     * function. If the `Manager` uses a `DepthSubscriber`, this will return a new one configured
     * for the same depth.
     *
     * @param {baja.Subscriber} subscriber
     * @returns {baja.Subscriber}
     */

    function makeTempSubscriber(subscriber) {
      return subscriber instanceof DepthSubscriber ? new DepthSubscriber(subscriber.getDepth()) : new baja.Subscriber();
    }
    /**
     * Called when the 'all descendants' command is toggled, or when a folder is added/
     * removed while the command is selected. The default implementation is to use the
     * manager's subscriber (if it uses the `subscriberMixin`) to either subscribe or
     * unsubscribe to the flattened set of folders.
     *
     * @returns {Promise.<*>}
     */


    function resubscribeForNewFolderDepth() {
      var model = target.getModel(),
          source = model && model.getComponentSource(),
          subscriber,
          tempSubscriber;

      if (hasSubscriberMixin(target) && source && hasFolderComponentSourceMixin(source)) {
        subscriber = target.getSubscriber();

        if (source.isFlattened()) {
          // If the source is currently flattened, we can just pass the folders to
          // subscriber as direct components whether the subscriber is a DepthSubscriber
          // or just a regular baja.Subscriber; either way, we should get the expected
          // results.
          return subscriber.subscribe(source.getFlattenedFolders());
        } else {
          // If the source is now un-flattened, we want to be subscribed to just the
          // root container and not the folders. This is made slightly more complicated
          // with the use of a DepthSubscriber. We can't just simply unsubscribe the
          // folders, as some of those folders and their children may be within the
          // overlapping depth relative to the root container, leading to some things
          // being unexpectedly unsubscribed when they fall within the configured depth of
          // the root.
          //
          // The simplest solution is to create a temporary subscriber (configured to
          // the same depth if the manager is using a DepthSubscriber) and use it to
          // subscribe the root container and its children to the appropriate depth. The
          // manager's subscriber will then be told to unsubscribe everything (the
          // temporary one holding the root and the appropriate level of children in
          // subscription). The main subscriber will then re-subscribe to the root,
          // which should involve no network calls as the temporary subscriber was
          // already keeping those components subscribed. The temporary subscriber can
          // then be discarded.
          tempSubscriber = makeTempSubscriber(subscriber);
          return tempSubscriber.subscribe(source.getContainer()).then(function () {
            return subscriber.unsubscribeAll();
          }).then(function () {
            return subscriber.subscribe(source.getContainer());
          }).then(function () {
            return tempSubscriber.unsubscribeAll();
          });
        }
      }

      return Promise.resolve();
    }

    target.resubscribeForNewFolderDepth = target.resubscribeForNewFolderDepth || resubscribeForNewFolderDepth;
    /**
     * Returns `true` if the Manager has a configured folder type and the 'all descendants'
     * command was selected when the manager state was last restored. If this returns `true`,
     * the component source will be loaded with the flattened folder structure before the
     * table model is created at load time.
     *
     * @private
     * @returns {boolean} true if the flattened folder hierarchy should be loaded.
     */

    target.$shouldRestoreFolderDescendants = function (deserializedState) {
      return !!(findCommand(this, AllDescendantsCommand) && MgrStateHandler.shouldRestoreAllDescendants(deserializedState));
    };
    /**
     * Override the method called when the model is created but before the main table
     * widget is loaded with the model. It will inspect the deserialized state to see
     * if the model should be flattened in order to restore the 'all descendants'
     * command.
     *
     * @private
     *
     * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model - the main table model
     * @param {Object} deserializedState - the state deserialized by `MgrStateHandler`.
     * @returns {Promise.<*>}
     */


    target.$beforeMainTableLoaded = function (model, deserializedState) {
      var that = this;
      return superBeforeMainTableLoaded.apply(that, arguments).then(function () {
        if (that.$shouldRestoreFolderDescendants(deserializedState)) {
          // If the 'all descendants' command was saved in the selected state
          // by the manager state handler, enable it on the model (loading the
          // required components at this point before creating the table DOM)
          // and tell it to update its rows. Doing this after loading the model
          // into the Table widget and recreating the table contents is just too
          // expensive.
          return restoreFolderDescendants(that, model);
        }
      });
    };

    function restoreFolderDescendants(manager, model) {
      return model.setAllDescendants(manager, true).then(function () {
        return model.reloadForNewFolderDepth();
      });
    }
    /**
     * Override the method called after the model has been loaded into the main
     * table widget. If the 'all descendants' command is being restored in the
     * selected state, the method will be called to re-subscribe to the new set
     * of components for the flattened hierarchy that will have been loaded into
     * the model by this point.
     *
     * @private
     *
     * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model - the main table model
     * @param {Object} deserializedState - the state deserialized by `MgrStateHandler`.
     * @returns {Promise.<*>}
     */


    target.$afterMainTableLoaded = function (model, deserializedState) {
      var that = this;
      return superAfterMainTableLoaded.apply(that, arguments).then(function () {
        if (that.$shouldRestoreFolderDescendants(deserializedState)) {
          return that.resubscribeForNewFolderDepth();
        }
      });
    };
  };

  return exports;
});
