/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara*/

/* jshint browser: true */
define(['module', 'baja!', 'Promise', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'underscore', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/servlets/views', 'nmodule/webEditors/rc/wb/menu/menuAgents', 'nmodule/webEditors/rc/wb/menu/newComponents', 'nmodule/webEditors/rc/wb/menu/Separator'], function (module, baja, Promise, Command, CommandGroup, _, asyncUtils, feDialogs, views, menuAgents, newComponents, Separator) {
  'use strict';

  var doRequire = asyncUtils.doRequire,
      NEW_COMMAND_GROUP_DISPLAY_NAME = '%lexicon(webEditors:menu.new.label)%',
      config = module.config();
  /**
   * Check to see if the HIDDEN flag is set on the slot.
   *
   * @inner
   * @param {baja.Slot} slot
   * @returns {boolean}
   */

  function visible(slot) {
    return !(slot.getFlags() & baja.Flags.HIDDEN);
  }
  /**
   * Check the component permissions and `OPERATOR` flag to see if this
   * action can be invoked.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {baja.Slot} slot the action slot
   * @returns {Boolean}
   */


  function canInvoke(comp, slot) {
    var perm = comp.getPermissions();

    if (perm.hasAdminInvoke()) {
      return true;
    }

    return !!(comp.getFlags(slot) & baja.Flags.OPERATOR) && perm.hasOperatorInvoke();
  }
  /**
   * Filter out items that the web profile thinks are valid but aren't really
   * valid. e.g. BWbWebProfile thinks pxEditor:PxEditor is a valid view, but
   * it really isn't - it gets filtered out in Workbench/applet by
   * NavMenuUtil.makeViewsMenu.
   *
   * @param {BajaViewInfo} item
   * @returns {boolean}
   */


  function isValidViewItem(item) {
    var ord = item.ord;
    return !(ord && ord.match(/view:pxEditor:PxEditor$/));
  }
  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/wb/menu/menuUtils
   */


  var exports = {};
  /**
   * Create a `CommandGroup` that will hold one command for each fireable
   * action on the component.
   *
   * @param {baja.Component} comp
   * @returns {module:bajaux/commands/CommandGroup}
   */

  exports.makeActionsGroup = function makeActionsGroup(comp) {
    var actionGroup = new CommandGroup({
      displayName: '%lexicon(webEditors:menu.actions.label)%'
    });
    comp.getSlots().actions().filter(visible).each(function (slot) {
      actionGroup.add(new Command({
        displayName: comp.getDisplayName(slot),
        enabled: canInvoke(comp, slot),
        func: function func() {
          return feDialogs.action({
            component: comp,
            slot: slot
          });
        }
      }));
    });
    return actionGroup;
  };
  /**
   * The display name used by the 'New' group of commands
   */


  exports.NEW_COMMAND_GROUP_DISPLAY_NAME = NEW_COMMAND_GROUP_DISPLAY_NAME;
  /**
   * Retrieve the different components to show in the "New" section of the
   * context menu, and assemble them into a command group. Each command will
   * simply invoke the Add Slot command for that particular component.
   *
   * @param {baja.Component} comp the component we may add slots to
   * @param {module:nmodule/webEditors/rc/wb/commands/AddSlotCommand} cmd
   * the command to invoke
   * @returns {Promise} promise to be resolved with a `CommandGroup`
   */

  exports.makeNewGroup = function makeNewGroup(comp, cmd) {
    var newGroup = new CommandGroup({
      displayName: exports.NEW_COMMAND_GROUP_DISPLAY_NAME
    });

    if (!comp.getPermissions().hasAdminWrite()) {
      return Promise.resolve(newGroup);
    }

    return newComponents().then(function (obj) {
      _.each(obj, function (value, prop) {
        if (value instanceof Separator) {
          newGroup.add(value);
        } else {
          newGroup.add(new Command({
            displayName: prop,
            icon: value.getType().getIcon().encodeToString(),
            func: function func(params) {
              return cmd.invoke(_.extend({
                display: 'simple',
                value: value
              }, params))["catch"](function (err) {
                feDialogs.error(err);
                throw err;
              });
            }
          }));
        }
      });

      return newGroup;
    });
  };
  /**
   * Retrieve the different views available on the given component and assemble
   * a CommandGroup, each one performing a hyperlink using that view.
   *
   * @inner
   * @param {baja.Ord} ord
   * @returns {Promise} promise to be resolved with a `CommandGroup`
   */


  exports.makeViewsGroup = function (ord) {
    var group = new CommandGroup({
      displayName: '%lexicon(webEditors:menu.views.label)%'
    });

    if (ord && ord !== baja.Ord.DEFAULT) {
      return views.listViews(ord).then(function (list) {
        _.each(_.filter(list, isValidViewItem), function (view) {
          group.add(new Command({
            displayName: view.displayName,
            icon: view.icon,
            func: function func() {
              return niagara.env.hyperlink(view.ord);
            }
          }));
        });

        return group;
      });
    } else {
      return Promise.resolve(group);
    }
  };
  /**
   * Create a default command group menu for the given subject, analogous to
   * `NavMenuUtil.makeMenu`. It will generate the command groups for each object
   * in the subject by resolving their menu agents, then attempt to merge them
   * all together so the resulting command group contains only those commands
   * capable of operating on the entire subject.
   *
   * @param {*} owner
   * @param {Array} subject - must contain baja values or nav nodes
   * @returns {Promise.<module:bajaux/commands/CommandGroup|null>} the merged
   * command group, or null if no group could be merged. Rejects if the owner
   * or subject is invalid.
   */


  exports.forSubject = function (owner, subject) {
    if (!owner) {
      return reject('owner required');
    }

    if (!Array.isArray(subject)) {
      return reject('subject must be an array');
    }

    if (_.any(subject, isInvalidMenuTarget)) {
      return reject('nav nodes or baja values required');
    }

    return Promise.all(subject.map(toGroupFromMenuAgent.bind(null, owner))).then(function (groups) {
      return _.reduce(groups, function (memo, group) {
        return memo.merge(group);
      });
    });
  };
  /**
   * Run the function only if the given key is allowed by the current profile.
   * If disallowed, the function will not be run and `null` will be resolved.
   * @param {string} key
   * @param {Function} func
   * @returns {Promise.<*|null>} promise to be resolved with the result of the
   * function, or `null` if disallowed
   */


  exports.ifAllowedByProfile = function (key, func) {
    if (getProfileTypeSpec() === 'hx:BasicHxProfile') {
      return Promise.resolve(null);
    }

    if (!require.specified('profileInfo')) {
      return Promise.resolve(func());
    }

    return doRequire('profileInfo').then(function (profileInfo) {
      return (profileInfo && profileInfo.getConfigOptions()[key] || null) && func();
    }, func);
  };
  /** Returns the module's config object.
   * @private
   * @return {Object}
   */


  exports.$getModuleConfig = function () {
    return config;
  };
  /** Returns whether the passed config object has the flag to use the
   *  minified tagdictionary build set to true.
   *
   * @return {Boolean}
   */


  exports.useTagDictionaryMinified = function () {
    var moduleConfig = exports.$getModuleConfig();
    return moduleConfig && moduleConfig.tagdictionaryMinified;
  };

  function toGroupFromMenuAgent(owner, obj) {
    return menuAgents.getAgentFor(obj).then(function (agent) {
      if (!agent) {
        throw new Error('no menu agent registered');
      }

      return agent(owner, obj);
    });
  }

  function isInvalidMenuTarget(value) {
    return !value || typeof value.getNavTypeSpec !== "function" && !baja.hasType(value);
  }

  function reject(msg) {
    return Promise.reject(new Error(msg));
  }

  function getProfileTypeSpec() {
    return typeof niagara !== 'undefined' && niagara.env.profile;
  }

  return exports;
});
