/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/job/JobBar
 */
define(['baja!', 'jquery', 'underscore', 'Promise', 'dialogs', 'bajaux/mixin/subscriberMixIn', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/job/jobSupport', 'nmodule/webEditors/rc/wb/table/Table', 'hbs!nmodule/webEditors/rc/wb/job/template/JobBar', 'baja!baja:Job'], function (baja, $, _, Promise, dialogs, subscribable, Command, CommandButtonGroup, fe, BaseEditor, typeUtils, jobSupport, Table, tplJobBar) {
  "use strict";

  var LOG_COMMAND_INDEX = 0,
      CANCEL_COMMAND_INDEX = 1,
      CANCEL_ICON_ORD = 'module://icons/x16/stop.png',
      DISPOSE_ICON_ORD = 'module://icons/x16/close.png',
      SHOW_LOG_ICON_ORD = 'module://icons/x16/doubleArrowRight.png'; ////////////////////////////////////////////////////////////////
  // Cancel Command
  ////////////////////////////////////////////////////////////////

  /**
   * Command for cancelling or disposing a BJob via the actions on the component.
   * The action taken will depend on the present state of the job - if it's running
   * then the job will be cancelled. If it's not running, the job will be disposed.
   * This means that two consecutive clicks on a button for a running job will cancel
   * then dispose the job.
   *
   * @constructor
   * @inner
   *
   * @param bar the job bar widget instance
   */

  var CancelCommand = function CancelCommand(bar) {
    var that = this;
    Command.call(that, {
      icon: DISPOSE_ICON_ORD,
      enabled: true,

      /**
       * Invoked to cancel or dispose a currently running job. The action invoked will
       * depend on the state of the job - if it's currently running, this command will
       * perform the 'cancel' action on the job. If the job is not currently running,
       * it will perform the 'dispose' action on the job.
       *
       * @returns {Promise}
       */
      func: function func() {
        var job = bar.$getJob(); // Once clicked, we disable the button. If we have cancelled
        // the job, we'll re-enable it once we get the event to tell
        // us the job has ended, at which point it can be clicked
        // again to be disposed. If we are doing a dispose, there will be
        // no need to enable it again.

        that.setEnabled(false);
        return isRunning(job) ? job.cancel() : job.dispose();
      }
    });
  };

  CancelCommand.prototype = Object.create(Command.prototype);
  CancelCommand.prototype.constructor = CancelCommand;
  /**
   * Update the command based on the newly notified job state. This is
   * used to change the icon, depending on whether a button click will
   * cause a dispose or cancel action.
   *
   * @private
   * @param {Boolean} validJob true if the job is 'valid', not yet disposed.
   * @param {Boolean} running true if the job is running and can be canceled.
   */

  CancelCommand.prototype.$updateForJobState = function (validJob, running) {
    if (validJob) {
      this.setEnabled(true);
    }

    this.setIcon(running ? CANCEL_ICON_ORD : DISPOSE_ICON_ORD);
  }; ////////////////////////////////////////////////////////////////
  // Log Command
  ////////////////////////////////////////////////////////////////

  /**
   * Command to read the JobLog via the action on the BJob and display it in
   * a dialog.
   *
   * @constructor
   * @inner
   *
   * @param bar the job bar widget instance
   */


  var LogCommand = function LogCommand(bar) {
    var that = this,
        readLog = jobSupport.readLogItems,
        makeRow = jobSupport.makeLogTableRow,
        makeModel = jobSupport.makeLogModel,
        showDetails = jobSupport.showItemDetailsDialog;

    function configureDoubleClick(elem) {
      elem.on('dblclick', 'tr', function (e) {
        var tr = $(e.target),
            table = tr.closest('.job-log-table').data('widget'),
            item = _.head(table.getSubject(tr));

        showDetails(item);
      });
    }

    Command.call(that, {
      icon: SHOW_LOG_ICON_ORD,
      enabled: true,

      /**
       * Function called to read the jog log via an action on the BJob.
       * @returns {Promise}
       */
      func: function func() {
        var job = bar.$getJob();
        /**
         * Create and show the dialog containing the table of log entries.
         */

        function showLogDialog(params) {
          // eslint-disable-next-line promise/avoid-new
          return new Promise(function (resolve, reject) {
            dialogs.showOk({
              title: params.title,
              content: function content(dlg, _content) {
                return fe.buildFor(params).then(function (ed) {
                  _content.prepend(ed.jq());

                  configureDoubleClick(params.dom);
                  dlg.ok(function () {
                    ed.jq().off();
                    return ed.destroy();
                  });
                })["catch"](reject);
              }
            });
          });
        }

        return readLog(job).then(function (items) {
          var rows = _.map(items, makeRow),
              model = makeModel(rows, {});

          return showLogDialog({
            dom: $('<table class="ux-table job-log-table"/>'),
            value: model,
            type: Table
          });
        });
      }
    });
  };

  LogCommand.prototype = Object.create(Command.prototype);
  LogCommand.prototype.constructor = LogCommand; ////////////////////////////////////////////////////////////////
  // JobBar
  ////////////////////////////////////////////////////////////////

  /**
   * Return true if the given job is currently in the running state.
   *
   * @param job
   * @returns {boolean}
   */

  function isRunning(job) {
    return job && job.get('jobState').getTag() === 'running';
  }
  /**
   * Helper function to show or hide the given dom element based on the boolean
   * 'show' parameter.
   *
   * @param {jQuery} elem a dom element.
   * @param {Boolean} show true to make the element visible, false to hide it.
   */


  function setElementVisible(elem, show) {
    elem.toggle(!!show);
  }
  /**
   * Return the jquery element for the progress bar. This is the element that will have
   * its CSS width value manipulated to show the current percentage of the job's progress.
   *
   * @param widget the job bar widget
   * @returns {jQuery}
   */


  function getProgressElement(widget) {
    return widget.jq().find('.job-bar-progress');
  }
  /**
   * Return the jquery element for the message. This element is normally only shown when
   * the job is not running.
   *
   * @param widget the job bar widget
   * @returns {jQuery}
   */


  function getMessageElement(widget) {
    return widget.jq().find('.job-bar-message');
  }
  /**
   * Return the jquery element for the name. This element normally contains a nicely formatted
   * string based on the job type's display name.
   *
   * @param widget the job bar widget
   * @returns {jQuery}
   */


  function getNameElement(widget) {
    return widget.jq().find('.job-bar-name');
  }
  /**
   * Return the element used for the job's icon. Note that this is not the same
   * as the state icon for the job, it will likely be the icon returned by the
   * job component's type.
   *
   * @param widget the job bar widget
   * @returns {jQuery}
   */


  function getJobComponentIconElement(widget) {
    return widget.jq().find('.job-bar-component-icon');
  }
  /**
   * Return the element used for the job state icon, e.g. the icon for the 'running' state.
   *
   * @param widget the job bar widget
   * @returns {jQuery}
   */


  function getJobStateIconElement(widget) {
    return widget.jq().find('.job-bar-state-icon');
  }
  /**
   * Set the text for the name of the BJob. The name is usually derived from the
   * display name of the job's type.
   *
   * @param widget the job bar widget
   * @param {String} name
   */


  function setName(widget, name) {
    getNameElement(widget).text(name);
  }
  /**
   * Set the message text. This is based on the state of the BJob.
   *
   * @param widget the job bar widget
   * @param {String} state the text describing the job's current state.
   */


  function setMessage(widget, state) {
    getMessageElement(widget).text(state);
  }
  /**
   * Update the position of the progress bar element, based on the current percentage
   * progress of the job.
   *
   * @param widget the job bar widget
   * @param {Number} percentage The progress of the job..
   */


  function setProgress(widget, percentage) {
    var elem = getProgressElement(widget);

    if (percentage > -1) {
      elem.css('width', String(percentage) + '%');
    }
  }
  /**
   * Update the position of the progress bar based on the current percentage
   * progress of the job. This will not update the bar if the job does not
   * report its progress (i.e. it returns -1).
   *
   * @param job
   * @param widget
   */


  function updateFromJobProgress(job, widget) {
    var progress = job.get('progress').valueOf();

    if (progress > -1) {
      setProgress(widget, progress);
    }
  }
  /**
   * Update the widget from a changed event on the 'jobState' slot.
   * This will hide or show the progress bar, and update the message
   * text and state icon.
   *
   * It's safe to call this function when the widget does not have a
   * job loaded (it will be called from widget initialization). If
   * there's not a job at the time this is called, the appropriate bits
   * of the UI will be hidden.
   *
   * @param job
   * @param widget
   */


  function updateFromJobState(job, widget) {
    var state = job && job.isMounted() && job.get('jobState'),
        validJob = state && state.getTag() !== 'unknown',

    /* 'unknown' => disposed job */
    running = validJob && isRunning(job),
        stateIcon = getJobStateIconElement(widget),
        jobLogCmd = widget.getCommandGroup().get(LOG_COMMAND_INDEX),
        cancelCmd = widget.getCommandGroup().get(CANCEL_COMMAND_INDEX);
    jobLogCmd.setEnabled(validJob);
    cancelCmd.setEnabled(validJob);
    cancelCmd.$updateForJobState(validJob, running);
    setElementVisible(stateIcon, validJob);
    setElementVisible(getMessageElement(widget), validJob && !running);
    setElementVisible(getProgressElement(widget), validJob && running);
    setElementVisible(getNameElement(widget), validJob);
    setElementVisible(getJobComponentIconElement(widget), validJob);

    if (state) {
      setMessage(widget, state.getDisplayTag());
      stateIcon.attr('src', '/ord/' + jobSupport.stateToIconOrd(state));
    }
  }
  /**
   * The name to be displayed for the given Job - returns the job's
   * toString() implementation.
   *
   *
   * For the standard baja.Job implementation, this is based on the
   * display name for the Job's type, with the word 'Job' removed from the
   * end, if it is found there. Thus, the returned name for the type
   * BStationDiscoveryJob, which has no lexiconized display name, will
   * be 'Station Discovery'.
   *
   * @inner
   * @param job
   * @returns {Promise.<String>}
   */


  function nameFromJobType(job) {
    return Promise.resolve(job.toString());
  }
  /**
   * Widget to show the current status of a job (running, cancelled, etc) along
   * with its current progress, and buttons to cancel/dispose the job and view the log.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/webEditors/rc/wb/job/JobBar
   */


  var JobBar = function JobBar(params) {
    var that = this; // The 'hideCommandButtons' parameter is internal, intended to be used in situations
    // where a job bar is embedded in a context that does not want the buttons to be
    // seen by default. This will be used by the JobDialog, which will provide its own
    // cancel button and log table.

    params = _.defaults(params || {}, {
      hideCommandButtons: false
    });
    BaseEditor.apply(that, arguments);
    subscribable(that);
    that.$hideCommandButtons = params.hideCommandButtons;
    that.getCommandGroup().add(new LogCommand(that), new CancelCommand(that));
  };

  JobBar.prototype = Object.create(BaseEditor.prototype);
  JobBar.prototype.constructor = JobBar;
  /**
   * Load the template for the job bar. This is an override point for
   * subclasses that may want to layout the UI differently depending on
   * context.
   *
   * @private
   * @param {jQuery} dom
   */

  JobBar.prototype.loadTemplate = function (dom) {
    dom.html(tplJobBar({}));
  };
  /**
   * Initialize the widget.
   *
   * @param {JQuery} dom the DOM element to load the widget into.
   */


  JobBar.prototype.doInitialize = function (dom) {
    var that = this,
        commandContainer;
    that.loadTemplate(dom);
    dom.addClass('JobBar');
    commandContainer = dom.find('.commandContainer');

    if (that.$hideCommandButtons) {
      commandContainer.hide();
    }

    return fe.buildFor({
      dom: $('<div/>').appendTo(commandContainer),
      type: CommandButtonGroup,
      value: that.getCommandGroup(),
      properties: {
        toolbar: true
      }
    }).then(function () {
      updateFromJobState(null, that);
    });
  };
  /**
   * Load the value into the widget. The value must be a BJob component instance.
   *
   * @param {baja.Component} value a the job component, as described above.
   * @returns {Promise}
   */


  JobBar.prototype.doLoad = function (value) {
    var that = this,
        sub = that.getSubscriber();

    if (that.$job) {
      delete that.$job;
    }

    if (!baja.hasType(value, 'baja:Job')) {
      return Promise.reject(new Error('JobBar must be loaded with a BJob instance.'));
    }

    that.$job = value;
    sub.attach({
      'changed': function changed(prop) {
        if (prop.getName() === 'progress') {
          updateFromJobProgress(value, that);
        } else if (prop.getName() === 'jobState') {
          updateFromJobState(value, that);
        }
      }
    });
    setProgress(that, 0);
    updateFromJobProgress(value, that);
    updateFromJobState(value, that);
    return nameFromJobType(value).then(function (name) {
      setName(that, name);
      getJobComponentIconElement(that).attr('src', value.getIcon().getImageUris()[0]);
    });
  };
  /**
   * Destroy the widget. This will clean up the `Subscriber` attached to the
   * associated job.
   *
   * @returns {Promise|*}
   */


  JobBar.prototype.doDestroy = function () {
    var that = this;
    that.jq().removeClass('JobBar');

    if (that.$job) {
      delete that.$job;
    }
  };
  /**
   * Enable or disable the UI elements.
   * @param {Boolean} enabled
   */


  JobBar.prototype.doEnabled = function (enabled) {
    var cmds = this.getCommandGroup();
    enabled = !this.isReadonly() && enabled;
    cmds.get(LOG_COMMAND_INDEX).setEnabled(enabled);
    cmds.get(CANCEL_COMMAND_INDEX).setEnabled(enabled);
    return this.getChildWidgets().setAllEnabled(enabled);
  };
  /**
   * Enable or disable the UI elements.
   * @param {Boolean} readonly
   */


  JobBar.prototype.doReadonly = function (readonly) {
    var cmds = this.getCommandGroup();
    cmds.get(LOG_COMMAND_INDEX).setEnabled(this.isEnabled());
    cmds.get(CANCEL_COMMAND_INDEX).setEnabled(!readonly && this.isEnabled());
    return this.getChildWidgets().setAllReadonly(readonly);
  };
  /**
   * Returns the currently loaded job instance.
   * @private
   */


  JobBar.prototype.$getJob = function () {
    return this.$job;
  };

  return JobBar;
});
