/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint browser: true*/

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/file/TextFileEditor
 */
define(['baja!', 'Promise', 'jquery', 'bajaux/Widget', 'nmodule/webEditors/rc/wb/mixin/textEditorMixIn', 'nmodule/js/rc/csrf/csrfUtil'], function (baja, Promise, $, Widget, textEditorMixIn, csrfUtil) {
  'use strict';
  /**
   * An editor for viewing and editing text files.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/webEditors/rc/wb/file/TextFileEditor
   */

  var TextFileEditor = function TextFileEditor() {
    var that = this;
    Widget.apply(that, arguments);
    that.properties().add("readonly", false);
    that.properties().add("showRun", true);
    textEditorMixIn(that);
  };

  TextFileEditor.prototype = Object.create(Widget.prototype);
  TextFileEditor.prototype.constructor = TextFileEditor;
  /**
   * Return true if the editor should be readonly.
   *
   * @private
   * @inner
   *
   * @param  widget The widget instance.
   * @returns {Boolean} Return true if readonly.
   */

  function isReadonly(widget) {
    return widget.value().isReadonly() || widget.properties().getValue("readonly");
  }
  /**
   * Read the contents of a file.
   *
   * @inner
   * @private
   *
   * @param  file The file to read.
   * @return {Promise} A promise that resolves with the file's contents.
   */


  function readFile(file) {
    return baja.isOffline() ? baja.rpc("type:web:FileRpc", "readTextFile", file.getNavOrd().toString()) : Promise.resolve($.ajax(file.getReadUri() + '?snoop=false', {
      dataType: "text"
    }));
  }
  /**
   * Write a file.
   *
   * @inner
   * @private
   *
   * @param  file The file to write too.
   * @param {String} contents The contents of the file to write too.
   * @return {Promise} A promise that resolves when the file has been written.
   */


  function writeFile(file, contents) {
    return baja.isOffline() ? baja.rpc("type:web:FileRpc", "writeTextFile", file.getNavOrd().toString(), contents) : Promise.resolve($.ajax(file.getWriteUri(), {
      dataType: "text",
      method: "PUT",
      data: contents,
      headers: {
        'x-niagara-csrfToken': csrfUtil.getCsrfToken()
      }
    }));
  }

  TextFileEditor.prototype.doLoad = function (file) {
    var that = this,
        mode,
        readonly = isReadonly(that);

    if (!readonly) {
      that.$addSaveCmd();
    }

    switch (file.getMimeType()) {
      case "text/css":
        mode = "css";
        break;

      case "text/javascript":
        mode = "javascript";

        if (that.properties().getValue("showRun") && !readonly) {
          that.$addRunCmd();
        }

        break;

      case "text/html":
        mode = "html";
        break;

      case "text/xml":
        mode = "xml";
        break;
    }

    switch (file.getExtension().toLowerCase()) {
      case "java":
        mode = "java";
        break;

      case "json":
        mode = "json";
        break;

      case "less":
        mode = "less";
        break;

      case "hbs":
        mode = "handlebars";
        break;

      case "vm":
      case "pxvm":
        mode = "velocity";
        break;
    }

    that.$initEditor(mode);
    that.$setEditorReadOnly(readonly);

    if (!readonly) {
      // If something in the editor changes then mark the widget as modified.
      that.$onEditorChange(function () {
        that.setModified(true);
      });
    }

    return readFile(file).then(function (text) {
      that.$setEditorText(text);
    });
  };

  TextFileEditor.prototype.doSave = function () {
    var that = this,
        file = that.value();
    return !that.$editor || isReadonly(that) ? Promise.resolve() : writeFile(file, that.$getEditorText());
  };

  return TextFileEditor;
});
