/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/multisheet/ShowHiddenSlotsCommand
 */
define(['baja!', 'bajaux/commands/ToggleCommand'], function (baja, ToggleCommand) {
  'use strict';
  /**
   * Check whether I have admin write permissions on the given component.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {Boolean}
   */

  function hasAdminWrite(comp) {
    return baja.hasType(comp, 'baja:Component') && comp.getPermissions().hasAdminWrite();
  }
  /**
   * Check whether the multisheet is in a state where I should be allowed to
   * view hidden slots on it.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/MultiSheet} multiSheet
   * @returns {Boolean}
   */


  function canShowHiddenSlots(multiSheet) {
    return !multiSheet.isModified() && multiSheet.isSlotMode() && hasAdminWrite(multiSheet.value());
  }
  /**
   * Command for toggling whether a MultiSheet is showing hidden slots.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/commands/multisheet/ShowHiddenSlotsCommand
   * @extends module:bajaux/commands/ToggleCommand
   * @param {module:nmodule/webEditors/rc/wb/MultiSheet} multiSheet
   */


  var ShowHiddenSlotsCommand = function ShowHiddenSlotsCommand(multiSheet) {
    var that = this;
    ToggleCommand.call(that, {
      module: 'webEditors',
      lex: 'commands.showHiddenSlots',
      selected: multiSheet.isShowHiddenSlots(),
      enabled: false,
      func: function func() {
        that.toggle();
        return multiSheet.setShowHiddenSlots(that.isSelected());
      }
    });
    multiSheet.on('loaded', function () {
      that.$updateEnabled();
    });

    that.$updateEnabled = function () {
      that.setEnabled(canShowHiddenSlots(multiSheet));
    };
  };

  ShowHiddenSlotsCommand.prototype = Object.create(ToggleCommand.prototype);
  ShowHiddenSlotsCommand.prototype.constructor = ShowHiddenSlotsCommand;
  return ShowHiddenSlotsCommand;
});
