/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ReorderSlotsCommand
 */
define(['Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand', 'nmodule/webEditors/rc/wb/commands/ReorderSlotsDialog'], function (Promise, _, feDialogs, ComponentEditorCommand, ReorderSlotsDialog) {
  'use strict'; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Get the dynamic slots from the component.
   *
   * @inner
   * @param {baja.Complex} comp
   * @returns {Array.<baja.Slot>}
   */

  function getDynamicSlots(comp) {
    return comp.getSlots().dynamic().toArray();
  }
  /**
   * Return true if the given slots are not a valid argument for the component's
   * reorder operation. All dynamic slots on the component should be accounted
   * for with no extras.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots
   * @returns {Boolean} if slots are invalid
   */


  function slotMismatch(comp, slots) {
    var dynamic = getDynamicSlots(comp);

    if (dynamic.length !== slots.length) {
      return true;
    } //make everyone strings for indexOf


    dynamic = _.map(dynamic, String);
    slots = _.map(slots, String);

    for (var i = 0; i < slots.length; i++) {
      if (slots.indexOf(dynamic[i]) === -1) {
        return true;
      }
    }
  }
  /**
   * Perform the slot reorder on the component.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot|String>} slotOrder the desired order of dynamic
   * slots
   * @returns {Promise} promise to be resolved if the reorder succeeds
   */


  function doReorder(comp, slotOrder) {
    if (_.isArray(slotOrder) && !slotMismatch(comp, slotOrder)) {
      return comp.reorder({
        dynamicProperties: slotOrder
      });
    } else {
      return Promise.reject(new Error('could not determine slots to reorder'));
    }
  }
  /**
   * Validate that the slot to move and target slot represent a valid reorder
   * operation on the component, then return a list of dynamic slot names
   * for the component.
   *
   * @inner
   * @param {baja.Complex} comp
   * @param {String} toMove the slot being moved
   * @param {String} targetSlot the slot that `toMove` is moving next to
   * @returns {Array.<String>} an array of slot names
   * @throws {Error} if the component and slots do not represent a valid
   * reorder operation
   */


  function getValidSlotNames(comp, toMove, targetSlot) {
    if (!comp || !comp.has(targetSlot) || !comp.has(toMove) || comp.getSlot(toMove).isFrozen()) {
      throw new Error('could not find valid slots on Component');
    }

    return _.map(getDynamicSlots(comp), String);
  }
  /**
   * Get the resulting slot order after moving a particular slot before or
   * after another slot.
   *
   * @inner
   * @returns {Array.<String>}
   */


  function getReorderedSlots(comp, params) {
    var toMove = params.toMove,
        before = params.before,
        after = params.after,
        targetSlot = String(before || after);
    var allDynamicSlots = getValidSlotNames(comp, toMove, targetSlot);

    if (toMove === targetSlot) {
      return allDynamicSlots;
    }

    var slots = [toMove],
        unmovedDynamicSlots = _.difference(allDynamicSlots, slots),
        idx = _.indexOf(unmovedDynamicSlots, targetSlot);

    if (comp.getSlot(targetSlot).isFrozen()) {
      return slots.concat(unmovedDynamicSlots);
    }

    unmovedDynamicSlots.splice(after ? idx + 1 : idx, 0, toMove);
    return unmovedDynamicSlots;
  }
  /**
   * Given a component, return an array of dynamic slot names with one slot moved
   * around so that is before the target slot.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {String} toMove the slot to move around
   * @param {String} targetSlot move the slot after this slot. If `targetSlot`
   * is a frozen slot, the slot will be made the first dynamic slot.
   * @returns {Array.<String>} array of dynamic slot names with `toMove`
   * moved over so that it is before `targetSlot`
   */


  function moveBefore(comp, toMove, targetSlot) {
    return getReorderedSlots(comp, {
      toMove: toMove,
      before: targetSlot
    });
  }
  /**
   * Given a component, return an array of dynamic slot names with one slot moved
   * around so that is after the target slot.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {String} toMove the slot to move around
   * @param {String} targetSlot move the slot after this slot. If `targetSlot`
   * is a frozen slot, the slot will be made the first dynamic slot.
   * @returns {Array.<String>} array of dynamic slot names with `toMove`
   * moved over so that it is after `targetSlot`
   */


  function moveAfter(comp, toMove, targetSlot) {
    return getReorderedSlots(comp, {
      toMove: toMove,
      after: targetSlot
    });
  } ////////////////////////////////////////////////////////////////
  // ReorderSlotsCommand
  ////////////////////////////////////////////////////////////////

  /**
   * A command for reordering slots on an editor's `Component` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/ReorderSlotsCommand
   * @param {baja.Component} component the component whose slots to reorder
   * @throws {Error} if no `Widget` provided
   */


  var ReorderSlotsCommand = function ReorderSlotsCommand(component) {
    ComponentEditorCommand.call(this, {
      module: 'webEditors',
      lex: 'commands.reorder'
    }, component);
  };

  ReorderSlotsCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  ReorderSlotsCommand.prototype.constructor = ReorderSlotsCommand;
  /**
   * Specify that during the next order operation, one slot should be
   * moved so that it comes directly before another slot.
   *
   * @param {String|baja.Slot} toMove the slot to move
   * @param {String|baja.Slot} targetSlot the slot that `toMove` should
   * come directly before
   * @returns {Array.<String>} reordered array of dynamic slot names
   */

  ReorderSlotsCommand.prototype.moveBefore = function (toMove, targetSlot) {
    return moveBefore(this.getComponent(), toMove, targetSlot);
  };
  /**
   * Specify that during the next order operation, one slot should be
   * moved so that it comes directly after another slot.
   *
   * @param {String|baja.Slot} toMove the slot to move
   * @param {String|baja.Slot} targetSlot the slot that `toMove` should
   * come directly after
   * @returns {Array.<String>} reordered array of dynamic slot names
   */


  ReorderSlotsCommand.prototype.moveAfter = function (toMove, targetSlot) {
    return moveAfter(this.getComponent(), toMove, targetSlot);
  };
  /**
   * Make sure we have admin write permissions.
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} true if I can reorder slots on this component
   */


  ReorderSlotsCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite();
  };
  /**
   * Directly reorders the slots on the editor's loaded component, or shows
   * a dialog allowing the user to configure the slot order manually.
   *
   * @param {Object} [params]
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots
   * @param {Object} [params]
   * @param {Array.<baja.Slot|String>} [params.dynamicSlots] the desired slot
   * order for the component loaded into the editor. If this parameter is
   * omitted, it will be prompted from the user using a `ReorderSlotsDialog`.
   * @returns {Promise} promise to be resolved after the component's
   * slots have been reordered (or the user has clicked Cancel).
   */


  ReorderSlotsCommand.prototype.performCommand = function (comp, slots, params) {
    var dynamicSlots = params && params.dynamicSlots,
        toReorder = slots.length ? comp.get(slots[0]) : comp;

    if (!dynamicSlots || !dynamicSlots.length) {
      return feDialogs.showFor({
        value: toReorder,
        type: ReorderSlotsDialog
      }).then(function (slotOrder) {
        if (slotOrder) {
          return doReorder(toReorder, slotOrder);
        }
      });
    }

    return doReorder(toReorder, dynamicSlots);
  };

  return ReorderSlotsCommand;
});
