/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/RenameCommand
 */
define(['baja!', 'lex!webEditors', 'Promise', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, lexs, Promise, feDialogs, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * A command for renaming slots on an editor's `Component` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/RenameCommand
   * @param {baja.Component} component the component on which to rename slots
   * @throws {Error} if no `Widget` provided
   */

  var RenameCommand = function RenameCommand(component) {
    var that = this;
    ComponentEditorCommand.call(that, {
      module: 'webEditors',
      lex: 'commands.rename'
    }, component);
  };

  RenameCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  RenameCommand.prototype.constructor = RenameCommand;
  /**
   * Make sure we have admin write permissions and we're not renaming a
   * frozen slot.
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} true if I can rename this slot
   */

  RenameCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite() && !slot.isFrozen();
  }; //TODO: support find/replace to renames

  /**
   * Shows a dialog for the component's currently configured slot name,
   * and renames the slot after the user clicks OK.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots the slots to rename (*Note*: only the
   * first given slot will be renamed)
   * @returns {Promise}
   */


  RenameCommand.prototype.performCommand = function (comp, slots) {
    var slot = slots[0];

    if (!slot) {
      return Promise.reject(new Error('slot to rename required'));
    }

    var oldName = slot.getName();
    return feDialogs.showFor({
      title: webEditorsLex.get('commands.rename.description'),
      value: baja.SlotPath.unescape(oldName),
      formFactor: 'mini'
    }).then(function (newName) {
      if (newName === null) {
        return;
      }

      if (newName.trim() === '') {
        throw new Error(webEditorsLex.get('enterName'));
      }

      newName = baja.SlotPath.escape(newName.trim());

      if (newName && newName !== oldName) {
        //TODO: is this a bajascript bug? display name does not follow the renamed slot
        return comp.rename({
          slot: oldName,
          newName: newName
        });
      }
    })["catch"](feDialogs.error);
  };

  return RenameCommand;
});
