/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/LinkCommand
 */
define(['lex!webEditors', 'Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/wb/links/LinkPad', 'nmodule/webEditors/rc/wb/util/TransferDataManager', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (lexs, Promise, _, Command, LinkPad, TransferDataManager, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0];
  var transferDataManager = TransferDataManager.getInstance(); ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/LinkCommand
   * @param {baja.Component} component the link source component
   * @param {String} lexKey
   * @throws {Error} if no `Widget` provided
   */

  var LinkCommand = function LinkCommand(component, lexKey) {
    this.$lexKey = lexKey;
    ComponentEditorCommand.call(this, {}, component);
  };

  LinkCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  LinkCommand.prototype.constructor = LinkCommand;
  /**
   * Give the command an opportunity to initialize.
   *
   * @param component
   * @returns {Promise} promise that resolves after the command has been initialized.
   */

  LinkCommand.prototype.init = function (component) {
    var that = this,
        prop = component.getPropertyInParent(),
        isFrozen = !prop || prop.isFrozen(); // If LinkMark is set and the target component isn't frozen,
    // enable this command

    return that.getLinkMark().then(function (linkMark) {
      if (linkMark) {
        that.setEnabled(!isFrozen);
      }

      return that;
    });
  };
  /**
   * Make sure can link from this component.
   *
   * @param {baja.Component} comp
   * @returns {Promise.<Boolean>} Resolves to true if I can link from this component.
   */


  LinkCommand.prototype.canPerformCommand = function (comp) {
    return this.getLinkMark().then(function (linkMark) {
      return !!linkMark;
    });
  };
  /**
   * Link To/From this component
   *
   * Display the dialog that allows the user to choose which slots to link.
   * To/From is determined by derived implementation.
   *
   * @param {baja.Component} component
   * @param {Array.<baja.Slot>} slots, optional (may be `null` or an empty array)
   * @param {Object} params
   * @returns {Promise} promise to be resolved after
   */


  LinkCommand.prototype.performCommand = function (component, slots, params) {
    var that = this,
        slot;

    if (Array.isArray(slots) && slots.length > 0) {
      slot = slots[0];
    } // TODO support multiple slots


    if (slot) {
      component = component.get(slot);
    }

    return that.getSourceTarget(component).then(function (sourceTarget) {
      // MakeLink is implemented in doDialog; is that correct?
      return LinkPad.doDialog({
        sources: sourceTarget.sources,
        targets: sourceTarget.targets,
        slot: 'link',
        properties: {
          editIncoming: true,
          editOutgoing: true,
          useHandles: true
        },
        readonly: false
      });
    });
  };
  /**
   * Get the marked link.
   *
   * @returns {Promise} promise to be resolved with an array of `NavNode`s
   *      that was marked with LinkMarkCommand.
   */


  LinkCommand.prototype.getLinkMark = function () {
    return transferDataManager.getLinkMark();
  };
  /**
   * Get the LinkMark display name
   *
   * @returns {Promise.<Array.<String>>} resolves to an array of strings of marked component
   * display names
   */


  LinkCommand.prototype.getLinkMarkDisplayName = function () {
    var that = this;
    return that.getLinkMark().then(function (linkMarks) {
      if (linkMarks) {
        return _.map(linkMarks, function (linkMark) {
          return linkMark.getNavDisplayName();
        });
      }

      return [''];
    });
  };
  /**
   * Returns an object with source and target components populated,
   * given the specified component.
   *
   * LinkToCommand will use component as the target and the marked link
   * as the source, whereas LinkFromCommand will use component as the
   * source and the marked link as the target.
   *
   * @param {baja.Component} component - the source or target component.
   * @return {Promise} that resolves to an `Object`
   */


  LinkCommand.prototype.getSourceTarget = function (component) {
    return Promise.reject(new Error('LinkCommand getSourceTarget() not implemented'));
  };
  /**
   * @return {Promise.<String>} resolves to a string that is the display name of this
   * command.
   */


  LinkCommand.prototype.toDisplayName = function () {
    var that = this;
    return that.getLinkMarkDisplayName().then(function (linkMarkDisplayNames) {
      return webEditorsLex.get({
        key: that.$lexKey + '.displayName',
        args: linkMarkDisplayNames
      });
    });
  };

  return LinkCommand;
});
