/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/DuplicateCommand
 */
define(['baja!', 'lex!webEditors', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, lexs, Promise, _, feDialogs, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0],
      copy = baja.transfer.copy;
  /**
   * A command for duplicating slots
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/DuplicateCommand
   * @param {baja.Component} component the component to duplicate
   * @throws {Error} if no `Widget` provided
   */

  var DuplicateCommand = function DuplicateCommand(component) {
    ComponentEditorCommand.call(this, {
      module: 'webEditors',
      lex: 'commands.duplicate'
    }, component);
  };

  DuplicateCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  DuplicateCommand.prototype.constructor = DuplicateCommand;
  /**
   * Make sure we have admin write permissions
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} true if this slot can be duplicated
   */

  DuplicateCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite();
  };
  /**
   * Duplicates the selected slots from the editor's loaded Component.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots the slots to Duplicate
   * @returns {Promise}
   */


  DuplicateCommand.prototype.performCommand = function (comp, slots) {
    var name, value;

    switch (slots.length) {
      case 0:
        return Promise.resolve();

      case 1:
        //prompt for dialog when duplicating one slot
        value = comp.get(slots[0]);
        name = comp.getUniqueName(slots[0].getName());
        return feDialogs.showFor({
          title: webEditorsLex.get('commands.duplicate.displayName'),
          value: name,
          formFactor: 'mini'
        }).then(function (name) {
          if (name && baja.SlotPath.isValidName(name)) {
            return copy({
              names: [name],
              sourceOrds: [value.getNavOrd()],
              target: comp
            });
          }
        });

      default:
        //multi-slot duplication
        var sourceOrds = _.map(slots, function (slot) {
          return comp.get(slot).getNavOrd();
        });

        return copy({
          sourceOrds: sourceOrds,
          target: comp
        });
    }
  };

  return DuplicateCommand;
});
