/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ConfigFlagsCommand
 */
define(['baja!', 'lex!webEditors', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/FlagConfig', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, lexs, Promise, _, feDialogs, FlagConfig, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Set the given flags on all specified slots on the component.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots
   * @param {Number} flags
   * @returns {Promise} promise to be resolved when the flags are set
   * on all slots
   */

  function setFlagsOnAllSlots(comp, slots, flags) {
    var batch = new baja.comm.Batch(),
        sets = _.map(slots, function (slot) {
      if (comp.getFlags(slot) !== flags) {
        return comp.setFlags({
          slot: slot,
          flags: flags,
          batch: batch
        });
      }
    });

    batch.commit();
    return Promise.all(sets);
  }
  /**
   * Show a FlagConfig for the existing slot flags retrieved from the
   * component's selected slots, and return the user-entered value.
   *
   * @inner
   * @param {Array.<baja.Slot>} slots
   * @param {Number} oldFlags
   * @returns {Promise} promise to be resolved with the new flags, or
   * null if the user clicked Cancel
   */


  function editFlagsForSlots(slots, oldFlags) {
    var allDynamic = true,
        allFrozen = true,
        allActions = true;

    _.each(slots, function (slot) {
      if (slot.isFrozen()) {
        allDynamic = false;
      } else {
        allFrozen = false;
      }

      if (!slot.isAction()) {
        allActions = false;
      }
    });

    return feDialogs.showFor({
      properties: {
        dynamic: allDynamic,
        frozen: allFrozen,
        action: allActions
      },
      value: oldFlags,
      title: webEditorsLex.get('commands.configFlags.description'),
      type: FlagConfig
    });
  }
  /**
   * Return slot flags that ALL the given slots have.
   *
   * @inner
   * @param {baja.Complex} comp
   * @param {Array.<baja.Slot|String>} slots
   * @returns {Number} merged flags
   */


  function mergeFlags(comp, slots) {
    if (!slots.length) {
      return 0;
    }

    return _.reduce(slots, function (flags, slot) {
      return flags & comp.getFlags(slot);
    }, 0xFFFFFFFF);
  } ////////////////////////////////////////////////////////////////
  // ConfigFlagsCommand definition
  ////////////////////////////////////////////////////////////////

  /**
   * A command for configuring slot facets on an editor's `Component` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/ConfigFlagsCommand
   * @param {baja.Component} component the component on which to configure
   * slot flags
   * @throws {Error} if no `Widget` provided
   */


  var ConfigFlagsCommand = function ConfigFlagsCommand(component) {
    //TODO: add module/lex to jsdoc
    ComponentEditorCommand.call(this, {
      module: 'webEditors',
      lex: 'commands.configFlags'
    }, component);
  };

  ConfigFlagsCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  ConfigFlagsCommand.prototype.constructor = ConfigFlagsCommand;
  /**
   * Make sure we have admin write permissions on the component.
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} true if I can configure flags on this slot
   */

  ConfigFlagsCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite();
  }; //TODO: accept flags param directly

  /**
   * Shows a dialog for the component's currently configured slot flags,
   * and saves the flags back to the slot after the user clicks OK.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots slots on which to set flags
   * @returns {Promise} promise to be resolved after user has entered
   * new flags and they have been written to the component
   */


  ConfigFlagsCommand.prototype.performCommand = function (comp, slots) {
    var oldFlags = mergeFlags(comp, slots);
    return editFlagsForSlots(slots, oldFlags).then(function (newFlags) {
      if (newFlags !== null) {
        return setFlagsOnAllSlots(comp, slots, newFlags);
      }
    });
  };

  return ConfigFlagsCommand;
});
