/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ConfigFacetsCommand
 */
define(['baja!', 'lex!webEditors', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, lexs, Promise, _, feDialogs, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Merge the slot facets on all the specified slots into one facets object.
   * This facets object will be presented to the user for editing.
   *
   * @inner
   * @param {baja.Complex} comp
   * @param {Array.<baja.Slot>} slots
   * @returns {baja.Facets}
   */

  function mergeSlotFacets(comp, slots) {
    return _.reduce(slots, function (facets, slot) {
      return baja.Facets.make(facets, comp.getSlot(slot).getFacets());
    }, baja.Facets.DEFAULT);
  }
  /**
   * Set the given facets on all the specified component slots.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots
   * @param {baja.Facets} facets
   * @returns {Promise} promise to be resolved after the facets have
   * been set on all specified slots
   */


  function setFacetsOnAllSlots(comp, slots, facets) {
    var batch = new baja.comm.Batch(),
        sets = _.map(slots, function (slot) {
      if (!facets.equivalent(comp.getFacets(slot))) {
        return comp.setFacets({
          slot: slot,
          facets: facets
        });
      }
    });

    batch.commit();
    return Promise.all(sets);
  }
  /**
   * A command for configuring slot facets.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/ConfigFacetsCommand
   * @param {baja.Component} component the component on which to configure slot
   * facets
   * @throws {Error} if no `Widget` provided
   */


  var ConfigFacetsCommand = function ConfigFacetsCommand(component) {
    ComponentEditorCommand.call(this, {
      module: 'webEditors',
      lex: 'commands.configFacets'
    }, component);
  };

  ConfigFacetsCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  ConfigFacetsCommand.prototype.constructor = ConfigFacetsCommand;
  /**
   * Make sure we have admin write privileges on the component and the slot
   * is not frozen.
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} if facets can be configured on this component slot
   */

  ConfigFacetsCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite() && !slot.isFrozen();
  };
  /**
   * Shows a dialog for the component's currently configured slot facets,
   * and saves the facets back to the slot(s) after the user clicks OK.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots slots on which to set facets
   * @returns {Promise} promise to be resolved after the user has entered
   * new facets and they have been written to the component
   */


  ConfigFacetsCommand.prototype.performCommand = function (comp, slots) {
    var oldFacets = mergeSlotFacets(comp, slots);
    return feDialogs.showFor({
      value: oldFacets,
      title: webEditorsLex.get('commands.configFacets.description'),
      formFactor: 'compact'
    }).then(function (newFacets) {
      if (newFacets !== null) {
        return setFacetsOnAllSlots(comp, slots, newFacets);
      }
    });
  };

  return ConfigFacetsCommand;
});
